/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.twitterplugin.twitter;

import android.content.Context;
import android.util.Log;

import twitter4j.DirectMessage;
import twitter4j.PagableResponseList;
import twitter4j.Paging;
import twitter4j.Status;
import twitter4j.StatusUpdate;
import twitter4j.Twitter;
import twitter4j.TwitterException;
import twitter4j.User;
import twitter4j.conf.Configuration;
import twitter4j.conf.ConfigurationBuilder;
import twitter4j.http.AccessToken;

import com.sonyericsson.eventstream.twitterplugin.PluginConstants.Config;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.TwitterConf;
import com.sonyericsson.eventstream.twitterplugin.twitter.TwitterPluginException.StatusCode;
import com.sonyericsson.eventstream.twitterplugin.utility.PreferencesHelper;

import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.util.LinkedList;
import java.util.List;

import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;

public class TwitterCommunication {

    /** There is a special case where we get a status code of minus one */
    private static final int STATUS_CODE_MINUS_ONE = -1;

    private static final String USER_AGENT = "Timescape";

    private static final int RETRY_COUNT = 3;

    private static final int MAX_CONNECTIONS = 10;

    /** Holds the factory to get the twitter class */
    private TwitterFactoryWrapper mTwitterFactory;

    /** Holds the real twitter access from the API */
    private Twitter mTwitter;

    private Context mContext;

    public TwitterCommunication(Context context) {
        mContext = context;

        String consumerKey = getCustomerKey(CredentialHandler.getInstance().getCredential(
                CredentialHandler.TWITTER_API_KEY));
        String consumerSecret = getCustomerKey(CredentialHandler.getInstance().getCredential(
                CredentialHandler.TWITTER_SECRET_KEY));
        Configuration config = createConfiguration(consumerKey, consumerSecret);
        mTwitterFactory = new TwitterFactoryWrapper(config);
    }

    /**
     * Authenticate a username and password towards Twitter using OAuth.
     *
     * @param userName The username for this account.
     * @param password The password for this account.
     * @return True if login was successful, false otherwise.
     */
    public boolean authenticateTwitterAccount(String userName, String password) throws TwitterPluginException {
        if (userName == null || password == null || userName.length() == 0 || password.length() == 0) {
            return false;
        }
        try {
            PreferencesHelper.clearAuthentication(mContext);

            Twitter twitter = mTwitterFactory.getInstance(userName, password);

            AccessToken accessToken = twitter.getOAuthAccessToken();
            if (accessToken == null) {
                Log.e(Config.LOG_TAG, "Twitter returned null access token");
                return false;
            }
            String token = accessToken.getToken();
            String tokenSecret = accessToken.getTokenSecret();
            accessToken = new AccessToken(token, tokenSecret);

            twitter = mTwitterFactory.getOAuthAuthorizedInstance(accessToken);
            User authenticatedUser = twitter.verifyCredentials();
            if (authenticatedUser != null) {
                PreferencesHelper.setToken(mContext, token, tokenSecret);
                return true;
            }
        } catch (TwitterException e) {
            if (e.isCausedByNetworkIssue()) {
                throw new TwitterPluginException(e);
            }
            // This is a special case where we have full signal strength
            // but no network connection, credentials are ok and we get a
            // status code of -1.
            if (e.getStatusCode() == STATUS_CODE_MINUS_ONE) {
                throw new TwitterPluginException(StatusCode.SERVER_UNAVAILABLE);
            }
            throw new TwitterPluginException(StatusCode.UNAUTHORIZED);
        }
        return false;
    }

    /**
     * Gets a Twitter object that has passed authentication.
     * If the credentials are wrong or we have no network,
     * a TwitterException is thrown.
     *
     * @return twitter object which is authenticated
     * @throws TwitterException
     */
    private Twitter getAuthenticatedTwitter() throws TwitterException {
        if (mTwitter == null) {
            String token = PreferencesHelper.getTokenKey(mContext);
            String tokenSecret = PreferencesHelper.getTokenSecretKey(mContext);

            AccessToken accessToken = new AccessToken(token, tokenSecret);
            mTwitter = mTwitterFactory.getOAuthAuthorizedInstance(accessToken);
            mTwitter.verifyCredentials();
        }
        return mTwitter;
    }

    /**
     * Update Twitter status
     *
     * @param status message to be set in Twitter
     */
    public void updateStatus(String status) throws TwitterPluginException {
        try {
            if (status == null) {
                Log.e(Config.LOG_TAG, "Status message is null, no update is performed");
                return;
            }
            StatusUpdate statusUpdate = new StatusUpdate(status);
            Twitter twitter = getAuthenticatedTwitter();
            twitter.updateStatus(statusUpdate);
        } catch (TwitterException e) {
            throw new TwitterPluginException(e);
        }
    }

    /**
     * Will set the paging since id for statuses
     *
     * @param statuses is list of statuses.
     */
    public void setPagingSinceForStatus(List<Status> statuses) {
        if (statuses != null && statuses.size() > 0) {
            long lastId = statuses.get(0).getId();
            PreferencesHelper.setPagingSinceId(mContext, TwitterConf.PAGING_TYPE_STATUS, lastId);
        }
    }

    /**
     * Will set the paging since id for direct messages
     *
     * @param messages is list of direct messages.
     */
    public void setPagingSinceForMessages(List<DirectMessage> messages) {
        if (messages != null && messages.size() > 0) {
            long lastId = messages.get(0).getId();
            PreferencesHelper.setPagingSinceId(mContext, TwitterConf.PAGING_TYPE_MESSAGE, lastId);
        }
    }

    /**
     * Get the statuses for the authenticated user, note that this
     * will filter out own status information.
     * @return List of statuses, will not return null
     * @throws TwitterPluginException
     */
    public List<Status> getHomeTimeline() throws TwitterPluginException {
        try {
            List<Status> filteredStatuses = new LinkedList<Status>();

            Twitter twitter = getAuthenticatedTwitter();
            Paging statusPaging = getPaging(TwitterConf.PAGING_TYPE_STATUS);
            List<Status> statuses = twitter.getHomeTimeline(statusPaging);
            if (statuses != null) {
                // Filter out own statuses
                User ownUser = getOwnUser();
                for (Status status : statuses) {
                    User user = status.getUser();
                    if (user != null && user.getId() != ownUser.getId()) {
                        filteredStatuses.add(status);
                    }
                }
            }
            return filteredStatuses;
        } catch (TwitterException e) {
            throw new TwitterPluginException(e);
        }
    }

    /**
     * Get the direct messages for the authenticated user.
     * @return List of messages, will not return null
     * @throws TwitterPluginException
     */
    public List<DirectMessage> getDirectMessages() throws TwitterPluginException {
        try {
            Twitter twitter = getAuthenticatedTwitter();
            Paging messagePaging = getPaging(TwitterConf.PAGING_TYPE_MESSAGE);
            List<DirectMessage> messages = twitter.getDirectMessages(messagePaging);
            if (messages == null) {
                return new LinkedList<DirectMessage>();
            }
            return messages;
        } catch (TwitterException e) {
            throw new TwitterPluginException(e);
        }
    }

    /**
     * Will return a list of users from Twitter
     *
     * @param twitter Twitter component
     * @return List of users, in case of error this can be null
     */
    public PagableResponseList<User> getFriendsList() throws TwitterPluginException {
        PagableResponseList <User> users;
        //if more than 100 friends
        PagableResponseList <User> userTemp;

        try {
            Twitter twitter = getAuthenticatedTwitter();
            users = twitter.getFriendsStatuses();

            if(users.hasNext()) {
                userTemp = twitter.getFriendsStatuses(users.getNextCursor());
                users.addAll(userTemp);
                while (userTemp != null && (userTemp.hasNext())){
                    userTemp = twitter.getFriendsStatuses(userTemp.getNextCursor());
                    users.addAll(userTemp);
                }
            }
        } catch (TwitterException e) {
            throw new TwitterPluginException(e);
        }
        return users;
    }

    public void logout() {
        cleanup();
        mContext = null;
    }

    /**
     * Return the correct paging information for the next PluginConstants call.
     *
     * @param pagingType TwitterConf.PAGING_TYPE_MESSAGE ||
     *            TwitterConf.PAGING_TYPE_STATUS
     * @return A Paging object used in Twitter4J calls.
     */
    private Paging getPaging(String pagingType) {
        Paging paging = new Paging();
        paging.setCount(TwitterConf.PAGING_COUNT);
        if (pagingType.equals(TwitterConf.PAGING_TYPE_MESSAGE)
                || pagingType.equals(TwitterConf.PAGING_TYPE_STATUS)) {
            long since_id = PreferencesHelper.getPagingSinceId(mContext, pagingType);
            if (since_id > 0) {
                paging.setSinceId(since_id);
            }
        }
        return paging;
    }

    /**
     * Gets the customer key.
     *
     * @param cipherText text to decrypt
     * @return the decrypted text
     */
    private static String getCustomerKey(String cipherText) {

        byte[] plainText = null;
        byte[] cipherTextArray = new BigInteger(cipherText, 16).toByteArray();
        SecretKeySpec secretKeySpec = new SecretKeySpec(new byte[] {
                0x07, 0x17, 0x02, 0x0f, 0x13, 0x05, 0x15, 0x12, 0x08, 0x00, 0x01,
                0x03, 0x16, 0x0c, 0x0d, 0x11, 0x10, 0x04, 0x06, 0x09, 0x14, 0x0a,
                0x0b, 0x0e}, "AES");

        try {
            Cipher cipher = Cipher.getInstance("AES/ECB/PKCS5Padding", "BC");
            cipher.init(Cipher.DECRYPT_MODE, secretKeySpec);

            plainText = cipher.doFinal(cipherTextArray);
            return new String(plainText, "UTF-8").trim();
        } catch (RuntimeException e) {
            Log.e(Config.LOG_TAG, "Could not translate the key, internal runtime error");
        } catch (UnsupportedEncodingException e) {
            Log.e(Config.LOG_TAG, "Encodig not supported");
        } catch (Exception e) {
            Log.e(Config.LOG_TAG, "Could not translate the key");
        }
        return "";
    }

    /**
     * Will create a twitter configuration
     * @return the configuration
     */
    private Configuration createConfiguration(String oAuthConsumerKey, String oAuthConsumerSecret) {
        ConfigurationBuilder builder = new ConfigurationBuilder();
        builder.setOAuthConsumerKey(oAuthConsumerKey);
        builder.setOAuthConsumerSecret(oAuthConsumerSecret);
        builder.setUserAgent(USER_AGENT);
        builder.setHttpRetryCount(RETRY_COUNT);
        builder.setHttpMaxTotalConnections(MAX_CONNECTIONS);
        return builder.build();
    }

    /**
     * Get the logged in user object.
     * It's garanteed that the returning object is non-null
     *
     * @return user object
     * @throws TwitterPluginException if something failed.
     */
    public User getOwnUser() throws TwitterPluginException {
        User user = null;
        try {
            Twitter twitter = getAuthenticatedTwitter();
            int id = twitter.getId();
            user = twitter.showUser(id);
        } catch (TwitterException e) {
            throw new TwitterPluginException(e);
        } catch (IllegalStateException e) {
            throw new TwitterPluginException(TwitterPluginException.StatusCode.UNAUTHORIZED);
        }
        if (user == null) {
            throw new TwitterPluginException(TwitterPluginException.StatusCode.UNAUTHORIZED);
        }
        return user;
    }

    /**
     * Cleanup used resources in Twitter
     */
    public void cleanup() {
        if (mTwitter != null) {
            try {
                mTwitter.shutdown();
            } catch (Exception e) {
                if (Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "Exception thrown when cleanup up resources. We can, however, not do anything about it.");
                }
            }
        }
        mTwitter = null;
    }
}
