/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.twitterplugin.service;

import static com.sonyericsson.eventstream.twitterplugin.PluginConstants.EVENTSTREAM_EVENT_PROVIDER_URI;
import static com.sonyericsson.eventstream.twitterplugin.PluginConstants.EVENTSTREAM_FRIEND_PROVIDER_URI;
import static com.sonyericsson.eventstream.twitterplugin.PluginConstants.EVENTSTREAM_PLUGIN_PROVIDER_URI;
import static com.sonyericsson.eventstream.twitterplugin.PluginConstants.EVENTSTREAM_SOURCE_PROVIDER_URI;

import android.app.IntentService;
import android.content.ComponentName;
import android.content.ContentResolver;
import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Intent;
import android.content.res.Resources;
import android.database.Cursor;
import android.net.Uri;
import android.net.Uri.Builder;
import android.util.Log;

import twitter4j.DirectMessage;
import twitter4j.Status;
import twitter4j.User;

import com.sonyericsson.eventstream.twitterplugin.PluginConstants;
import com.sonyericsson.eventstream.twitterplugin.R;
import com.sonyericsson.eventstream.twitterplugin.TwitterPluginApplication;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.Config;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.EventColumns;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.EventStreamIntentExtraParams;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.FriendColumns;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.InternalIntentExtraParams;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.PluginColumns;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.SourceColumns;
import com.sonyericsson.eventstream.twitterplugin.PluginConstants.TwitterConf;
import com.sonyericsson.eventstream.twitterplugin.twitter.TwitterCommunication;
import com.sonyericsson.eventstream.twitterplugin.twitter.TwitterPluginException;
import com.sonyericsson.eventstream.twitterplugin.twitter.TwitterPluginException.StatusCode;
import com.sonyericsson.eventstream.twitterplugin.utility.ContactLookupHelper;
import com.sonyericsson.eventstream.twitterplugin.utility.EventStreamHelper;
import com.sonyericsson.eventstream.twitterplugin.utility.NotificationHelper;
import com.sonyericsson.eventstream.twitterplugin.utility.PreferencesHelper;
import com.sonyericsson.eventstream.twitterplugin.view.ConfigurationActivity;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class TwitterPluginService extends IntentService {
    /** Used to indicate that this is a message type in EVENT_KEY */
    private static final String EVENT_KEY_MESSAGE_TYPE = "message";

    /** Used to indicate that this is a status type in EVENT_KEY */
    private static final String EVENT_KEY_STATUS_TYPE = "status";

    /** Supported API version in EventStream*/
    private static final int SUPPORTED_API_VERSION = 1;

    /** If we don't have an EventStream source id, this is the value to be used */
    private static final int ILLEGAL_SOURCE_ID = -1;

    /** Tells EventStream that we have status support*/
    private static final int HAS_STATUS_SUPPORT = 1;

    private static final int BULK_INSERT_MAX_COUNT = 50;

    private static final int BULK_INSERT_DELAY = 20; //ms

    /** The different states in EventStream*/
    private enum Configuration {
        CONFIGURED,
        NOT_CONFIGURED
    }

    /**
     * The wrapper around the Twitter4J API that we use to
     * communicate with
     */
    private TwitterCommunication mTwitterCommunication;

    public TwitterPluginService() {
        super("TwitterPluginService");
    }

    @Override
    protected void onHandleIntent(Intent intent) {
        if (intent == null) {
            return;
        }

        String action = intent.getAction();
        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Twitter service intent: " + action);
        }

        if (isInBadState()) {
            if (!PluginConstants.TWITTER_REGISTER_PLUGIN.equals(action)) {
                recoverFromBadStateAndRegisterAgain();
                return;
            } else {
                clearAllData();
            }
        }

        if (action != null) {
            if (mTwitterCommunication == null) {
                mTwitterCommunication = new TwitterCommunication(this);
            }

            if (PluginConstants.TWITTER_REGISTER_PLUGIN.equals(action)) {
                refreshRegistration();
            } else if (PluginConstants.TWITTER_AUTHENTICATE.equals(action)) {
                authenticate(intent);
            } else if (PluginConstants.TWITTER_LOGOUT.equals(action)) {
                logout();
            } else if (PluginConstants.TWITTER_REFRESH_REQUEST.equals(action)) {
                if (PreferencesHelper.hasToken(this) && EventStreamHelper.isPluginRegistered(getContentResolver())) {
                    int sourceId = refreshRegistration();
                    if (sourceId != ILLEGAL_SOURCE_ID) {
                        refreshTwitterInformation(sourceId);
                    }
                }
            } else if (PluginConstants.TWITTER_UPDATE_STATUS.equals(action)) {
                String status = intent.getStringExtra(EventStreamIntentExtraParams.EXTRA_STATUS_UPDATE_MESSAGE);
                if (PreferencesHelper.hasToken(this) && EventStreamHelper.isPluginRegistered(getContentResolver())) {
                    updateStatus(status);
                }
            } else if (PluginConstants.TWITTER_LOCALE_CHANGED.equals(action)) {
                refreshPluginRegistrationInformation();
            }
        }
        cleanup();
    }

    /**
     * Check if the plugin is in a bad state.
     *
     * A bad state is when some one has done clear data on the plugin while the
     * plugin was registered in EventStream or if some one cleared the
     * EventStream data while the plugin still thinks it's registered.
     *
     * @return true if in a bad state, false otherwise
     */
    private boolean isInBadState() {
        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Check to see if we are in a bad state");
        }
        if (PreferencesHelper.isRegistered(getApplicationContext())
                && !EventStreamHelper.isPluginRegistered(getContentResolver())) {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "We thought we were registered but not according to EventStream");
            }
            return true;
        }

        if (!PreferencesHelper.isRegistered(getApplicationContext())
                && EventStreamHelper.isPluginRegistered(getContentResolver())) {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "EventStream thought we were registered but we are not");
            }
            return true;
        }
        return false;
    }

    /**
     * Authenticate the current user.
     *
     * @param intent to extract the username and password
     */
    private void authenticate(Intent intent) {
        int sourceId = refreshRegistration();

        PreferencesHelper.clearAuthentication(this);

        String userName = intent.getStringExtra(InternalIntentExtraParams.USERNAME);
        String password = intent.getStringExtra(InternalIntentExtraParams.PASSWORD);
        if (authenticateTwitterAccount(userName, password)) {
            if (sourceId != ILLEGAL_SOURCE_ID) {
                refreshTwitterInformation(sourceId);
            }
        }
    }

    /**
     * This will logout from the EventStream but keep the registration.
     * This will only clear data.
     */
    private void logout() {
        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Log out");
        }
        if (mTwitterCommunication != null) {
            mTwitterCommunication.logout();
        }

        NotificationHelper.removeFromStatusBar(getApplicationContext());
        PreferencesHelper.logout(getApplicationContext());
        EventStreamHelper.clearData(getContentResolver());
        setAsLoggedOutInEventStream();
        clearStatusInSourceTable();
        ((TwitterPluginApplication) getApplication()).setState(TwitterPluginApplication.State.SERVICE_NOT_AUTHENTICATED);
    }

   /**
    * Since we are in a bad state, we must clear out everything.
    */
   private void recoverFromBadStateAndRegisterAgain() {
       if (Config.DEBUG) {
           Log.d(Config.LOG_TAG, "Recover from bad state. Clear all data and register again.");
       }
       if (mTwitterCommunication != null) {
           mTwitterCommunication.logout();
       }

       NotificationHelper.removeFromStatusBar(getApplicationContext());
       clearAllData();
       refreshRegistration();
       ((TwitterPluginApplication) getApplication()).setState(TwitterPluginApplication.State.SERVICE_NOT_CONFIGURED);
   }

   /**
    * Clear all data from preferences and EventStream
    */
   private void clearAllData() {
       PreferencesHelper.clearAll(getApplicationContext());
       EventStreamHelper.clearAll(getContentResolver());
   }

   /**
    * Clean up used resources.
    */
    private void cleanup() {
        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Cleanup used resources");
        }
        if (mTwitterCommunication != null) {
            mTwitterCommunication.cleanup();
        }
        mTwitterCommunication = null;
    }

    /**
     * Get the friends list as it is written in EventStream
     *
     * @return a Map with Twitter friend id as key and EventStreamFriend as
     *         value
     */
    private Map<Integer, EventStreamFriend> getFriendsListFromEventStream() {
        ContentResolver cr = getContentResolver();
        Map<Integer, EventStreamFriend> friendsList = new Hashtable<Integer, EventStreamFriend>();
        Cursor cursor = null;
        try {
            cursor = cr.query(PluginConstants.EVENTSTREAM_FRIEND_PROVIDER_URI, null, null, null, null);
            if (cursor != null && cursor.moveToFirst()) {
                do {
                    int eventStreamId = cursor.getInt(cursor.getColumnIndex(FriendColumns._ID));
                    String rawContactUri = cursor.getString(cursor.getColumnIndex(FriendColumns.CONTACTS_REFERENCE));
                    String userData = cursor.getString(cursor.getColumnIndex(FriendColumns.FRIEND_KEY));
                    try {
                        EventStreamFriend friend = new EventStreamFriend();
                        Integer twitterFriendId = new Integer(userData);
                        friend.userId = twitterFriendId.intValue();
                        friend.eventStreamId = eventStreamId;
                        friend.rawContactUri = rawContactUri;
                        friendsList.put(twitterFriendId, friend);
                    } catch (NumberFormatException nfe) {
                        Log.e(Config.LOG_TAG, "Error adding friend into list");
                    }
                } while (cursor.moveToNext());
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return friendsList;
    }


   /**
    * This will refresh the friends list in EventStream.
    * If any friend is removed or added it will be reflected
    * in the EventStream.
    *
    * @param twitter
    * @return
    */
    private void refreshFriendsListInEventStream(int sourceId) throws TwitterPluginException {

       Map<Integer, EventStreamFriend> eventStreamFriends = getFriendsListFromEventStream();

       // if we get null from this method, something went wrong
       List<User> twitterFriends = mTwitterCommunication.getFriendsList();
       if (twitterFriends == null) {
           return;
       }

       // Insert or update the friend
       for (User user : twitterFriends) {
           int userId = user.getId();
           EventStreamFriend friend = eventStreamFriends.get(userId);
           if (friend != null) {
               // Remove the known friend from the event stream list
               eventStreamFriends.remove(user.getId());
           } else {
               Integer id = insertFriendIntoEventStream(user, sourceId);
               if (Config.DEBUG && id != null) {
                   Log.d(Config.LOG_TAG, "Friend added id:" + id);
               }
           }
       }
       // Clean up not used friends and content
       Set<Integer> keySet = eventStreamFriends.keySet();
       for (Integer eventStreamId : keySet) {
           ContentResolver cr = getContentResolver();
           cr.delete(EVENTSTREAM_EVENT_PROVIDER_URI, EventColumns.FRIEND_KEY + "=?", new String[] {eventStreamId.toString()});
           cr.delete(EVENTSTREAM_FRIEND_PROVIDER_URI, FriendColumns.FRIEND_KEY + "=?", new String[] {eventStreamId.toString()});
       }
   }

    /**
     * Update the current status in EventStream.
     *
     * @param status to set
     * @param timestamp to set
     */
    private void updateStatusInEventStream(String status, long timestamp) {
        if (status == null) {
            return;
        }
        ContentValues contentValues = new ContentValues();
        contentValues.put(SourceColumns.CURRENT_STATUS, status);
        contentValues.put(SourceColumns.STATUS_TIMESTAMP, timestamp);
        getContentResolver().update(PluginConstants.EVENTSTREAM_SOURCE_PROVIDER_URI, contentValues, null, null);
    }

    /**
     * Send a new status message to Twitter.
     *
     * @param status The new status message.
     */
    public void updateStatus(String status) {
        try {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Performing update status: " + status);
            }
            if (status != null && status.length() <= TwitterConf.MAX_CHARACTERS) {
                mTwitterCommunication.updateStatus(status);
                updateStatusInEventStream(status, System.currentTimeMillis());
            }
        } catch (TwitterPluginException e) {
            String errorMessage = e.getStatusCodeNotificationMessage(this);
            TwitterPluginApplication application = (TwitterPluginApplication) getApplication();
            boolean visible = application.isConfigurationActivityVisible();
            if (!visible && errorMessage != null) {
                NotificationHelper.notifyError(this, errorMessage);
            }

            if (e.getStatusCode() == StatusCode.UNAUTHORIZED) {
                logout();
            }
        }
    }

    /**
     * Authenticate a username and password towards Twitter using OAuth.
     *
     * @param userName The username for this account.
     * @param password The password for this account.
     * @return True if login was successful, false otherwise.
     */
    boolean authenticateTwitterAccount(String userName, String password) {
        try {
            ((TwitterPluginApplication) getApplication()).setState(TwitterPluginApplication.State.SERVICE_AUTHENTICATION_IN_PROGRESS);

            boolean authenticated = mTwitterCommunication.authenticateTwitterAccount(userName, password);
            if (authenticated) {
                String screenName = userName;
                User ownUser = mTwitterCommunication.getOwnUser();
                if (ownUser != null) {
                    // Set the screen name, input user name can be both email and account name
                    screenName = ownUser.getScreenName();
                }
                setPluginAsConfiguredInEventStream(screenName);
                PreferencesHelper.setAcceptedDisclaimer(getApplicationContext());
                ((TwitterPluginApplication) getApplication()).setState(TwitterPluginApplication.State.SERVICE_AUTHENTICATION_SUCCESS);
                return true;
            }
            ((TwitterPluginApplication) getApplication()).setState(TwitterPluginApplication.State.SERVICE_AUTHENTICATION_FAILED);

        } catch (TwitterPluginException e) {
            TwitterPluginApplication application = (TwitterPluginApplication) getApplication();
            application.setState(TwitterPluginApplication.State.SERVICE_AUTHENTICATION_FAILED);

            String errorMessage = e.getStatusCodeNotificationMessage(this);
            boolean visible = application.isConfigurationActivityVisible();
            if (!visible && errorMessage != null) {
                NotificationHelper.notifyError(this, errorMessage);
            }
        }
        return false;
    }

    /**
     * This will sync all Twitter information we need with the EventStream
     * and do the autolinking with a Twitter contact.
     */
    private void refreshTwitterInformation(int sourceId) {
        NotificationHelper.notifyInStatusBar(this, R.drawable.notif_data_fetch_ongoing_icn,
                getResources().getString(R.string.ts_twitter_notify_fetch_title_txt),
                getResources().getString(R.string.ts_twitter_notify_fetch_ongoing_txt));

        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "refreshTwitterInformation");
        }

        try {
            // update status
            User ownUser = mTwitterCommunication.getOwnUser();
            Status ownStatus = ownUser.getStatus();
            if (ownStatus != null && ownStatus.getText() != null && ownStatus.getCreatedAt() != null) {
                updateStatusInEventStream(ownStatus.getText(), ownStatus.getCreatedAt().getTime());
            }

            refreshFriendsListInEventStream(sourceId);
            // Get home time line
            List<Status> statuses = mTwitterCommunication.getHomeTimeline();
            ArrayList<ContentValues> statusValues = getContentValuesForStatuses(
                    statuses, sourceId);

            // get direct messages
            List<DirectMessage> messages = mTwitterCommunication.getDirectMessages();
            ArrayList<ContentValues> messageValues = getContentValuesForMessages(
                    messages, sourceId);

            statusValues.addAll(messageValues);

            List<ContentValues> bulkValues = new ArrayList<ContentValues>();
            int inserted = 0;
            for (int i = 0; i < statusValues.size(); i++) {

                bulkValues.add(statusValues.get(i));

                if (bulkValues.size() >= BULK_INSERT_MAX_COUNT) {
                    inserted +=
                        getContentResolver().bulkInsert(EVENTSTREAM_EVENT_PROVIDER_URI,
                                (ContentValues[])bulkValues.toArray(new ContentValues[bulkValues.size()]));
                    bulkValues.clear();
                    // Give eventstream some time to run queries
                    try {
                        Thread.sleep(BULK_INSERT_DELAY);
                    } catch (InterruptedException e) {
                        // Do nothing
                    }
                }
            }

            if (bulkValues.size() > 0) {
                inserted +=
                    getContentResolver().bulkInsert(EVENTSTREAM_EVENT_PROVIDER_URI,
                            (ContentValues[])bulkValues.toArray(new ContentValues[bulkValues.size()]));
            }

            if (inserted != 0) {
                mTwitterCommunication.setPagingSinceForStatus(statuses);
                mTwitterCommunication.setPagingSinceForMessages(messages);
            }

            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Inserted " + inserted + " new events.");
            }

            refreshEventStreamLookupKey(ownUser.getScreenName());

            NotificationHelper.removeFromStatusBar(this);

        } catch (TwitterPluginException e) {
            String errorMessage = e.getStatusCodeNotificationMessage(this);
            boolean visible = ((TwitterPluginApplication)getApplication()).isConfigurationActivityVisible();
            if (!visible && errorMessage != null) {
                NotificationHelper.notifyError(this, errorMessage);
            }

            if (e.getStatusCode() == StatusCode.UNAUTHORIZED) {
                logout();
            }
        } catch (Exception e) {
            NotificationHelper.removeFromStatusBar(this);
            Log.e(Config.LOG_TAG, "RefreshTwitterInformation failed.", e);
        }
    }

    /**
     * Create a ContentValues list of Statuses.
     *
     * @param statues that we want to create the list of
     * @param sourceId that we are registered with
     * @return A list of ContentValues based on the information from
     *         Status
     */
    public ArrayList<ContentValues> getContentValuesForStatuses(List<Status> statuses,
            int sourceId)  {
        ArrayList<ContentValues> values = new ArrayList<ContentValues>();
        for (Status status : statuses) {
            values.add(createContentValuesFromStatus(status, sourceId));
        }
        return values;
    }

    /**
     * This will create a status ContentValues object.
     *
     * @param status information to be used
     * @param sourceId that this message belongs to
     * @return a ContentValues instance
     */
   public ContentValues createContentValuesFromStatus(Status status, int sourceId) {
       User user = status.getUser();
       ContentValues event = new ContentValues();
       event.put(EventColumns.SOURCE_ID, sourceId);
       event.put(EventColumns.FRIEND_KEY, Long.toString(user.getId()));
       event.put(EventColumns.MESSAGE, status.getText());
       event.put(EventColumns.IMAGE_URI, user.getProfileImageURL().toString());
       event.put(EventColumns.PUBLISHED_TIME, status.getCreatedAt().getTime());
       String uniqueKey = user.getScreenName() + PluginConstants.EVENT_KEY_DATA_SEPARATOR +
           EVENT_KEY_STATUS_TYPE + PluginConstants.EVENT_KEY_DATA_SEPARATOR + status.getId();
       event.put(EventColumns.EVENT_KEY, uniqueKey);
       event.put(EventColumns.PERSONAL, 0);
       event.put(EventColumns.OUTGOING, 0);
       return event;
   }

    /**
     * Create a ContentValues list of DirectMessages.
     *
     * @param messages that we want to create the list of
     * @param sourceId that we are registered with
     * @return A list of ContentValues based on the information from
     *         DirectMessage
     */
   public ArrayList<ContentValues> getContentValuesForMessages(List<DirectMessage> messages,
            int sourceId)  {
        ArrayList<ContentValues> values = new ArrayList<ContentValues>();
        for (DirectMessage message : messages) {
            values.add(createContentValuesFromMessage(message, sourceId));
        }
        return values;
    }

   /**
    * This will create a message ContentValues object.
    *
    * @param message information to be used
    * @param sourceId that this message belongs to
    * @return a ContentValues instance
    */
   public ContentValues createContentValuesFromMessage(DirectMessage message, int sourceId) {
       User user = message.getSender();
       ContentValues event = new ContentValues();
       event.put(EventColumns.SOURCE_ID, sourceId);
       event.put(EventColumns.FRIEND_KEY, Integer.toString(user.getId()));
       event.put(EventColumns.MESSAGE, message.getText());
       event.put(EventColumns.IMAGE_URI, user.getProfileImageURL().toString());
       event.put(EventColumns.PUBLISHED_TIME, message.getCreatedAt().getTime());
       String uniqueKey = user.getScreenName() + PluginConstants.EVENT_KEY_DATA_SEPARATOR +
           EVENT_KEY_MESSAGE_TYPE + PluginConstants.EVENT_KEY_DATA_SEPARATOR + message.getId();
       event.put(EventColumns.EVENT_KEY, uniqueKey);
       event.put(EventColumns.PERSONAL, 0);
       event.put(EventColumns.OUTGOING, 0);
       return event;
   }

    /**
     * This will insert a friend in the EventStreams event table
     * @param user to insert
     * @return EventStream id for the friend.
     */
    private Integer insertFriendIntoEventStream(User user, int sourceId) {
        Integer id = null;
        ContentValues values = new ContentValues();

        values.put(FriendColumns.SOURCE_ID, sourceId);
        values.put(FriendColumns.DISPLAY_NAME, user.getName());
        values.put(FriendColumns.FRIEND_KEY, Integer.toString(user.getId()));
        values.put(FriendColumns.PROFILE_IMAGE_URI,
                user.getProfileImageURL().toString());

        Uri uri = getContentResolver().insert(PluginConstants.EVENTSTREAM_FRIEND_PROVIDER_URI, values);
        if (uri != null) {
            String eventStreamId = uri.getLastPathSegment();
            id = new Integer(eventStreamId);
        }
        return id;
    }

    /**
     * Make sure that we are in sync with contacts and EventStream.
     *
     * @param accountName that should be to get the correct sync account in
     *            contacts database.
     */
    private void refreshEventStreamLookupKey(String accountName) {
        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Refresh EventStream lookup key");
        }

        Map<Integer, EventStreamFriend> eventStreamFriends = getFriendsListFromEventStream();
        Collection<EventStreamFriend> values = eventStreamFriends.values();
        for ( EventStreamFriend friendData : values) {
            if (friendData.rawContactUri == null || "".equals(friendData.rawContactUri)) {
                friendData.rawContactUri = ContactLookupHelper.getRawContactUriInContacts(getContentResolver(),
                        friendData.userId);
                if (friendData.rawContactUri != null) {
                    updateLookupKeyInEventStream(friendData);
                }
            } else {
                if (!ContactLookupHelper.rawContactExist(getContentResolver(), friendData.rawContactUri)) {
                    if (Config.DEBUG) {
                        Log.d(Config.LOG_TAG, "Couldn't find lookup key so clearing it for Twitter user " + friendData.userId);
                    }
                    friendData.rawContactUri = null;
                    updateLookupKeyInEventStream(friendData);
                }
            }
        }
    }

    /**
     * This will update the lookup key in the friend column.
     *
     * @param friendData that should be updated.
     */
    private void updateLookupKeyInEventStream(EventStreamFriend friendData) {
        if (friendData == null) {
            return;
        }

        try {
            ContentValues values = new ContentValues();
            if (friendData.rawContactUri != null) {
                values.put(FriendColumns.CONTACTS_REFERENCE, friendData.rawContactUri);
            } else {
                values.putNull(FriendColumns.CONTACTS_REFERENCE);
            }
            getContentResolver().update(PluginConstants.EVENTSTREAM_FRIEND_PROVIDER_URI,
                    values, FriendColumns._ID + "= ?",
                    new String[] {String.valueOf(friendData.eventStreamId)});
        } catch (Exception e) {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Unexpected exception thrown when updating friend data. Cause is = " + e.getMessage());
            }
        }
    }

    /**
     * This will register the plugin if it's not registered
     * and make sure that we can be found in the source table.
     *
     * @return sourceId
     */
    public int refreshRegistration() {
        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Start registration of plugin.");
        }

        if (!EventStreamHelper.isPluginRegistered(getContentResolver())) {
            registerPlugin();
        }

        int sourceId = getSourceTableId();
        if (sourceId == ILLEGAL_SOURCE_ID) {
            sourceId = registerInSourceTable();
        }
        return sourceId;
    }

    /**
     * Register the plugin in EventStream
     */
    private void registerPlugin() {
        String configText = getString(R.string.ts_twitter_register_txt);
        ContentValues contentValues = createPluginContentValues(
                Configuration.NOT_CONFIGURED, configText);
        insertPluginRegistrationData(contentValues);
        PreferencesHelper.setRegistered(getApplicationContext());
    }

    /**
     * This is used when we want to refresh the plugin
     * information we have in EventStream.
     */
    private void refreshPluginRegistrationInformation() {
        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Refresh plugin registration information");
        }
        try {
            if (EventStreamHelper.isPluginRegistered(getContentResolver()) && mTwitterCommunication != null) {
                try {
                    User user = mTwitterCommunication.getOwnUser();
                    if (user != null) {
                        if (Config.DEBUG) {
                            Log.d(Config.LOG_TAG, "A user is logged in. Will update the plugin registration");
                        }
                        setPluginAsConfiguredInEventStream(user.getScreenName());
                    }
                } catch (TwitterPluginException e) {
                    if (e.getStatusCode() == TwitterPluginException.StatusCode.UNAUTHORIZED) {
                        if (Config.DEBUG) {
                            Log.d(Config.LOG_TAG, "No user is logged in. Will update the plugin registration");
                        }
                        setAsLoggedOutInEventStream();
                    } else {
                        throw new TwitterPluginException(e.getStatusCode());
                    }
                }
            }
        } catch (TwitterPluginException e) {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Unable to refresh the registered Twitter plugin information.");
            }
        }
    }

    /**
     * Insert the registration data in EventStream content provider
     *
     * @param contentValues of the information we want to register
     */
    private void insertPluginRegistrationData(ContentValues contentValues) {
        ContentValues pluginRegistrationValues = contentValues;
        ContentResolver contentResolver = getContentResolver();
        contentResolver.insert(EVENTSTREAM_PLUGIN_PROVIDER_URI, pluginRegistrationValues);
    }

    /**
     * This will update the EventStream with text to show that we are logged in.
     *
     * @param userName of the logged in Twitter user.
     */
    private void setPluginAsConfiguredInEventStream(String userName) {
        String configText = getString(R.string.ts_twitter_logout_txt);
        configText += " " + userName;
        ContentValues contentValues = createPluginContentValues(
                Configuration.CONFIGURED, configText);
        updatePluginRegistrationData(contentValues);
    }

    /**
     * This will update the EventStream with text to show that we are logged out.
     */
    private void setAsLoggedOutInEventStream() {
        String configText = getString(R.string.ts_twitter_register_txt);
        ContentValues contentValues = createPluginContentValues(
                Configuration.NOT_CONFIGURED, configText);
        updatePluginRegistrationData(contentValues);
    }

    /**
     * Update the registration data in EventStream content provider
     *
     * @param contentValues of the information we want to register
     */
    private void updatePluginRegistrationData(ContentValues contentValues) {
        ContentValues pluginRegistrationValues = contentValues;
        ContentResolver contentResolver = getContentResolver();
        contentResolver.update(EVENTSTREAM_PLUGIN_PROVIDER_URI, pluginRegistrationValues, null, null);
    }

    /**
     * Create the ContentValues object we need in order to register
     * us in the EventStream.
     *
     * @param configStatus the status of the configuration (logged in or not)
     * @return a ContentValues object to be used when loggin in.
     */
    private ContentValues createPluginContentValues(Configuration configStatus, String configureText) {
        String packageName = getPackageName();
        ComponentName configComponentName = new ComponentName(packageName, ConfigurationActivity.class.getName());
        Resources resources = getResources();
        ContentValues values = new ContentValues();
        values.put(PluginColumns.NAME, resources.getString(R.string.ts_twitter_name_txt));

        Builder iconUriBuilder = new Uri.Builder().scheme(ContentResolver.SCHEME_ANDROID_RESOURCE)
        .authority(getPackageName()).appendPath(Integer.toString(R.drawable.twitter_icn));
        values.put(PluginColumns.ICON_URI, iconUriBuilder.toString());

        values.put(PluginColumns.API_VERSION, SUPPORTED_API_VERSION);
        values.put(PluginColumns.PLUGIN_KEY, PluginConstants.PLUGIN_IDENTIFIER);
        values.put(PluginColumns.CONFIGURATION_ACTIVITY, configComponentName.flattenToShortString());
        values.put(PluginColumns.CONFIGURATION_STATE, configStatus.ordinal());
        values.put(PluginColumns.CONFIGURATION_TEXT, configureText);
        values.put(PluginColumns.STATUS_SUPPORT, HAS_STATUS_SUPPORT);
        values.put(PluginColumns.STATUS_TEXT_MAX_LENGTH, TwitterConf.MAX_CHARACTERS);
        return values;
    }

    /**
     * Read the source id value from the EventStream source table.
     *
     * @return the source id
     */
    private int getSourceTableId() {
        Cursor cursor = null;
        int eventStreamSourceId = ILLEGAL_SOURCE_ID;
        try {
            ContentResolver contentResolver = getContentResolver();
            cursor = contentResolver.query(EVENTSTREAM_SOURCE_PROVIDER_URI, new String[]{SourceColumns._ID}, null, null, null);
            if (cursor != null && cursor.moveToFirst()) {
                eventStreamSourceId = cursor.getInt(cursor.getColumnIndex(SourceColumns._ID));
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }

        return eventStreamSourceId;
    }

    /**
     * Register the plugin in the source table.
     *
     * @return the EventStream source id.
     */
    private int registerInSourceTable() {
        ContentResolver contentResolver = getContentResolver();
        ContentValues sourceRegistrationValues = createSourceRegistrationValues(null);
        Uri uri = contentResolver.insert(EVENTSTREAM_SOURCE_PROVIDER_URI, sourceRegistrationValues);
        int eventStreamSourceId = ILLEGAL_SOURCE_ID;
        if (uri != null) {
            Long id = ContentUris.parseId(uri);
            eventStreamSourceId = id.intValue();
        }
        return eventStreamSourceId;
    }

   /**
    * Clear the status and time stamp in the source table in EventStream.
    */
   private void clearStatusInSourceTable() {
       ContentResolver contentResolver = getContentResolver();
       ContentValues contentValues = new ContentValues();
       contentValues.putNull(SourceColumns.STATUS_TIMESTAMP);
       contentValues.putNull(SourceColumns.CURRENT_STATUS);
       contentResolver.update(EVENTSTREAM_SOURCE_PROVIDER_URI, contentValues, null, null);
   }

    /**
     * Build up the registration values for the source table.
     *
     * @param currentStatus our current status, if null nothing will be written
     * @return ContentValues object
     */
    public ContentValues createSourceRegistrationValues(String currentStatus) {
        Resources resources = getResources();
        ContentValues sourceRegistrationValues = new ContentValues();

        sourceRegistrationValues.put(SourceColumns.NAME, resources.getString(R.string.ts_twitter_name_txt));

        Builder iconUriBuilder = new Uri.Builder().scheme(ContentResolver.SCHEME_ANDROID_RESOURCE)
        .authority(getPackageName()).appendPath(Integer.toString(R.drawable.twitter_icn));
        sourceRegistrationValues.put(SourceColumns.ICON_URI, iconUriBuilder.toString());

        sourceRegistrationValues.put(SourceColumns.ENABLED, (short)1);
        if (currentStatus != null) {
            sourceRegistrationValues.put(SourceColumns.CURRENT_STATUS, currentStatus);
        }
        return sourceRegistrationValues;
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        mTwitterCommunication = null;
    }

    /**
     * Class to hold an event stream friend content
     */
    static class EventStreamFriend {
        int userId;
        int eventStreamId;
        String rawContactUri;
    }
}