/*
 * Copyright (C) 2010 Sony Ericsson Mobile Communications AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
 *
 */

package com.sonyericsson.eventstream.rssplugin;

import com.sonyericsson.eventstream.rssplugin.PluginConstants.Config;
import com.sonyericsson.eventstream.rssplugin.PluginConstants.EventStream;
import com.sonyericsson.eventstream.rssplugin.PluginConstants.ServiceIntentCmd;
import com.sonyericsson.eventstream.rssplugin.db.RssFeedProvider;

import org.apache.http.HttpResponse;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.DefaultHttpClient;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;
import org.xmlpull.v1.XmlPullParserFactory;

import android.app.IntentService;
import android.content.ComponentName;
import android.content.ContentResolver;
import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.net.Uri;
import android.net.Uri.Builder;
import android.util.Log;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.TimeZone;

/**
 * This service will handle registration of the plugin and rss source,
 * and provides event data to the EventStream.
 */
public class RssPluginService extends IntentService {

    /** The version of the plugin */
    private static final int PLUGIN_VERSION = 1;

    /** Rss feed channel tag */
    private static final String CHANNEL_TAG = "channel";

    /** Rss feed item tag */
    private static final String ITEM_TAG = "item";

    /** Rss feed title tag */
    private static final String TITLE_TAG = "title";

    /** Rss feed link tag */
    private static final String LINK_TAG = "link";

    /** Rss feed description tag */
    private static final String DESCRIPTION_TAG = "description";

    /** Rss feed guid tag */
    private static final String GUID_TAG = "guid";

    /** Rss feed published date tag */
    private static final String PUB_DATE_TAG = "pubDate";

    /** Rss feed media thumbnail tag */
    private static final String THUMBNAIL_TAG = "thumbnail";

    /** Rss feed media thumnail url key */
    private static final String THUMBNAIL_URL_KEY = "url";

    /** Format of the date (Thu, 01 Jan 2004 19:48:21 GMT) */
    private static final String DATE_FORMAT = "EEE, dd MMM yyyy HH:mm:ss Z";

    /** The Id of the Rss Plugin Source. If the source is not registered in Timescape this value is set to -1 */
    private int mRssSourceId;

    private SimpleDateFormat dateFormat;

    public RssPluginService() {
        super("RssPluginService");
    }

    @Override
    protected void onHandleIntent(Intent intent) {
        mRssSourceId = getSourceId();
        dateFormat = new SimpleDateFormat(DATE_FORMAT, Locale.ENGLISH);
        dateFormat.setTimeZone(TimeZone.getTimeZone("GMT"));

        if (intent != null) {
            String serviceCommand = intent.getStringExtra(ServiceIntentCmd.SERVICE_COMMAND_KEY);
            if (ServiceIntentCmd.REGISTER_PLUGIN.equals(serviceCommand)) {
                if (Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "Start RSS service for registering plugin.");
                }
                registerOrUpdatePlugin();
            } else if (ServiceIntentCmd.REFRESH_REQUEST.equals(serviceCommand)) {
                if (mRssSourceId != -1) { // If the source is registered
                    if (Config.DEBUG) {
                        Log.d(Config.LOG_TAG, "Start RSS service for refresh request.");
                    }
                    if (isSourceEnabled()) { // If the source is enabled by the user
                        refreshAllChannels();
                    }
                } else {
                    if (Config.DEBUG) {
                        Log.d(Config.LOG_TAG, "RSS source is not registered. Ignoring refresh request");
                    }
                }
            }
        }
    }

    private int getSourceId() {
        int sourceId;
        Cursor cursor = null;
        try {
            cursor = getContentResolver().query(EventStream.EVENTSTREAM_SOURCE_PROVIDER_URI, new String[]{EventStream.SourceColumns.ID_COLUMN}, null, null, null);
            if (cursor != null && cursor.moveToFirst() && cursor.getCount() > 0) { // Retrieve the source id
                sourceId = cursor.getInt(cursor.getColumnIndexOrThrow(EventStream.SourceColumns.ID_COLUMN));
            } else {
                sourceId = -1;
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return sourceId;
    }

    private boolean isSourceEnabled() {
        int enabled;
        Cursor cursor = null;
        try {
            cursor = getContentResolver().query(EventStream.EVENTSTREAM_SOURCE_PROVIDER_URI, new String[]{EventStream.SourceColumns.ID_COLUMN, EventStream.SourceColumns.ENABLED}, null, null, null);
            if (cursor != null && cursor.moveToFirst() && cursor.getCount() > 0) { // Retrieve the source id
                enabled = cursor.getInt(cursor.getColumnIndexOrThrow(EventStream.SourceColumns.ENABLED));
            } else {
                enabled = 0;
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return (enabled == 1);
    }

    /**
     * Registeres the plugin
     */
    private void registerOrUpdatePlugin() {
        if(Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Start registration of plugin.");
        }
        // first we update/insert into the plugins table
        insertOrUpdatePlugin();

        // Next, we update/insert into the source table
        insertOrUpdateSource();
    }

    /**
     * Inserts this plugin into the EventStream plugin table. If the plugin is already stored, plugin data is updated
     */
    private void insertOrUpdatePlugin() {
        Cursor cursor = null;
        try {
            cursor = getContentResolver().query(EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI, null, null, null, null);

            ContentValues pluginRegistrationValues = new ContentValues();
            pluginRegistrationValues.put(EventStream.PluginColumns.STATUS_SUPPORT, EventStream.StatusSupport.HAS_SUPPORT_FALSE);
            pluginRegistrationValues.put(EventStream.PluginColumns.NAME, getResources().getString(R.string.rss_plugin_name));
            Builder iconUriBuilder = new Uri.Builder()
                            .scheme(ContentResolver.SCHEME_ANDROID_RESOURCE)
                            .authority(getPackageName())
                            .appendPath(Integer.toString(R.drawable.icon));
            pluginRegistrationValues.put(EventStream.PluginColumns.ICON_URI, iconUriBuilder.toString());
            pluginRegistrationValues.put(EventStream.PluginColumns.API_VERSION, PLUGIN_VERSION);
            ComponentName componentName = new ComponentName(getPackageName(), RssPluginConfig.class.getName());
            pluginRegistrationValues.put(EventStream.PluginColumns.CONFIGURATION_ACTIVITY, componentName.flattenToShortString());
            pluginRegistrationValues.put(EventStream.PluginColumns.CONFIGURATION_STATE, EventStream.ConfigState.CONFIGURATION_NOT_NEEDED);

            if (cursor != null && cursor.moveToNext() && cursor.getCount() > 0) { // Update the plugin data
                getContentResolver().update(EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI, pluginRegistrationValues, null, null);
                if(Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "Update of registration.");
                }
            } else { // Register ourself for the first time
                getContentResolver().insert(EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI, pluginRegistrationValues);
                if(Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "Registered RSS plugin in Event Stream.");
                }
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
    };

    /**
     * Inserts this source into the EventStream sources table. If the source is already stored, source data is updated
     */
    private void insertOrUpdateSource() {
        Cursor cursor = null;
        try {
            ContentValues sourceRegistrationValues = new ContentValues();
            sourceRegistrationValues.putNull(EventStream.SourceColumns.CURRENT_STATUS);
            sourceRegistrationValues.put(EventStream.SourceColumns.NAME, getResources().getString(R.string.rss_source_name));
            Builder iconUriBuilder = new Uri.Builder()
                        .scheme(ContentResolver.SCHEME_ANDROID_RESOURCE)
                        .authority(getPackageName())
                        .appendPath(Integer.toString(R.drawable.icon));
            sourceRegistrationValues.put(EventStream.SourceColumns.ICON_URI, iconUriBuilder.toString());

            cursor = getContentResolver().query(EventStream.EVENTSTREAM_SOURCE_PROVIDER_URI, new String[]{EventStream.SourceColumns.ID_COLUMN}, null, null, null);
            if (cursor != null && cursor.moveToFirst() && cursor.getCount() > 0) { // Update the source data
                if(Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "Updating the source registration.");
                }
                mRssSourceId = cursor.getInt(cursor.getColumnIndexOrThrow(EventStream.SourceColumns.ID_COLUMN));
                int updated = getContentResolver().update(EventStream.EVENTSTREAM_SOURCE_PROVIDER_URI, sourceRegistrationValues, null, null);
                if(Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "Updating the source registration " + (updated == 1 ? "succeded" : "failed"));
                }
            } else { // Register the source for the first time
                sourceRegistrationValues.put(EventStream.SourceColumns.ENABLED, 1);
                Uri uri = getContentResolver().insert(EventStream.EVENTSTREAM_SOURCE_PROVIDER_URI, sourceRegistrationValues);
                mRssSourceId = (int)ContentUris.parseId(uri);
                if(Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "Registered RSS source in Event Stream: " + uri);
                }
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
    }

    /**
     * Refreshes the Rss feed data for all channels.
     */
    private void refreshAllChannels() {
        if (!isConnected()) {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Could not connect to network.");
            }
        }
        List<Channel> channelDbList = getAllChannelsFromDb();
        List<Channel> channelList = new ArrayList<Channel>();
        for (Channel channel: channelDbList) {
            Channel parsedChannel = parseChannel(channel);
            if (parsedChannel != null) {
                channelList.add(parsedChannel);
            }
        }
        // Update the EventStream Events table
        synchronizeEvents(channelList);
    }

    /**
     * Parses the Rss feed data for the input channel url as a Channel object.
     *
     * @param channelUrl the channel Url.
     * @return the channel object.
     */
    private Channel parseChannel(Channel channel) {
        if (channel != null && channel.getUrl() != null && !channel.getUrl().equals("")) {
            try {
                channel = parseFeed(channel.getUrl());
            } catch (XmlPullParserException e) {
                if (Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "Could not parse channel feed.");
                }
                e.printStackTrace();
            } catch (URISyntaxException e) {
                if (Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "Could not parse channel feed.");
                }
                e.printStackTrace();
            } catch (IOException e) {
                if (Config.DEBUG) {
                    Log.d(Config.LOG_TAG, "Unable to refresh channel with url: "+channel.getUrl());
                }
                e.printStackTrace();
            }
        }
        return channel;
    }

    /**
     * Insert the Rss channels events in the event table.
     *
     * @param channel the channel.
     * @return the number of items added.
     */
    protected int synchronizeEvents(List<Channel> channelList) {
        ContentResolver contentResolver = getContentResolver();
        ArrayList<ContentValues> eventstreamEventList = new ArrayList<ContentValues>();
        for (Channel channel : channelList) {
            List<Item> itemList = channel.getItemList();
            for (Item item : itemList) {
                // Check if the item is the the RssPlugin database
                Cursor cursor = null;
                Integer itemId = null;
                try {
                    cursor = contentResolver.query(
                            RssFeedProvider.ITEM_CONTENT_URI,
                            null,
                            RssFeedProvider.ITEM_URL_COLUMN + " = ?",
                            new String[]{String.valueOf(item.getGuid())},
                            null);
                    if (cursor != null && cursor.getCount() != 0){
                        cursor.moveToFirst();
                        itemId = cursor.getInt(cursor.getColumnIndexOrThrow(RssFeedProvider.ITEM_ID_COLUMN));
                    } else {
                        ContentValues values = new ContentValues();
                        values.put(RssFeedProvider.ITEM_URL_COLUMN, item.getGuid());
                        values.put(RssFeedProvider.ITEM_CHANNEL_ID_COLUMN, channel.getId());
                        Uri uri = getContentResolver().insert(RssFeedProvider.ITEM_CONTENT_URI, values);
                        if (uri != null) {
                            itemId = (int)ContentUris.parseId(uri);
                        }
                    }
                } finally {
                    if (cursor != null) {
                        cursor.close();
                    }
                }
                if (itemId != null) {
                    try {
                        cursor = contentResolver.query(
                                EventStream.EVENTSTREAM_EVENT_PROVIDER_URI,
                                null,
                                EventStream.EventColumns.EVENT_KEY + " = ?",
                                new String[]{String.valueOf(itemId)},
                                null);
                        if (cursor != null && cursor.getCount() == 0) {
                            ContentValues values = new ContentValues();
                            values.put(EventStream.EventColumns.EVENT_KEY, itemId);
                            values.put(EventStream.EventColumns.SOURCE_ID, mRssSourceId);
                            values.put(EventStream.EventColumns.TITLE, channel.getTitle());
                            values.put(EventStream.EventColumns.MESSAGE, item.getTitle());
                            values.put(EventStream.EventColumns.PUBLISHED_TIME, item.getTimeOfDate());
                            values.put(EventStream.EventColumns.IMAGE_URI, item.getMediaThumbnail());
                            values.put(EventStream.EventColumns.OUTGOING, 0);
                            values.put(EventStream.EventColumns.PERSONAL, 0);
                            eventstreamEventList.add(values);
                        }
                    } finally {
                        if (cursor != null) {
                            cursor.close();
                        }
                    }
                }
            }
        }
        ContentValues[] bulkContentValues = new ContentValues[eventstreamEventList.size()];
        int contentValuesIndex = 0;
        for (ContentValues values : eventstreamEventList) {
            bulkContentValues[contentValuesIndex] = values;
            contentValuesIndex++;
        }
        return contentResolver.bulkInsert(EventStream.EVENTSTREAM_EVENT_PROVIDER_URI, bulkContentValues);
    }

    /**
     * Parse the RSS feed url as a channel
     *
     * @param url The RSS feed url
     * @return the parsed Channel
     * @throws XmlPullParserException
     * @throws IOException
     * @throws URISyntaxException
     */
    private Channel parseFeed(String url) throws XmlPullParserException, IOException, URISyntaxException {
        Channel channel = null;
        XmlPullParserFactory factory = XmlPullParserFactory.newInstance();
        factory.setNamespaceAware(true);
        XmlPullParser xpp = factory.newPullParser();
        InputStream urlData = getUrlData(url);
        if (urlData != null) {
            xpp.setInput(new InputStreamReader(urlData));
            int event;
            while ((event = xpp.next()) != XmlPullParser.END_DOCUMENT) {
                if (event == XmlPullParser.START_TAG) {
                    String tag = xpp.getName().trim();
                    // Find the Channel tag
                    if (CHANNEL_TAG.equalsIgnoreCase(tag)) {
                        channel = parseChannelTag(xpp);
                    }
                }
            }
        }
        return channel;
    }

    /**
     * Parses the Channel tag values from the XMLPullParser
     *
     * @param xpp The parser
     * @return the parsed Channel object
     * @throws XmlPullParserException
     * @throws IOException
     */
    private Channel parseChannelTag(XmlPullParser xpp)
            throws XmlPullParserException, IOException {
        Channel channel = new Channel();
        ArrayList<Item> itemList = new ArrayList<Item>();
        int event;
        String tag;
        boolean foundEndChannelTag = false;
        while (!foundEndChannelTag) {
            event = xpp.next();
            if (event == XmlPullParser.START_TAG) {
                tag = xpp.getName().trim();
                if (TITLE_TAG.equalsIgnoreCase(tag)) {
                    channel.setTitle(xpp.nextText().trim());
                } else if (LINK_TAG.equalsIgnoreCase(tag)) {
                    String link = xpp.nextText().trim();
                    if (link != null && !link.equals("")) {
                        channel.setLink(link);
                    }
                } else if (event == XmlPullParser.START_TAG) {
                    // Find the Item tag
                    if (ITEM_TAG.equalsIgnoreCase(tag)) {
                        Item item = parseItemTag(xpp);
                        itemList.add(item);
                    }
                }
            } else if (event == XmlPullParser.END_TAG) {
                tag = xpp.getName().trim();
                if (CHANNEL_TAG.equalsIgnoreCase(tag)) {
                    foundEndChannelTag = true;
                }
            }
        }
        channel.setItemList(itemList);
        return channel;
    }

    /**
     * Parses the Item tag values from the XMLPullParser
     *
     * @param xpp The parser
     * @return the parsed Item object
     * @throws XmlPullParserException
     * @throws IOException
     */
    private Item parseItemTag(XmlPullParser xpp) throws XmlPullParserException, IOException {
        Item item = new Item();
        boolean foundEndItemTag = false;
        int event;
        String tag;
        while (!foundEndItemTag) {
            event = xpp.next();
            if (event == XmlPullParser.START_TAG) {
                tag = xpp.getName().trim();
                if (TITLE_TAG.equalsIgnoreCase(tag)) {
                    item.setTitle(xpp.nextText().trim());
                } else if (DESCRIPTION_TAG.equalsIgnoreCase(tag)) {
                    item.setDescription(xpp.nextText().trim());
                } else if (GUID_TAG.equalsIgnoreCase(tag)) {
                    item.setGuid(xpp.nextText().trim());
                } else if (LINK_TAG.equalsIgnoreCase(tag)) {
                    String link = xpp.nextText().trim();
                    if (link != null && !link.equals("")) {
                        item.setLink(link);
                    }
                } else if (PUB_DATE_TAG.equalsIgnoreCase(tag)) {
                    item.setTimeOfDate(parse(xpp.nextText().trim()));
                } else if (THUMBNAIL_TAG.equals(tag)) {
                    String media_url = xpp.getAttributeValue(null, THUMBNAIL_URL_KEY);
                    item.setMediaThumbnail(media_url);
                }
            } else if (event == XmlPullParser.END_TAG) {
                tag = xpp.getName().trim();
                if (ITEM_TAG.equalsIgnoreCase(tag)) {
                    foundEndItemTag = true;
                }
            }
        }
        return item;
    }

    /**
     * Get data from the Rss channel url
     *
     * @param url The url
     * @return The url data
     *
     * @throws URISyntaxException
     * @throws ClientProtocolException
     * @throws IOException
     */
    private InputStream getUrlData(String url) throws URISyntaxException, ClientProtocolException, IOException {
        DefaultHttpClient client = new DefaultHttpClient();
        HttpGet method = new HttpGet(new URI(url));
        HttpResponse res = client.execute(method);
        return res.getEntity().getContent();
    }

    /**
     * Returns the connection state of the network.
     *
     * @return true if the network is connected, false otherwise.
     */
    private boolean isConnected() {
        ConnectivityManager cm = (ConnectivityManager) getSystemService(
                        Context.CONNECTIVITY_SERVICE);
        NetworkInfo ni = cm.getActiveNetworkInfo();
        if (ni == null) {
            return false;
        }
        return ni.isConnected();
    }

    /**
     * Gets all the Channels.
     *
     * @return a List of all Channels.
     */
    private List<Channel> getAllChannelsFromDb() {
        Cursor cursor = null;
        List<Channel> channelList = new ArrayList<Channel>();
        try {
            cursor = getContentResolver().query(RssFeedProvider.CHANNEL_CONTENT_URI, new String[]{RssFeedProvider.CHANNEL_URL_COLUMN, RssFeedProvider.CHANNEL_ID_COLUMN}, null, null, null);
            if (cursor != null && cursor.getCount() != 0 && cursor.moveToFirst()) {
                do {
                    Channel channel = new Channel();
                    String url = cursor.getString(cursor.getColumnIndex(RssFeedProvider.CHANNEL_URL_COLUMN));
                    int id = cursor.getInt(cursor.getColumnIndex(RssFeedProvider.CHANNEL_ID_COLUMN));
                    channel.setUrl(url);
                    channel.setId(id);
                    channelList.add(channel);
                } while (cursor.moveToNext());
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return channelList;
    }

    /**
     * Parses the date.
     *
     * @param date the date.
     * @return time parsed from the input String.
     */
    private long parse(String date) {
        Date parsedDate = null;
        try {
            parsedDate = dateFormat.parse(date);
        } catch (ParseException e) {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Could not parse date as a Date object.");
            }
            e.printStackTrace();
        }
        if (parsedDate == null) {
            parsedDate = new Date();
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "unknown: " + date);
            }
        }
        return parsedDate.getTime();
    }
}

