/*
 * Copyright (C) 2010 Sony Ericsson Mobile Communications AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
 *
 */

package com.sonyericsson.eventstream.rssplugin;

import com.sonyericsson.eventstream.rssplugin.db.RssFeedProvider;
import com.sonyericsson.eventstream.rssplugin.PluginConstants.Config;
import com.sonyericsson.eventstream.rssplugin.PluginConstants.EventStream;
import com.sonyericsson.eventstream.rssplugin.PluginConstants.ServiceIntentCmd;

import android.content.BroadcastReceiver;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.net.Uri;
import android.os.Bundle;
import android.util.Log;

/**
 * Listen for broadcasts from Event Stream and launches the RssPluginService.
 */
public class EventStreamListener extends BroadcastReceiver {

    @Override
    public void onReceive(Context context, Intent intent) {
        launchRssService(context, intent);
    }

    /**
     * Launches RssService with the corresponding intent
     *
     * @param context The context
     * @param intent The broadcast intent
     */
    private void launchRssService(Context context, Intent intent) {
        String action = intent.getAction();
        Intent serviceIntent = new Intent();
        serviceIntent.setComponent(new ComponentName(context, RssPluginService.class));
        if (EventStream.Intents.REFRESH_REQUEST_INTENT.equals(action)) {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Recieved refresh request from Event Stream.");
            }
            serviceIntent.putExtra(ServiceIntentCmd.SERVICE_COMMAND_KEY, ServiceIntentCmd.REFRESH_REQUEST);
            context.startService(serviceIntent);

        } else if (EventStream.Intents.REGISTER_PLUGINS_REQUEST_INTENT.equals(action)) {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Recieved plugin registration request from Event Stream.");
            }
            serviceIntent.putExtra(ServiceIntentCmd.SERVICE_COMMAND_KEY, ServiceIntentCmd.REGISTER_PLUGIN);
            context.startService(serviceIntent);

        } else if (EventStream.Intents.VIEW_EVENT_INTENT.equals(action)) {
            if (Config.DEBUG) {
                Log.d(Config.LOG_TAG, "Recieved detail view request from Event Stream.");
            }
            launchRssWebClient(context, intent);
        }
    }

    /**
     * Launches the Rss Web Client. The Event shown corresponds to the eventId from the launch intent.
     */
    private void launchRssWebClient(Context context, Intent intent) {
        Bundle extras = intent.getExtras();
        if (extras != null) {
            String eventKey = extras.getString(EventStream.IntentData.EVENT_KEY_EXTRA);
            if (eventKey != null ) {
                String url = getEventUrl(context, eventKey);
                if (url != null) {
                    Intent webViewIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));
                    webViewIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                    context.startActivity(webViewIntent);
                }
            }
        }
    }

    /**
     * Get the EventUrl.
     *
     * @param eventId The Id of the event
     * @return the url of the event.
     */
    private String getEventUrl(Context context, String eventKey) {
        Cursor cursor = null;
        String eventUrl = null;
        try {
            cursor = context.getContentResolver().query(RssFeedProvider.ITEM_CONTENT_URI,
                    new String[]{RssFeedProvider.ITEM_URL_COLUMN}, EventStream.EventColumns.ID_COLUMN + " = ?",
                    new String[] {eventKey}, null);
            if (cursor != null && cursor.getCount() != 0 && cursor.moveToFirst()) {
                eventUrl = cursor.getString(cursor.getColumnIndex(RssFeedProvider.ITEM_URL_COLUMN));
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return eventUrl;
    }
}
