
package com.sonyericsson.eventstream.facebookplugin.util;

import com.sonyericsson.eventstream.facebookplugin.util.MockObject.FunctionCall.Result;

import java.util.ArrayList;

import junit.framework.Assert;

/**
 * A mock object that can be sub-classed or used to enable mocking of components
 * for unit testing. A test can set up a set of expected function calls and can
 * verify that all calls are received and no unexpected calls are received.
 * <p>
 * Normal usage in the test is to first create a mock object and setup the
 * expected function calls and wanted return values. Then inject the mock object
 * in the class that is tested and perform the intended call(s) on that class
 * and, at last, call verifyAllExpectations() on the MockObject.
 * <p>
 * Normal usage when implementing a mocked object is to either extend this class
 * or use it directly and in each function call reportFunctionCall(). If the
 * return value of reportFunctionCall() is not null, then Result.mValue should
 * be returned by the mocked object. If the return value is null, the mocked
 * object is free to return whatever it wants (mostly a default value).
 *
 * @author 23048641
 */
public class MockObject {
    /** Used to signify that any combination of arguments can be used */
    public static final String ANY_ARGUMENTS = "__ANY_ARGUMENTS__";

    /** Used to signify that a particular argument can have any value */
    public static final String ANY_ARGUMENT = "__ANY_ARGUMENT__";

    /** Used to signify that a particular call can be called any times */
    public static final int ANY_COUNT = Integer.MIN_VALUE;

    /** The list of expected calls for this MockObject */
    final private ArrayList<FunctionCall> mExpectedCalls = new ArrayList<FunctionCall>();

    /**
     * If true, we allow unexpected calls. If false, we fail on unexpected calls
     */
    private boolean mAllowUnexpectedCalls;

    /**
     * Small class to encapsulate an expected function call.
     *
     * @author 23048641
     */
    public class FunctionCall {

        /**
         * Class that represents a result from a function call. The reason for
         * this class is to be able to differentiate between a null return value
         * (Result object not null, value null) and a not specified return value
         * (Result object null) value.
         *
         * @author 23048641
         */
        public class Result {

            /** The value that is returned */
            public Object mValue;

            /**
             * Creates a Result object and sets mValue to value.
             *
             * @param value The value that should be returned
             */
            private Result(Object value) {
                mValue = value;
            }
        }

        /** The name of the function being called */
        private String mFunctionName;

        /** The number or times it has been called */
        private int mActualHitCount;

        /** The number of times it's expected to be called */
        private int mExpectedHitCount;

        /** The arguments that are expected */
        private String mExpectedArguments;

        /** The value to return */
        private Result mReturnValue;

        /** Creates a FunctionCall */
        private FunctionCall(String functionName, int expectedHitCount, String expectedArguments) {
            mFunctionName = functionName;
            mExpectedHitCount = expectedHitCount;
            mExpectedArguments = expectedArguments;
            mActualHitCount = 0;
            mReturnValue = null;
        }

        /**
         * Set an object to return when the function is called
         *
         * @param returnValue
         */
        public void andReturn(Object returnValue) {
            mReturnValue = new Result(returnValue);
        }
    }

    /**
     * Add an expectation to the mock object without specifying any count or
     * arguments
     *
     * @param functionName The name of the expected function
     * @return The newly added expected function call
     */
    final public FunctionCall expect(String functionName) {
        return expect(functionName, ANY_COUNT, ANY_ARGUMENTS);
    }

    /**
     * Add an expectation to the mock object without specifying any arguments
     *
     * @param functionName The name of the expected function
     * @param expectedHitCount The number of times it's expected to be called.
     *            Negative values means at least that amount of calls (-5 ->
     *            expected to be called at least 5 times).
     * @return The newly added expected function call
     */
    final public FunctionCall expect(String functionName, int expectedHitCount) {
        return expect(functionName, expectedHitCount, ANY_ARGUMENTS);
    }

    /**
     * Add an expectation to the mock object
     *
     * @param functionName The name of the expected function
     * @param expectedHitCount The number of times it's expected to be called.
     *            Negative values means at least that amount of calls (-5 ->
     *            expected to be called at least 5 times).
     * @param expectedArguments The expected arguments it's called with
     * @return The newly added expected function call
     */
    final public FunctionCall expect(String functionName, int expectedHitCount,
            String expectedArguments) {
        FunctionCall newFunctionCall = new FunctionCall(functionName, expectedHitCount,
                expectedArguments);

        // Replace?
        for (FunctionCall functionCall : mExpectedCalls) {
            if (functionCall.mFunctionName.equals(newFunctionCall.mFunctionName)) {
                if (expectedArguments.equals(functionCall.mExpectedArguments) &&
                    functionCall.mExpectedHitCount == ANY_COUNT) {
                    mExpectedCalls.remove(functionCall);

                    break;
                }
            }
        }

        mExpectedCalls.add(newFunctionCall);
        return newFunctionCall;
    }

    /**
     * Can be used to make the mock object allow (not fail) if it gets an
     * unexpected call.
     *
     * @param allow If true, unexpected calls are allowed. If false, unexpected
     *            calls will fail the test.
     */
    final public void allowUnexpectedCalls(boolean allow) {
        mAllowUnexpectedCalls = allow;
    }

    /**
     * Used to verify that all expectations has been fulfilled at the end of the
     * test. If not an assert will be triggered.
     */
    final public void verifyAllExpectaions() {
        for (FunctionCall functionCall : mExpectedCalls) {
            if (functionCall.mExpectedHitCount >= 0) {
                // a specific amount of calls has been specified
                Assert.assertEquals("Function " + functionCall.mFunctionName + "("
                        + functionCall.mExpectedArguments + ") not called the expected times,",
                        functionCall.mExpectedHitCount, functionCall.mActualHitCount);
            } else if (functionCall.mExpectedHitCount != ANY_COUNT) {
                // a minimum amount of calls has been specified
                Assert.assertTrue("Function " + functionCall.mFunctionName + "("
                        + functionCall.mExpectedArguments + ") not called the expected at least "
                        + -functionCall.mExpectedHitCount + " time(s), was only called "
                        + functionCall.mActualHitCount + " times",
                        -functionCall.mExpectedHitCount <= functionCall.mActualHitCount);
            }
        }
    }

    /**
     * Remove all expectations from the mock object
     */
    final public void clearExpectations() {
        mExpectedCalls.clear();
    }

    /**
     * Used by sub-classes to report that they have gotten a call to a function.
     *
     * @param function The name of the function that has been called
     * @param arguments The arguments that it was called with. Arguments should
     *            be separated by a ';' and passed through String.valueOf().
     * @return If not null, the Result object that contains the value that
     *         should be returned.
     */
    final public Result reportFunctionCall(String function, String arguments) {
        String assertMessage = null;
        for (FunctionCall functionCall : mExpectedCalls) {
            if (functionCall.mFunctionName.equals(function)
                    && verifyArguments(functionCall.mExpectedArguments, arguments)) {
                // increase hit count
                functionCall.mActualHitCount++;

                // verify hit count
                if (functionCall.mExpectedHitCount >= 0) {
                    if (functionCall.mActualHitCount > functionCall.mExpectedHitCount) {
                        // we have more hits on this call than expected.
                        // However, there might be another function call that
                        // might catch this call. Save the assert message for
                        // later and decrease the hit count (so we don't fail if
                        // we do find another function call), then continue
                        // looking for another matching function call.
                        assertMessage = "One to many calls to " + function + "(" + arguments
                                + "), expected " + functionCall.mExpectedHitCount
                                + " hits, this is hit " + functionCall.mActualHitCount;
                        functionCall.mActualHitCount--;
                        continue;
                    }
                }
                // Normal hit. If any assertMessage was set, set it to null.
                assertMessage = null;
                return functionCall.mReturnValue;
            }
        }

        if (assertMessage != null) {
            // We got to many hits on a function call. Fail the test.
            Assert.fail(assertMessage);
        }

        // If we get here, the function call was unexpected

        if (!mAllowUnexpectedCalls) {
            Assert.fail("Unexpected call to " + function + " with arguments: " + arguments);
        }
        return null;
    }

    /**
     * Checks if the arguments matches the expected arguments.
     *
     * @param expected The expected arguments
     * @param actual The actual arguments
     * @return True if it matches, false otherwise
     */
    final private boolean verifyArguments(String expected, String actual) {
        if (expected.equals(ANY_ARGUMENTS)) {
            return true;
        }
        String[] expectedArguments = expected.split(";");
        String[] actualArguments = actual.split(";");
        if (expectedArguments.length != actualArguments.length) {
            return false;
        }
        for (int i = 0; i < expectedArguments.length; i++) {
            if (!expectedArguments[i].equals(ANY_ARGUMENT)
                    && !expectedArguments[i].equals(actualArguments[i])) {
                return false;
            }
        }

        return true;
    }
}
