/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

/**
 * @file
 * @author Niklas Karlsson (niklas.karlsson@sonyericsson.com)
 */
package com.sonyericsson.eventstream.facebookplugin.util;

import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants;
import com.sonyericsson.eventstream.facebookplugin.FacebookPluginConfig;
import com.sonyericsson.eventstream.facebookplugin.R;
import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.EventTable;
import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.SourceTable;

import static com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.SOURCE_PROVIDER_URI;

import android.content.ComponentName;
import android.content.ContentResolver;
import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.net.Uri;

import java.util.ArrayList;
import java.util.List;

public class DatabaseHelper {
    public static void removePluginFromEventStream(Context context) {
        context.getContentResolver().delete(
                Uri.parse("content://com.sonyericsson.eventstream/events"), null, null);
        context.getContentResolver().delete(
                Uri.parse("content://com.sonyericsson.eventstream/friends"), null, null);
        context.getContentResolver().delete(
                Uri.parse("content://com.sonyericsson.eventstream/sources"), null, null);
        context.getContentResolver().delete(
                Uri.parse("content://com.sonyericsson.eventstream/plugins"), null, null);
    }

    public static int getFriendCountFromEventStream(Context context) {
        int result = 0;
        Cursor cursor = null;

        try {
            cursor = context.getContentResolver().query(Uri.parse("content://com.sonyericsson.eventstream/friends"),
                    null, null, null, null);

            result = cursor.getCount();
        } catch (Exception exception) {

        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }

        return result;
    }

    public static int addFriend(Context context, long sourceId, String displayName,
            String imageUri, String friendKey, String rawContactUri) {
        ContentValues values = new ContentValues();

        values.put("source_id", sourceId);
        if (displayName != null) {
            values.put("display_name", displayName);
        } else {
            values.putNull("display_name");
        }

        if (imageUri != null) {
            values.put("profile_image_uri", imageUri);
        } else {
            values.putNull("profile_image_uri");
        }

        if (friendKey != null) {
            values.put("friend_key", friendKey);
        } else {
            values.putNull("friend_key");
        }

        if (rawContactUri != null) {
            values.put("contacts_reference", rawContactUri);
        } else {
            values.putNull("contacts_reference");
        }
        Uri uri = context.getContentResolver().insert(EventStreamConstants.FRIEND_PROVIDER_URI, values);

        if (uri != null) {
            return (int) ContentUris.parseId(uri);
        } else {
            return -1;
        }
    }

    public static int addEvent(Context context, long sourceId, String eventId, String message,
            String picture, String friendId, long timestamp) {
        ContentValues eventValues = new ContentValues();

        eventValues.put(EventTable.SOURCE_ID, sourceId);
        eventValues.put(EventTable.EVENT_KEY, eventId);
        if (message != null) {
            eventValues.put(EventTable.MESSAGE, message);
        }
        if (picture != null) {
            eventValues.put(EventTable.IMAGE_URI, picture);
        }
        eventValues.put(EventTable.PUBLISHED_TIME, timestamp);
        if (friendId != null) {
            eventValues.put(EventTable.FRIEND_KEY, friendId);
        }
        eventValues.put(EventTable.OUTGOING, 0);
        eventValues.put(EventTable.PERSONAL, 0);
        Uri uri = context.getContentResolver().insert(Uri.parse("content://com.sonyericsson.eventstream/events"), eventValues);
        return (int) ContentUris.parseId(uri);
    }

    /**
     * How many events are stored in the event stream framework
     *
     * @return the number of events stored
     */
    public static int getEventCount(Context context) {
        int result = 0;
        Cursor cursor = context.getContentResolver().query(Uri.parse("content://com.sonyericsson.eventstream/events"),
                null, null, null, null);

        result = cursor.getCount();
        cursor.close();

        return result;
    }

    public static int registerPlugin(Context context) {
        int result;
        ContentValues values = new ContentValues();
        ComponentName componentName = new ComponentName(context, FacebookPluginConfig.class);

        values.put("name", context.getResources().getString(R.string.ts_facebook_service_name));
        values.put("icon_uri",
                ContentResolver.SCHEME_ANDROID_RESOURCE + "://" + context.getPackageName() + "/" + R.drawable.facebook_source_icn);
        values.put("api_version", 1);
        values.put("config_activity", componentName.flattenToString());
        values.put("status_support", 1);
        values.put("config_state", 1);
        context.getContentResolver().insert(Uri.parse("content://com.sonyericsson.eventstream/plugins"), values);

        ContentValues sourceValues = new ContentValues();
        sourceValues.put("name", context.getResources().getString(R.string.ts_facebook_service_name));
        sourceValues.put("icon_uri",
                ContentResolver.SCHEME_ANDROID_RESOURCE + "://" + context.getPackageName() + "/" + R.drawable.facebook_source_icn);
        sourceValues.put("enabled", 1);
        Uri uri = context.getContentResolver().insert(Uri.parse("content://com.sonyericsson.eventstream/sources"), sourceValues);
        result = (int) ContentUris.parseId(uri);

        return result;
    }

    public static boolean isRegistered(Context context) {
        boolean result = false;
        Uri uri = Uri.parse("content://com.sonyericsson.eventstream/plugins");
        Cursor cursor = context.getContentResolver().query(uri, null, null, null, null);

        try {
            if (cursor != null && cursor.getCount() == 1 && cursor.moveToNext()) {
                String name = cursor.getString(cursor.getColumnIndexOrThrow("name"));

                result = name != null && context.getResources().getString(R.string.ts_facebook_service_name).equals(name);
            }
        } catch (Exception exception) {
            // Deliberate empty
        } finally {
            cursor.close();
            cursor = null;
        }

        if (result) {
            try {
                uri = Uri.parse("content://com.sonyericsson.eventstream/sources");
                cursor = context.getContentResolver().query(uri, null, null, null, null);
                cursor.moveToNext();

                String name = cursor.getString(cursor.getColumnIndexOrThrow("name"));

                result = cursor.getCount() == 1 && cursor.getString(cursor.getColumnIndexOrThrow("name")).equals(name);
            } catch (Exception ex) {
                result = false;
            } finally {
                cursor.close();
            }
        }

        return result;
    }

    public static String getStatusUpdate(Context context) {
        String result = "";
        Cursor cursor = null;
        Uri uri = Uri.parse("content://com.sonyericsson.eventstream/sources");

        try {
            cursor = context.getContentResolver().query(uri, null, null, null, null);
            if (cursor != null && cursor.getCount() == 1 && cursor.moveToNext()) {
                result = cursor.getString(cursor.getColumnIndex("current_status"));
            }
        } catch (Exception exception) {
            // Do nothing!
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }

        return result;
    }

    public static boolean setOwnStatus(Context context, String ownStatus) {
        Uri uri = Uri.parse("content://com.sonyericsson.eventstream/sources");
        ContentValues values = new ContentValues();
        boolean result = false;

        try {
            values.put("current_status", ownStatus);
            values.put("status_timestamp", System.currentTimeMillis());

            result = (1 == context.getContentResolver().update(uri, values, null, null));
        } catch (Exception exception) {
            // Do nada...
        }
        return result;
    }

    public static String getDisplayName(Context context) {
        String result = "";
        Cursor cursor = null;
        Uri uri = Uri.parse("content://com.sonyericsson.eventstream/plugins");

        try {
            cursor = context.getContentResolver().query(uri, null, null, null, null);
            if (cursor != null && cursor.getCount() == 1 && cursor.moveToNext()) {
                result = cursor.getString(cursor.getColumnIndex("config_text"));
            }
        } catch (Exception exception) {
            // Do nothing!
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }

        return result;
    }

    public static String getOwnStatus(Context context) {
        String status = null;
        Cursor c = null;
        try {
            c = context.getContentResolver().query(SOURCE_PROVIDER_URI, new String[] {
                SourceTable.CURRENT_STATUS
            }, null, null, null);
            if (c != null && c.moveToFirst()) {
                status = c.getString(c.getColumnIndexOrThrow(SourceTable.CURRENT_STATUS));
            }
        } finally {
            if (c != null) {
                c.close();
            }
        }
        return status;
    }

    public static int getConfigurationState(Context context) {
        int result = -1;
        Cursor cursor = null;
        Uri uri = Uri.parse("content://com.sonyericsson.eventstream/plugins");

        try {
            cursor = context.getContentResolver().query(uri, null, null, null, null);
            if (cursor != null && cursor.getCount() == 1 && cursor.moveToNext()) {
                result = cursor.getInt(cursor.getColumnIndex("config_state"));
            }
        } catch (Exception exception) {
            // Do nothing!
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return result;
    }

    public static List<Object> readPluginData(Context context) {
        List<Object> data = new ArrayList<Object>();
        Cursor cursor = null;
        Uri uri = Uri.parse("content://com.sonyericsson.eventstream/plugins");

        try {
            cursor = context.getContentResolver().query(uri, null, null, null, null);
            if (cursor != null && cursor.getCount() == 1 && cursor.moveToNext()) {
                data.add(cursor.getString(cursor.getColumnIndex("name")));
                data.add(cursor.getString(cursor.getColumnIndex("icon_uri")));
                data.add(cursor.getInt(cursor.getColumnIndex("api_version")));
                data.add(cursor.getString(cursor.getColumnIndex("config_activity")));
                data.add(cursor.getInt(cursor.getColumnIndex("status_support")));
                data.add(cursor.getInt(cursor.getColumnIndex("config_state")));
            }
        } catch (Exception exception) {
            data = null;
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }

        return data;
    }

    public static List<Object> readSourceData(Context context) {
        List<Object> data = new ArrayList<Object>();
        Cursor cursor = null;
        Uri uri = Uri.parse("content://com.sonyericsson.eventstream/sources");

        try {
            cursor = context.getContentResolver().query(uri, null, null, null, null);
            if (cursor != null && cursor.getCount() == 1 && cursor.moveToNext()) {
                data.add(cursor.getString(cursor.getColumnIndex("name")));
                data.add(cursor.getString(cursor.getColumnIndex("icon_uri")));
                data.add(cursor.getString(cursor.getColumnIndex("enabled")));
                data.add(cursor.getString(cursor.getColumnIndex("current_status")));
                data.add(cursor.getString(cursor.getColumnIndex("status_timestamp")));
            }
        } catch (Exception exception) {
            data = null;
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }

        return data;
    }

    public static Long getRawContactId(Context context, String friendId) {
        Long rawContactId = null;
        String rawContactUri = "";
        Cursor cursor = null;
        Uri uri = Uri.parse("content://com.sonyericsson.eventstream/friends");

        try {
            ContentResolver cr = context.getContentResolver();

            cursor = cr.query(uri, null, "friend_key = '" + friendId + "'", null, null);
            if (cursor != null && cursor.moveToNext()) {
                rawContactUri = cursor.getString(cursor.getColumnIndex("contacts_reference"));
                String id = Uri.parse(rawContactUri).getLastPathSegment();
                rawContactId = Long.valueOf(id);
            }
        } catch (Exception exception) {

        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }

        return rawContactId;
    }
}
