/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.facebookplugin;

import static com.sonyericsson.eventstream.facebookplugin.Constants.AUTHENTICATE_INTENT;
import static com.sonyericsson.eventstream.facebookplugin.Constants.LAUNCH_BROWSER_INTENT;
import static com.sonyericsson.eventstream.facebookplugin.Constants.LOGOUT_INTENT;
import static com.sonyericsson.eventstream.facebookplugin.Constants.LOG_TAG;
import static com.sonyericsson.eventstream.facebookplugin.Constants.PASSWORD_EXTRA;
import static com.sonyericsson.eventstream.facebookplugin.Constants.PLUGIN_KEY;
import static com.sonyericsson.eventstream.facebookplugin.Constants.PLUGIN_KEY_PARAMETER;
import static com.sonyericsson.eventstream.facebookplugin.Constants.REFRESH_REQUEST_INTENT;
import static com.sonyericsson.eventstream.facebookplugin.Constants.REGISTER_PLUGIN_INTENT;
import static com.sonyericsson.eventstream.facebookplugin.Constants.LOCALE_CHANGED_INTENT;
import static com.sonyericsson.eventstream.facebookplugin.Constants.SEND_STATUS_UPDATE_INTENT;
import static com.sonyericsson.eventstream.facebookplugin.Constants.USERNAME_EXTRA;
import static com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.PLUGIN_PROVIDER_URI;

import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.Config;
import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.ConfigState;
import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.EventstreamIntentData;
import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.PluginTable;

import com.sonyericsson.eventstream.facebookplugin.Facebook.FacebookException;
import com.sonyericsson.eventstream.facebookplugin.Facebook.EventStorage.EventType;
import com.sonyericsson.eventstream.facebookplugin.FacebookPluginApplication.State;

import android.app.IntentService;

import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.database.SQLException;
import android.net.Uri;
import android.util.Log;
import android.view.Gravity;
import android.widget.Toast;

/**
 * @author Erik Hellman <erik.hellman@sonyericsson.com>
 */
public class FacebookService extends IntentService {

    public FacebookService() {
        super("FacebookService");
    }

    @Override
    public void onCreate() {
        super.onCreate();
    }

    @Override
    public void onHandleIntent(Intent intent) {
        if (intent == null) {
            return;
        }
        // At this point, all intent should have the plug-in parameter
        if (! intent.hasExtra(PLUGIN_KEY_PARAMETER) ||
                ! PLUGIN_KEY.equals(intent.getStringExtra(PLUGIN_KEY_PARAMETER))) {
            if (Config.DEBUG) {
                Log.d(LOG_TAG, "Premature return:" + intent.getStringExtra(PLUGIN_KEY_PARAMETER));
            }
            return;
        }

        String action = intent.getAction();
        if (Config.DEBUG) {
            Log.d(LOG_TAG, "FacebookService started with " + action);
        }

        if (REGISTER_PLUGIN_INTENT.equals(action)) {
            register();
        } else if (REFRESH_REQUEST_INTENT.equals(action)) {
            refresh();
        } else if (LOGOUT_INTENT.equals(action)) {
            logout();
        } else if (AUTHENTICATE_INTENT.equals(action)) {
            String username = intent.getStringExtra(USERNAME_EXTRA);
            String password = intent.getStringExtra(PASSWORD_EXTRA);

            // Authenticate...
            boolean success = authenticate(username, password);

            if (success) {
                refresh();
            }
        } else if (SEND_STATUS_UPDATE_INTENT.equals(action)) {
            String message = intent.getStringExtra(EventstreamIntentData.EXTRA_STATUS_UPDATE_MESSAGE);

            updateStatus(message, System.currentTimeMillis());
        } else if (LAUNCH_BROWSER_INTENT.equals(action)) {
            String eventKey = intent.getStringExtra(EventStreamConstants.EventstreamIntentData.EXTRA_EVENT_KEY);
            String friendKey = intent.getStringExtra(EventStreamConstants.EventstreamIntentData.EXTRA_FRIEND_KEY);

            launchBrowser(getApplicationContext(), eventKey, friendKey);
        }  else if (LOCALE_CHANGED_INTENT.equals(action)){
            handleLocaleChanged();
        }
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
    }

    @Override
    public void onLowMemory() {
        super.onLowMemory();
    }

    private void refresh() {
        refreshFacebookFriends();
        connectFriendsToContacts();
        retrieveLatestPosts();
    }

    private void register() {
        Database database = new Database(getApplicationContext());

        if (! database.isRegistered()) {
            Facebook facebook = FacebookFactory.getFacebook(getApplicationContext());

            if (facebook.isLoggedIn()) {
                logout();
            }
            database.registerPlugin();
        } else {
            database.registerPlugin();
        }
    }

    protected void launchBrowser(Context context, String databaseEventId, String friendKey) {
        try {
            String facebookEventId = Database.getEventId(databaseEventId);
            EventType eventType = Database.getEventType(databaseEventId);
            String myFacebookId = new Settings(context).getOwnId();

            // In case we don't have a friend key we have to try to calculate it...
            if (friendKey == null || friendKey.equals("")) {
                friendKey = Database.extractFriendKeyFromEventId(facebookEventId);
            }
            Uri uri = FacebookDeepLink.getDeepLink(context, myFacebookId, facebookEventId, friendKey, eventType);
            if (uri != null) {
                Intent browserIntent = new Intent(Intent.ACTION_VIEW, uri);

                browserIntent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                // Use the android browser, nothing else; don't want deep linking to expose vital data
                browserIntent.setPackage("com.android.browser");
                context.startActivity(browserIntent);
            }
        } catch (Exception exception) {
            if (Config.DEBUG) {
                Log.e(LOG_TAG, "Error when launching view for Facebook - notify the user with a Toast.");
            }
            Toast toast = Toast.makeText(context, context.getResources().
                    getString(R.string.ts_facebook_launch_error_message), Toast.LENGTH_SHORT);
            toast.setGravity(Gravity.CENTER, 0, 0);
            toast.show();
        }
    }

    private void logout() {
        // Delete all facebook data in Event Stream, remove settings and terminate facebook objects...
        try {
            ((FacebookPluginApplication) getApplication()).clearNotification();

            // Clear "local"/internal data
            new Settings(getApplicationContext()).removeSettings();

            // Clear data from event stream framework
            Database database = new Database(getApplicationContext());

            database.clearData();

            // Re-initialize facebook object...
            FacebookFactory.terminate(true);
        } catch (Exception e) {
            if (Config.DEBUG) {
                Log.e(LOG_TAG, "Error when deleting plugin information. Is Event Stream installed?");
            }
        } finally {
            ((FacebookPluginApplication) getApplication()).setState(FacebookPluginApplication.State.NOT_CONFIGURED);
        }
    }

    private boolean authenticate(final String email, final String password) {
        FacebookNotification notification = null;
        String displayName = null;
        Database database = new Database(getApplicationContext());
        Facebook facebook = FacebookFactory.getFacebook(getApplicationContext());

        // State that we are about to authenticate towards the facebook server...
        ((FacebookPluginApplication) getApplication()).setState(FacebookPluginApplication.State.AUTHENTICATION_IN_PROGRESS);

        try {
            displayName = facebook.authenticate(email, password);
        } catch (FacebookException exception) {
            switch (exception.cause) {
                case FacebookException.AUTHENTICATION_FAILED:
                    notification = new FacebookNotification(State.AUTHENTICATION_FAILED,
                            getResources().getString(R.string.ts_facebook_connection_failed));
                    break;
                case FacebookException.AUTHENTICATION_INVALID_CREDENTIALS:
                    notification = new FacebookNotification(State.AUTHENTICATION_BAD_CREDENTIALS,
                            getResources().getString(R.string.ts_facebook_incorrect_account_info));
                    break;
            }
        } finally {
            if (facebook.isLoggedIn()) {
                // Store display name...
                if (displayName != null) {
                    final String databaseName = getString(R.string.ts_facebook_logout_label) + " " + displayName;
                    Settings settings = new Settings(getApplicationContext());
                    settings.setDisplayName(displayName);
                    database.setConfigurationText(databaseName);
                }
                // State that we are configured
                database.setConfigurationState(ConfigState.CONFIGURED);
            }

            if (notification == null) {
                new Settings(getApplicationContext()).setHasAcceptedDisclaimer(true);
                notification = new FacebookNotification(State.AUTHENTICATED, null);
            }
            ((FacebookPluginApplication) getApplication()).setState(notification);
        }

        return facebook.isLoggedIn();
    }

    private void updateStatus(String message, long timestamp) {
        Facebook facebook = FacebookFactory.getFacebook(getApplicationContext());
        boolean success = false;

        try {
            success = facebook.updateStatus(message);
        } catch (FacebookException exception) {
            if (Config.DEBUG) {
                Log.e(LOG_TAG, "Failed to update status:" + exception.toString());
            }
            ((FacebookPluginApplication) getApplication()).fireNotification(State.STATUS_UPDATE_FAILED);
        }
        if (success) {
            ((FacebookPluginApplication) getApplication()).cancelNotifications();
            Database database = new Database(getApplicationContext());

            database.setOwnStatus(message, timestamp);
        }
    }

    private void retrieveLatestPosts() {
        Database database = new Database(getApplicationContext());
        Facebook facebook = FacebookFactory.getFacebook(getApplicationContext());

        try {
            if (Config.DEBUG) {
                Log.d(LOG_TAG, "Retrieve latest Facebook statuses.");
            }

            boolean result = facebook.retrieveLatestPosts(database, database);
            if (result) {
                ((FacebookPluginApplication) getApplication()).cancelNotifications();
            }
        } catch (FacebookException exception) {
            if (Config.DEBUG) {
                Log.e(LOG_TAG, "Error retrieving last posts:." + exception.toString());
                exception.printStackTrace();
            }
            if (exception.cause == FacebookException.AUTHENTICATION_FAILED) {
                ((FacebookPluginApplication) getApplication()).setState(State.AUTHENTICATION_FAILED);
                logout();
            }
        } finally {
            database.storeEvents();
        }
    }

    /**
     * Refresh our local friend list, handle added and removed friends
     */
    private void refreshFacebookFriends() {
        Database database = new Database(getApplicationContext());
        Facebook facebook = FacebookFactory.getFacebook(getApplicationContext());

        try {
            if (Config.DEBUG) {
                Log.d(LOG_TAG, "Register friends.");
            }

            facebook.refreshFacebookFriends(database);
        } catch (FacebookException exeption) {
            if (Config.DEBUG) {
                Log.e(LOG_TAG, "Register friends failed:" + exeption.toString());
            }
        } finally {
            database.storeFriends();
        }
    }

    /**
     * For each of our friends, try to make an automatic match towards the
     * contacts provider.
     */
    private void connectFriendsToContacts() {
        Database database = new Database(getApplicationContext());
        database.autoLinkFriends();
    }
    /*
     * update facebook_register_text according to locale change
     */
    private void handleLocaleChanged() {
        int result;
        String registerText = getResources().getString(R.string.facebook_register_txt);
        String nameText = getResources().getString(R.string.ts_facebook_service_name);

        putStringValueInColumn(PLUGIN_PROVIDER_URI, PluginTable.CONFIGURATION_TEXT,registerText);
        putStringValueInColumn(PLUGIN_PROVIDER_URI, PluginTable.NAME,nameText);
    }

    private void putStringValueInColumn (Uri tableUri, String columnName, String value) {
        ContentValues values = new ContentValues();
        int result;

        values.put(columnName, value);

        try {
            result = getContentResolver().update(
                    tableUri, values, null, null);
            if (result != 1) {
                Log.e(LOG_TAG, "Failed to update column: " + columnName + "in: " + tableUri + "result= " + result);
            }
        } catch (SQLException exception) {
            Log.w(LOG_TAG, "Failed to update column: " + columnName);
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(LOG_TAG, "Failed to update column: " + columnName);
            throw new RuntimeException(exception);
        }
    }
}
