/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.facebookplugin;

import static com.sonyericsson.eventstream.facebookplugin.Constants.LOG_TAG;

import com.sonyericsson.eventstream.facebookplugin.Facebook.EventStorage.EventType;

import android.content.Context;
import android.net.Uri;
import android.util.Config;
import android.util.Log;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.regex.PatternSyntaxException;

public class FacebookDeepLink {
    /** the dividing char of message ID. */
    private static final String MESSAGE_ID_DIVIDING_STRING = "-";

    /** the dividing char of status update messages. */
    private static final String STATUS_ID_DIVIDING_STRING = "_";

    private static final String LINK_ID_DIVIDING_STRING = "#";

    /** Facebook message url */
    private static final String FACEBOOK_MESSAGE_URL =
        "http://touch.facebook.com/#/message.php?id=%s&f=0";

    /** Facebook friends photo url */
    private static final String FACEBOOK_FRIENDS_PHOTO_URL =
        "http://touch.facebook.com/#profile.php?id=%s&v=photos";

    /** Facebook status update url */
    private static final String FACEBOOK_STATUS_UPDATE_URL =
        "http://touch.facebook.com/#/story.php?id=%s&v=feed&story_fbid=%s";

    private static String getFacebookLinkUri(String facebookId) {
        String uriStr = null;
        if (facebookId != null) {
            String tokens[] = null;
            tokens = facebookId.split(LINK_ID_DIVIDING_STRING);
            try {
                uriStr = URLDecoder.decode(tokens[1], "UTF-8");
            } catch (UnsupportedEncodingException e) {
                if (Config.DEBUG) {
                    Log.e(LOG_TAG, e.getMessage());
                    e.printStackTrace();
                }
            }
        }
        return uriStr;
    }

    private static String getFacebookMessageUri(String facebookMessageId) {
        String uriStr = null;

        if (facebookMessageId != null) {
            String[] idArray = null;

            try {
                idArray = facebookMessageId.split(MESSAGE_ID_DIVIDING_STRING);
            } catch (PatternSyntaxException e) {
                // failed to extract messageId, set it to null.
                idArray = null;
            }

            // Create the uri for the direct message
            if (idArray != null && idArray.length == 2) {
                uriStr = FACEBOOK_MESSAGE_URL;

                uriStr = String.format(uriStr, idArray[0]);
            } else {
                if (Config.DEBUG) {
                    Log.d(LOG_TAG, "Illegal format on message id (" + facebookMessageId
                            + ") should be on the format xxxxxxx-xx, returning null");
                }
            }
        }

        return uriStr;
    }

    /**
     * The format is [FRIEND_ID]_[MESSAGE_ID]
     */
    private static String getFacebookStatusUpdateUri(String facebookId, String friendId) {
        String uriStr = null;

        if (facebookId == null || friendId == null) {
            return null;
        }
        String[] idArray = null;

        try {
            idArray = facebookId.split(STATUS_ID_DIVIDING_STRING);
        } catch (PatternSyntaxException e) {
            // failed to extract messageId, set it to null.
            idArray = null;
        }

        // Create the uri for the status update
        if (idArray != null && idArray.length == 2) {
            uriStr = String.format(FACEBOOK_STATUS_UPDATE_URL, friendId, idArray[1]);
        }
        return uriStr;
    }

    private static String getFacebookPhotoUri(String friendId) {
        String uriString = null;

        if (friendId == null) {
            return null;
        }

        uriString = String.format(FACEBOOK_FRIENDS_PHOTO_URL, friendId);

        return uriString;
    }

    private static Uri generateDeepLink(Context context, String myFacebookId, String uriString) {
        FacebookSecurity security = FacebookFactory.getFacebookSecurity();
        Uri uri = security.generateDeepLink(context, myFacebookId, uriString);

        return uri;
    }

    public static Uri getDeepLink(Context context, String myFacebookId, String facebookEventId,
            String facebookFriendId, EventType eventType) {
        String uriString = null;
        Uri deepLinkedUri = null;

        if (Config.DEBUG) {
            Log.d(LOG_TAG, "my facebook id:" + myFacebookId);
            Log.d(LOG_TAG, "event id:" + facebookEventId);
            Log.d(LOG_TAG, "friend id:" + facebookFriendId);
        }

        switch (eventType) {
            case LINK_EVENT:
                uriString = getFacebookLinkUri(facebookEventId);
                break;
            case MESSAGE_EVENT:
                // Direct message
                uriString = getFacebookMessageUri(facebookEventId);
                break;
            case STATUS_EVENT:
                // Status update
                uriString = getFacebookStatusUpdateUri(facebookEventId, facebookFriendId);
                break;
            case PHOTO_EVENT:
                // Photo
                uriString = getFacebookPhotoUri(facebookFriendId);
                break;
            default:
            break;
        }

        if (uriString != null) {
            if (eventType == EventType.LINK_EVENT) {
                deepLinkedUri = Uri.parse(uriString);
            } else {
                deepLinkedUri = generateDeepLink(context, myFacebookId, uriString);
            }
        }

        return deepLinkedUri;
    }
}