
/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.calllogplugin;

import com.sonyericsson.eventstream.calllogplugin.PluginConstants;
import com.sonyericsson.eventstream.calllogplugin.R;
import com.sonyericsson.eventstream.calllogplugin.utilities.CallLogPluginTestCase;

import android.content.ComponentName;
import android.content.ContentUris;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.provider.CallLog;
import android.provider.CallLog.Calls;
import android.test.mock.MockContext;

public class FEventTests extends CallLogPluginTestCase {

    @Override
    protected void setUp() throws Exception {
        super.setUp();
        registerPlugin();
        sleep(2);
    }

    public void testAddMissedCall() {
        int numEvents = getNumberOfEvents();
        // 1. Add missed call
        String phoneNumber = "+46123123";
        int callId = addCall(Calls.MISSED_TYPE, phoneNumber, null);
        // 2. Check that it is added to event table
        sleep(3);
        assertEquals(numEvents + 1, getNumberOfEvents());
        Event e = getLastEvent();
        assertEquals(phoneNumber, e.message);
        assertEquals(Integer.toString(callId), e.eventKey);
    }

    public void testAddMissedCallUnknownNumber() {
        int numEvents = getNumberOfEvents();
        // 1. Add missed call
        String phoneNumber = PluginConstants.CallerInfo.UNKNOWN_NUMBER;
        int callId = addCall(Calls.MISSED_TYPE, phoneNumber, null);
        // 2. Check that it is added to event table
        sleep(3);
        assertEquals(numEvents + 1, getNumberOfEvents());
        Event e = getLastEvent();
        // no message when unknown number
        assertNull(e.message);
        // title used instead of friend name when unknown number
        assertEquals(getCallerInfoString(phoneNumber), e.title);
        assertEquals(Integer.toString(callId), e.eventKey);
    }

    public void testAddMissedCallPrivateNumber() {
        int numEvents = getNumberOfEvents();
        // 1. Add missed call
        String phoneNumber = PluginConstants.CallerInfo.PRIVATE_NUMBER;
        int callId = addCall(Calls.MISSED_TYPE, phoneNumber, null);
        // 2. Check that it is added to event table
        sleep(3);
        assertEquals(numEvents + 1, getNumberOfEvents());
        Event e = getLastEvent();
        // no message when private number
        assertNull(e.message);
        // title used instead of friend name when private number
        assertEquals(getCallerInfoString(phoneNumber), e.title);
        assertEquals(Integer.toString(callId), e.eventKey);
    }

    public void testAddMissedCallPayphoneNumber() {
        int numEvents = getNumberOfEvents();
        // 1. Add missed call
        String phoneNumber = PluginConstants.CallerInfo.PAYPHONE_NUMBER;
        int callId = addCall(Calls.MISSED_TYPE, phoneNumber, null);
        // 2. Check that it is added to event table
        sleep(3);
        assertEquals(numEvents + 1, getNumberOfEvents());
        Event e = getLastEvent();
        // no message when payphone number
        assertNull(e.message);
        // title used instead of friend name when payphone number
        assertEquals(getCallerInfoString(phoneNumber), e.title);
        assertEquals(Integer.toString(callId), e.eventKey);
    }

    public void testAddMissedCallEmptyNumber() {
        int numEvents = getNumberOfEvents();
        // 1. Add missed call
        String phoneNumber = "";
        int callId = addCall(Calls.MISSED_TYPE, phoneNumber, null);
        // 2. Check that it is added to event table
        sleep(3);
        assertEquals(numEvents + 1, getNumberOfEvents());
        Event e = getLastEvent();
        // no message when unknown number
        assertNull(e.message);
        // title used instead of friend name when payphone number
        assertEquals(getCallerInfoString(phoneNumber), e.title);
        assertEquals(Integer.toString(callId), e.eventKey);
    }

    public void testAddMissedCallNullNumber() {
        int numEvents = getNumberOfEvents();
        // 1. Add missed call
        String phoneNumber = null;
        int callId = addCall(Calls.MISSED_TYPE, phoneNumber, null);
        // 2. Check that it is added to event table
        sleep(3);
        assertEquals(numEvents + 1, getNumberOfEvents());
        Event e = getLastEvent();
        // no message when unknown number
        assertNull(e.message);
        // title used instead of friend name when payphone number
        assertEquals(getCallerInfoString(phoneNumber), e.title);
        assertEquals(Integer.toString(callId), e.eventKey);
    }

    public void testNumberMatching() {
        int numEvents = getNumberOfEvents();
        // 1. Add missed call
        String phoneNumber = "+4612312312";
        addCall(Calls.MISSED_TYPE, phoneNumber, null);
        String phoneNumber2 = "12312312";
        addCall(Calls.MISSED_TYPE, phoneNumber2, null);
        // 2. Check that it is added to event table
        sleep(3);
        assertEquals(numEvents + 2, getNumberOfEvents());
        Event e = getLastEvent();
        Event e2 = getNextLastEvent();
        // 3. The events should be linked with the same friend
        assertEquals(e.friendKey, e2.friendKey);
    }

    public void testContactsLinking() {
        // 1. Add contact
        String name = "Test Dummy";
        String phoneNumber = "+46123123";
        String dummyUri = addContact(name, phoneNumber);
        try {
            // 2. Add missed call from Contact
            addCall(Calls.MISSED_TYPE, phoneNumber, null);
            sleep(2);
            // 3. Check friend table for correct linking
            Event e = getLastEvent();
            Friend f = getFriend(e.message); // phone number is stored in the message field
            assertEquals(dummyUri, f.contactUri);
            assertEquals(name, f.displayName);
        } finally {
            removeContact(dummyUri);
        }
    }

    public void testUpdateFriend() {
        String name = "Test Dummy";
        String phoneNumber = "+46123123";

        // 1. Add missed call from unknown
        addCall(Calls.MISSED_TYPE, phoneNumber, null);
        // 2. Add person to contacts
        String dummyUri = addContact(name, phoneNumber);
        try {
            // 3. Add another missed call
            addCall(Calls.MISSED_TYPE, phoneNumber, null);
            sleep(2);
            // 4. Is the Friend table updated?
            Event e = getLastEvent();
            Friend f = getFriend(e.message); // phone number is stored in the message field
            assertEquals(dummyUri, f.contactUri);
            assertEquals(name, f.displayName);
        } finally {
            removeContact(dummyUri);
        }
    }

    public void testJunkCallID() {
        String junkCallID = "aaa";
        CallLogPluginService service = new CallLogPluginService();
        DummyContext context = new DummyContext();

        service.viewCallLogEvent(context,junkCallID);
        Intent intent = context.serviceIntent;
        assertNotNull("Intent was null", intent);
        assertNotNull("Intent.getAction returned null", intent.getAction());
        //setting type seems to set data to null.. just test type
        assertEquals("Type does not match", intent.getType(),Calls.CONTENT_TYPE );
    }

    public void testValidCallID() {
        String callId = "1234";
        //parse the int as viewCallLog event does to compare return value
        int intCallId = Integer.parseInt(callId);
        Uri comparisonUri = ContentUris.withAppendedId(CallLog.Calls.CONTENT_URI, intCallId);
        CallLogPluginService service = new CallLogPluginService();
        DummyContext context = new DummyContext();

        service.viewCallLogEvent(context, callId);
        Intent intent = context.serviceIntent;
        assertNotNull("Intent was null", intent);
        assertNotNull("Intent.getAction returned null", intent.getAction());
        //if parse int went OK, we will have a Uri in the data member
        assertEquals("Data does not match ", comparisonUri,intent.getData() );
    }

    public class DummyContext extends MockContext {

        public Intent serviceIntent = null;

        @Override
        public void startActivity(Intent intent) {
          serviceIntent = intent;
        }
    }

}
