/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.twitterplugin.view;

import android.app.AlertDialog;
import android.app.Dialog;
import android.app.Instrumentation;
import android.app.Instrumentation.ActivityMonitor;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.os.PatternMatcher;
import android.test.ActivityInstrumentationTestCase2;
import android.view.WindowManager.BadTokenException;
import android.widget.Button;
import android.widget.EditText;

import com.sonyericsson.eventstream.twitterplugin.R;
import com.sonyericsson.eventstream.twitterplugin.TwitterPluginApplication;
import com.sonyericsson.eventstream.twitterplugin.TwitterPluginApplication.State;
import com.sonyericsson.eventstream.twitterplugin.utility.PreferencesHelper;
import com.sonyericsson.eventstream.twitterplugin.utils.WaitForResult;

/**
 * Test class for the configuration activity
 * NOTE: Test of valid login handling is performed in PC test
 */
public class FConfigurationActivityTests extends ActivityInstrumentationTestCase2<ConfigurationActivity> {


    public FConfigurationActivityTests() {
        super("com.sonyericsson.eventstream.twitterplugin", ConfigurationActivity.class);
    }

    Context mContext;

    TwitterPluginApplication mApplication;

    ConfigurationActivity mActivity;

    private static final int TIME_OUT = 10000; //10s

    @Override
    protected void setUp() throws Exception {
        mContext = getInstrumentation().getTargetContext();

        // Clear all values
        SharedPreferences preferences = mContext.getSharedPreferences(
                PreferencesHelper.CONFIG_STORE, Context.MODE_PRIVATE);
        if (preferences != null) {
            SharedPreferences.Editor editor = preferences.edit();
            if (editor != null) {
                editor.clear();
                editor.commit();
            }
        }

        mActivity = getActivity();
        getInstrumentation().waitForIdleSync();
        mApplication = (TwitterPluginApplication)mActivity.getApplication();

        // Check that the default dialog is the disclaimer
        Dialog dialog = mActivity.mDisclaimerDialog;
        assertNotNull("Dialog should not be null", dialog);
        assertEquals("Dialog should be shown", true, dialog.isShowing());
        super.setUp();
    }

    @Override
    protected void tearDown() throws Exception {
        if (mApplication != null) {
            mApplication.setState(State.SERVICE_NOT_CONFIGURED);
        }
        super.tearDown();
    }

    public void testDisclaimerAccepted() {
        Dialog dialog = mActivity.mDisclaimerDialog;
        assertNotNull("Dialog should not be null", dialog);

        final Button ok = ((AlertDialog)dialog).getButton(Dialog.BUTTON_POSITIVE);
        assertNotNull(ok);

        // Click on the button
        getInstrumentation().runOnMainSync(new Runnable() {
            public void run() {
                ok.performClick();
            }
        });
        getInstrumentation().waitForIdleSync();

        dialog = mActivity.mDisclaimerDialog;
        if (dialog != null) {
            assertEquals("Dialog should not be shown", false, dialog.isShowing());
        }
    }

    public void testLoginDialogShowing() {
        mApplication.removeListener(mActivity);
        mApplication.setState(State.SERVICE_NOT_CONFIGURED);
        mActivity.onResume();
        getInstrumentation().waitForIdleSync();

        assertNotNull("Dialog should not be null", mActivity.mLoginDialog);
        assertEquals("Dialog should be shown", true, mActivity.mLoginDialog.isShowing());
    }

    public void testLoginDialogNotShowing() {
        mApplication.removeListener(mActivity);
        mApplication.setState(State.SERVICE_AUTHENTICATED);
        mActivity.onResume();
        Dialog dialog = mActivity.mLoginDialog;
        if (dialog != null) {
            assertNotNull("Dialog should be null", !dialog.isShowing());
        }
    }

    public void testProgressDialogShowingAtStartup() {
        mApplication.removeListener(mActivity);
        mApplication.setState(State.SERVICE_AUTHENTICATION_IN_PROGRESS);
        mActivity.onResume();
        assertNotNull("Dialog should not be null", mActivity.mProgressDialog);
        assertEquals("Dialog should be shown", true, mActivity.mProgressDialog.isShowing());
    }

    public void testProgressDialogShowingAtLogin() {
        mApplication.removeListener(mActivity);
        mApplication.setState(State.SERVICE_NOT_CONFIGURED);
        mActivity.onResume();
        getInstrumentation().waitForIdleSync();

        assertNotNull("Dialog should not be null", mActivity.mLoginDialog);
        assertEquals("Dialog should be shown", true, mActivity.mLoginDialog.isShowing());

        mApplication.setState(State.SERVICE_AUTHENTICATION_IN_PROGRESS);
        getInstrumentation().waitForIdleSync();

        assertNotNull("Dialog should not be null", mActivity.mProgressDialog);
        assertEquals("Dialog should be shown", true, mActivity.mProgressDialog.isShowing());
    }

    public void testProgressDialogNotShowing() {
        mApplication.removeListener(mActivity);
        mApplication.setState(State.SERVICE_AUTHENTICATION_FAILED);
        mActivity.onResume();
        getInstrumentation().waitForIdleSync();

        Dialog dialog = mActivity.mProgressDialog;
        if (dialog != null) {
            assertNotNull("Dialog should be null", !dialog.isShowing());
        }
    }

    public void testLogoutDialogShowing() {
        mApplication.removeListener(mActivity);
        mApplication.setState(State.SERVICE_AUTHENTICATED);
        mActivity.onResume();
        boolean dialogExist = true;
        try {
            mActivity.dismissDialog(ConfigurationActivity.DIALOG_LOGOUT_ID);
        } catch (IllegalArgumentException e) {
            dialogExist = false;
        }
        assertEquals("Dialog should exist", true, dialogExist);
    }

    public void testLogoutDialogNotShowing() {
        mApplication.removeListener(mActivity);
        mApplication.setState(State.SERVICE_AUTHENTICATION_SUCCESS);
        mActivity.onResume();
        getInstrumentation().waitForIdleSync();

        boolean dialogExist = true;
        try {
            mActivity.dismissDialog(ConfigurationActivity.DIALOG_LOGOUT_ID);
        } catch (IllegalArgumentException e) {
            dialogExist = false;
        }
        assertEquals("Dialog should not exist", false, dialogExist);
    }

    public void testWrongUsernameDialogShowing() {
        mApplication.setState(State.ACTIVITY_MISSING_USERNAME);
        getInstrumentation().waitForIdleSync();

        boolean dialogExist = true;
        try {
            mActivity.dismissDialog(ConfigurationActivity.DIALOG_ERROR_MISSING_USERNAME_ID);
        } catch (IllegalArgumentException e) {
            dialogExist = false;
        }
        assertEquals("Dialog should exist", true, dialogExist);
    }

    public void testWrongPasswordDialogShowing() {
        mApplication.setState(State.ACTIVITY_MISSING_PASSWORD);
        getInstrumentation().waitForIdleSync();

        final WaitForResult waitForResult = new WaitForResult();
        waitForResult.waitFor(TIME_OUT, new Runnable() {
            public void run() {
                boolean dialogExist = true;
                try {
                    mActivity.dismissDialog(ConfigurationActivity.DIALOG_ERROR_MISSING_PASSWORD_ID);
                } catch (IllegalArgumentException e) {
                    dialogExist = false;
                }
                waitForResult.setResult(dialogExist);
            }
        });

        assertEquals("Dialog should exist", true, waitForResult.isSuccess());
    }

    public void testNetworkFailedDialogShowing() {
        mApplication.setState(State.ACTIVITY_NETWORK_FAILED);
        getInstrumentation().waitForIdleSync();

        boolean dialogExist = true;
        try {
            mActivity.dismissDialog(ConfigurationActivity.DIALOG_ERROR_NETWORK_FAILED_ID);
        } catch (IllegalArgumentException e) {
            dialogExist = false;
        }
        assertEquals("Dialog should exist", true, dialogExist);
    }

    public void testLoginFailedDialogShowing() {
        mApplication.setState(State.SERVICE_AUTHENTICATION_FAILED);
        getInstrumentation().waitForIdleSync();

        boolean dialogExist = true;
        try {
            mActivity.dismissDialog(ConfigurationActivity.DIALOG_ERROR_INVALID_ACCOUNT_ID);
        } catch (IllegalArgumentException e) {
            dialogExist = false;
        }
        assertEquals("Dialog should exist", true, dialogExist);
    }

    public void testDismissProgressDialogWhenAuthenticatedState() {
        mApplication.removeListener(mActivity);
        mApplication.setState(State.SERVICE_AUTHENTICATION_IN_PROGRESS);
        mActivity.onResume();
        Dialog dialog = mActivity.mProgressDialog;

        assertNotNull("Dialog should not be null", dialog);
        assertEquals("Dialog should be shown", true, dialog.isShowing());

        mApplication.setState(State.SERVICE_AUTHENTICATED);
        dialog = mActivity.mProgressDialog;
        if (dialog != null) {
            assertNotNull("Dialog should be null", !dialog.isShowing());
        }
    }

    public void testCreateAccountActivityStarted() {
        mApplication.removeListener(mActivity);
        mApplication.setState(State.SERVICE_NOT_CONFIGURED);
        mActivity.onResume();
        getInstrumentation().waitForIdleSync();

        Dialog dialog = mActivity.mLoginDialog;
        assertNotNull("Dialog should not be null", dialog);
        assertEquals("Dialog should be shown", true, dialog.isShowing());

        final Button createButton = (Button)dialog.findViewById(R.id.create_btn);
        assertNotNull(createButton);

        // Set intent filter
        IntentFilter filter = new IntentFilter();
        filter.addAction(Intent.ACTION_VIEW);
        filter.addDataScheme("https");
        filter.addDataPath("mobile.twitter.com/signup", PatternMatcher.PATTERN_LITERAL);

        Instrumentation.ActivityMonitor monitor = new ActivityMonitor(
                filter, null, true);
        getInstrumentation().addMonitor(monitor);

        // Click on the button
        getInstrumentation().runOnMainSync(new Runnable() {
            public void run() {
                createButton.performClick();
            }
        });
        getInstrumentation().waitForIdleSync();
        assertEquals("Wrong hit count", 1, monitor.getHits());

        getInstrumentation().removeMonitor(monitor);
    }

    public void testWrongPasswordNotification() {
        mApplication.removeListener(mActivity);
        mApplication.setState(State.SERVICE_NOT_CONFIGURED);
        mActivity.onResume();
        getInstrumentation().waitForIdleSync();

        Dialog dialog = mActivity.mLoginDialog;
        assertNotNull("Dialog should not be null", dialog);
        assertEquals("Dialog should be shown", true, dialog.isShowing());

        final EditText username = (EditText)dialog.findViewById(R.id.login_username);
        assertNotNull(username);
        final EditText password = (EditText)dialog.findViewById(R.id.login_password);
        assertNotNull(password);
        final Button login = ((AlertDialog)dialog).getButton(Dialog.BUTTON_POSITIVE);
        assertNotNull(login);

        // Click on the button
        getInstrumentation().runOnMainSync(new Runnable() {
            public void run() {
                username.setText("username");
                password.setText("");
                login.performClick();
            }
        });
        getInstrumentation().waitForIdleSync();

        boolean dialogExist = true;
        try {
            mActivity.dismissDialog(ConfigurationActivity.DIALOG_ERROR_MISSING_PASSWORD_ID);
        } catch (IllegalArgumentException e) {
            dialogExist = false;
        }
        assertEquals("Dialog should exist", true, dialogExist);
    }

    public void testWrongUsernameNotification() {
        mApplication.removeListener(mActivity);
        mApplication.setState(State.SERVICE_NOT_CONFIGURED);
        mActivity.onResume();
        getInstrumentation().waitForIdleSync();

        Dialog dialog = mActivity.mLoginDialog;
        assertNotNull("Dialog should not be null", dialog);
        assertEquals("Dialog should be shown", true, dialog.isShowing());

        final EditText username = (EditText)dialog.findViewById(R.id.login_username);
        assertNotNull(username);
        final EditText password = (EditText)dialog.findViewById(R.id.login_password);
        assertNotNull(password);
        final Button login = ((AlertDialog)dialog).getButton(Dialog.BUTTON_POSITIVE);
        assertNotNull(login);

        // Click on the button
        getInstrumentation().runOnMainSync(new Runnable() {
            public void run() {
                username.setText("");
                password.setText("password");
                login.performClick();
            }
        });
        getInstrumentation().waitForIdleSync();

        boolean dialogExist = true;
        try {
            mActivity.dismissDialog(ConfigurationActivity.DIALOG_ERROR_MISSING_USERNAME_ID);
        } catch (IllegalArgumentException e) {
            dialogExist = false;
        }
        assertEquals("Dialog should exist", true, dialogExist);
    }

    public void testWrongLoginContentNotification() {
        mApplication.removeListener(mActivity);
        mApplication.setState(State.SERVICE_NOT_CONFIGURED);
        mActivity.onResume();
        getInstrumentation().waitForIdleSync();

        Dialog dialog = mActivity.mLoginDialog;
        assertNotNull("Dialog should not be null", dialog);
        assertEquals("Dialog should be shown", true, dialog.isShowing());

        final EditText username = (EditText)dialog.findViewById(R.id.login_username);
        assertNotNull(username);
        final EditText password = (EditText)dialog.findViewById(R.id.login_password);
        assertNotNull(password);
        final Button login = ((AlertDialog)dialog).getButton(Dialog.BUTTON_POSITIVE);
        assertNotNull(login);

        // Click on the button
        getInstrumentation().runOnMainSync(new Runnable() {
            public void run() {
                username.setText("");
                password.setText("");
                login.performClick();
            }
        });
        getInstrumentation().waitForIdleSync();

        boolean dialogExist = true;
        try {
            mActivity.dismissDialog(ConfigurationActivity.DIALOG_ERROR_MISSING_USERNAME_ID);
        } catch (IllegalArgumentException e) {
            dialogExist = false;
        }
        assertEquals("Dialog should exist", true, dialogExist);
    }

    public void testLoginDialogCancel() {
        mApplication.removeListener(mActivity);
        mApplication.setState(State.SERVICE_NOT_CONFIGURED);
        mActivity.onResume();
        getInstrumentation().waitForIdleSync();

        Dialog dialog = mActivity.mLoginDialog;
        assertNotNull("Dialog should not be null", dialog);
        assertEquals("Dialog should be shown", true, dialog.isShowing());

        final Button cancel = ((AlertDialog)dialog).getButton(Dialog.BUTTON_NEGATIVE);
        assertNotNull(cancel);

        // Click on the button
        getInstrumentation().runOnMainSync(new Runnable() {
            public void run() {
                cancel.performClick();
            }
        });
        getInstrumentation().waitForIdleSync();

        dialog = mActivity.mLoginDialog;
        if (dialog != null) {
            assertEquals("Dialog should not be shown", false, dialog.isShowing());
        }
    }

    public void testStateChangeAfterFinish() {
        mActivity.finish();
        Instrumentation instr = getInstrumentation();
        instr.waitForIdleSync();
        boolean exceptionThrown = false;
        try {

            mActivity.onStateChange(TwitterPluginApplication.State.ACTIVITY_SHOW_DISCLAIMER,
                    TwitterPluginApplication.State.SERVICE_NOT_AUTHENTICATED);
            instr.waitForIdleSync();
        } catch (BadTokenException e) {
            exceptionThrown = true;
        }
        assertFalse("Failed testcase: forbidden to execute in UIThread after the activity was finished!", exceptionThrown);
    }
}
