/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

/**
 * @file
 * @author Niklas Karlsson (niklas.karlsson@sonyericsson.com)
 */
package com.sonyericsson.eventstream.facebookplugin;

import com.sonyericsson.eventstream.facebookplugin.Facebook.ServiceState;
import com.sonyericsson.eventstream.facebookplugin.FacebookPluginApplication.CleanUpHandler;

import android.app.Instrumentation;
import android.test.InstrumentationTestCase;

import junit.framework.Assert;

/**
 *  Unit test of application class
 */
public class UFacebookPluginTestApplication extends InstrumentationTestCase {
    static boolean cleanupAlgorithmHasRun = false;
    FacebookPluginApplication mApp = null;

    @Override
    protected void setUp() throws Exception {
        if (mApp == null) {
            try {
                mApp = (FacebookPluginApplication)
                        Instrumentation.newApplication(FacebookPluginApplication.class,
                                    getInstrumentation().getTargetContext());
            } catch (Exception exception) {
                Assert.fail("Couldn't create application!");
            }
        }
    }

    /**
     * Test that clean up algorithm only
     * is triggered when starting to communicate
     * with facebook server
     */
    public void testCleanupAlgorithm() {
        CleanUpHandler handler = mApp.new CleanUpHandler () {
            protected boolean isCleanupNeeded(final long currentTime) {
                cleanupAlgorithmHasRun = true;
                return false;
            }
        };

        // Test start communication, login
        handler.onServiceStateChanged(ServiceState.PASSIVE, ServiceState.AUTHENTICATION_IN_PROGRESS);
        assertTrue("", cleanupAlgorithmHasRun);
        cleanupAlgorithmHasRun = false;

        // Test close communication, login finished
        handler.onServiceStateChanged(ServiceState.AUTHENTICATION_IN_PROGRESS, ServiceState.PASSIVE);
        assertFalse("", cleanupAlgorithmHasRun);

        // Test start communication, fetching data
        handler.onServiceStateChanged(ServiceState.PASSIVE, ServiceState.SERVER_OPERATION_IN_PROGRESS);
        assertTrue("", cleanupAlgorithmHasRun);
        cleanupAlgorithmHasRun = false;

        // Test close communication, fetching data finished
        handler.onServiceStateChanged(ServiceState.SERVER_OPERATION_IN_PROGRESS, ServiceState.PASSIVE);
        assertFalse("", cleanupAlgorithmHasRun);
    }

    /**
     * Test that the user isn't logged out if last communication
     * was less than 30 days ago
     * * Set that we are logged in 5 days ago
     * * Call callback
     * * Check that we are still logged in
     */
    public void testBelow30DayLimit() {
        Settings settings = new Settings(getInstrumentation().getContext());

        // Fake logged in
        settings.setAuthenticationToken("Dummy-token");
        settings.setOwnId("12233424");
        settings.setLastCommunicationTime(System.currentTimeMillis() - 5*24*60*60*1000L);

        // Call callback
        CleanUpHandler handler = mApp.new CleanUpHandler();
        handler.onServiceStateChanged(ServiceState.PASSIVE, ServiceState.SERVER_OPERATION_IN_PROGRESS);

        // Check settings...
        String authToken = settings.getAuthenticationToken();
        assertNotNull("Authentication token is null", authToken);
        String ownId = settings.getOwnId();
        assertEquals("Own id is invalid", "12233424", ownId);
    }

    /**
     * Test that the user is logged out if the last communication
     * with facebook was more than 30 days ago
     * * Set that we are logged in 30 days and 2 hours ago
     * * Call callback
     * * Check that we are logged out
     */
    public void testAbove30DayLimit() {
        Settings settings = new Settings(getInstrumentation().getContext());

        // Fake logged in
        settings.setAuthenticationToken("Dummy-token");
        settings.setOwnId("12233424");
        settings.setLastCommunicationTime(System.currentTimeMillis() - 30*24*60*60*1000L - 2*60*60*1000L);

        // Call callback
        CleanUpHandler handler = mApp.new CleanUpHandler();
        handler.onServiceStateChanged(ServiceState.PASSIVE, ServiceState.SERVER_OPERATION_IN_PROGRESS);

        // Check settings...
        String authToken = settings.getAuthenticationToken();
        assertNull("Authentication token isn't null", authToken);
        String ownId = settings.getOwnId();
        assertEquals("Own id isn't null", "", ownId);
    }
}