/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.facebookplugin;

import android.app.Activity;
import android.app.AlertDialog;
import android.app.Dialog;
import android.app.ProgressDialog;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.DialogInterface.OnCancelListener;
import android.content.DialogInterface.OnClickListener;
import android.net.Uri;
import android.os.Bundle;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.Window;
import android.widget.Button;
import android.widget.EditText;
import android.widget.TextView;

import com.sonyericsson.eventstream.facebookplugin.EventStreamConstants.Config;
import com.sonyericsson.eventstream.facebookplugin.FacebookPluginApplication.State;

import static com.sonyericsson.eventstream.facebookplugin.Constants.AUTHENTICATE_INTENT;
import static com.sonyericsson.eventstream.facebookplugin.Constants.PLUGIN_KEY;
import static com.sonyericsson.eventstream.facebookplugin.Constants.PLUGIN_KEY_PARAMETER;

/**
 * @author Erik Hellman <erik.hellman@sonyericsson.com>
 */
public class FacebookPluginConfig extends Activity implements OnClickListener,
        OnCancelListener, FacebookPluginApplication.StateListener {

    private static final String TAG = "FacebookPluginConfig";

    // Dialog constants used for onCreateDialog
    public static final int DIALOG_LOGIN = 0;
    public static final int DIALOG_LOGOUT = 1;
    public static final int DIALOG_PROGRESS = 2;
    public static final int DIALOG_ERROR = 3;
    public static final int DIALOG_DISCLAIMER = 4;

    // Bundle parameter names for dialog creation
    private final String TITLE = "title";
    private final String MESSAGE = "message";

    private String mErrorMessage = null;

    /**
     * The activity specific states
     */
    public enum ActivityStates {
        DISCLAIMER,
        ERROR,
        AUTHENTICATE,
        AUTHENTICATED
    };

    /** Keeps the activity specific states */
    private ActivityStates mCurrentActivityState;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        requestWindowFeature(Window.FEATURE_NO_TITLE);

        if (Config.DEBUG) {
            Log.d(TAG, "Config, onCreate");
        }

        setInitialStates();
    }

    /**
     * Set initial internal activity states
     */
    public void setInitialStates() {
        if (State.AUTHENTICATED.equals(getState())) {
            mCurrentActivityState = ActivityStates.AUTHENTICATED;
        } else {
            mCurrentActivityState = ActivityStates.AUTHENTICATE;
        }

        // Check if the user hasn't accepted the disclaimer
        if (!hasAcceptedDisclaimer()) {
            mCurrentActivityState = ActivityStates.DISCLAIMER;
        }
    }

    @Override
    protected void onResume() {
        super.onResume();

        // Start listen to plug-in state callback
        FacebookPluginApplication app = (FacebookPluginApplication) getApplication();
        app.addStateListener(this);

        // Check if the user hasn't accepted the disclaimer
        if (ActivityStates.DISCLAIMER.equals(getInternalState())) {
            setInternalState(ActivityStates.DISCLAIMER);
            return;
        }

        // Check the current state
        if (Config.DEBUG) {
            Log.d(TAG, "onResume state:" + getState());
        }
        switch (getState()) {
            case NOT_CONFIGURED:
                setInternalState(ActivityStates.AUTHENTICATE);
                break;

            case AUTHENTICATION_IN_PROGRESS:
                showDialog(DIALOG_PROGRESS);
                break;

            case AUTHENTICATED:
                closeDialog(DIALOG_PROGRESS);
                if (ActivityStates.AUTHENTICATE.equals(getInternalState())) {
                    finish();
                } else {
                    setInternalState(ActivityStates.AUTHENTICATED);
                }
                break;

            case AUTHENTICATION_FAILED:
                if (!ActivityStates.ERROR.equals(getInternalState())) {
                    setInternalState(ActivityStates.AUTHENTICATE);
                }
                break;

            case AUTHENTICATION_BAD_CREDENTIALS:
                if (!ActivityStates.ERROR.equals(getInternalState())) {
                    setInternalState(ActivityStates.AUTHENTICATE);
                }
                break;

            default:
                break;
        }
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
    }

    @Override
    protected void onPause() {
        super.onPause();

        // Remove callbacks
        FacebookPluginApplication app = (FacebookPluginApplication) getApplication();
        app.removeStateListener(this);

        closeDialog(DIALOG_PROGRESS);
    }

    @Override
    protected void onPrepareDialog(int id, Dialog dialog, Bundle args) {
        if (id == DIALOG_LOGIN) {
            EditText passwordEditor = (EditText) dialog.findViewById(R.id.login_password);
            passwordEditor.setText("");
        } else if (id == DIALOG_ERROR) {
            ((AlertDialog)dialog).setMessage(args.getString(MESSAGE));
        }
    }

    @Override
    protected Dialog onCreateDialog(int id, Bundle args) {
        if (Config.DEBUG) {
            Log.d(TAG, "onCreateDialog:" + id);
        }

        switch (id) {
            case DIALOG_LOGIN:
                return setupLoginDialog();
            case DIALOG_LOGOUT:
                return setupLogoutDialog();
            case DIALOG_PROGRESS:
                return setupProgressDialog();
            case DIALOG_ERROR:
                return setupErrorDialog();
            case DIALOG_DISCLAIMER:
                return setupDisclaimerDialog();
            default:
                return null;
        }
    }

    private Dialog setupDisclaimerDialog() {
        AlertDialog.Builder dialog = new AlertDialog.Builder(this, R.style.SEMCThemeAlertDialog);
        dialog.setIcon(R.drawable.facebook_popup_title_icn);
        dialog.setTitle(R.string.ts_facebook_disclaimer_title);
        View disclaimerView = getLayoutInflater().inflate(R.layout.disclaimer, null);
        TextView message = (TextView) disclaimerView.findViewById(R.id.disclaimerText);
        message.setText(R.string.ts_facebook_disclamer_message);
        dialog.setView(disclaimerView);
        dialog.setPositiveButton(R.string.gui_accept_terms_txt, new OnClickListener() {
            public void onClick(DialogInterface dialog, int whichButton) {
                setResult(RESULT_OK);
                setInternalState(ActivityStates.AUTHENTICATE);
            }
        });
        dialog.setNegativeButton(android.R.string.cancel, this);
        dialog.setOnCancelListener(this);
        return dialog.create();
    }

    private Dialog setupErrorDialog() {
        AlertDialog.Builder dialog = new AlertDialog.Builder(this, R.style.SEMCThemeAlertDialog);
        dialog.setIcon(android.R.drawable.ic_dialog_alert);
        dialog.setTitle(R.string.ts_facebook_authentication_failed_title);
        dialog.setMessage("");
        dialog.setPositiveButton(android.R.string.ok, new OnClickListener() {
            public void onClick(DialogInterface dialog, int whichButton) {
                setInternalState(ActivityStates.AUTHENTICATE);
            }
        });
        dialog.setOnCancelListener(this);
        return dialog.create();
    }

    private Dialog setupLoginDialog() {
        AlertDialog.Builder dialog = new AlertDialog.Builder(this, R.style.SEMCThemeAlertDialog);
        LayoutInflater inflater = LayoutInflater.from(this);
        final View loginView = inflater.inflate(R.layout.login, null);
        dialog.setView(loginView);
        dialog.setTitle(R.string.ts_facebook_login_label);
        dialog.setIcon(R.drawable.facebook_popup_title_icn);
        dialog.setPositiveButton(R.string.gui_sign_in_txt, new OnClickListener() {
            public void onClick(DialogInterface dialog2, int whichButton) {
                String username = ((TextView)loginView.findViewById(R.id.login_username)).getText()
                .toString();
                String password = ((TextView)loginView.findViewById(R.id.login_password)).getText()
                .toString();
                callFacebookLogin(username, password);
            }
        });
        Button btn = (Button) loginView.findViewById(R.id.create_btn);
        btn.setOnClickListener(new View.OnClickListener() {
            public void onClick(View v) {
                Intent intent = new Intent(Intent.ACTION_VIEW,
                        Uri.parse(Constants.FACEBOOK_REGISTER_URL));
                // launch activity
                if (intent != null) {
                    try {
                        startActivity(intent);
                    } catch (ActivityNotFoundException e) {
                        Log.e(TAG, "Failed to launch register account activity");
                    }
                }
            }
        });
        dialog.setNegativeButton(android.R.string.cancel, this);
        dialog.setOnCancelListener(this);
        return dialog.create();
    }

    private Dialog setupLogoutDialog() {
        AlertDialog.Builder dialog = new AlertDialog.Builder(this, R.style.SEMCThemeAlertDialog);
        dialog.setTitle(R.string.ts_facebook_logout_label);
        dialog.setMessage(R.string.ts_facebook_logout_message);
        dialog.setIcon(android.R.drawable.ic_dialog_alert);
        dialog.setPositiveButton(android.R.string.ok, new OnClickListener() {
            public void onClick(DialogInterface arg0, int arg1) {
                callFacebookLogout();
                finish();
            }
        });
        dialog.setNegativeButton(android.R.string.cancel, this);
        dialog.setOnCancelListener(this);
        return dialog.create();
    }

    private ProgressDialog setupProgressDialog() {
        ProgressDialog dialog = new ProgressDialog(this);
        dialog.setMessage(getText(R.string.ts_facebook_authentication_progress_text));
        dialog.setIndeterminate(true);
        dialog.setCancelable(false);
        return dialog;
    }

    private void callFacebookLogout() {
        Intent logoutIntent = new Intent(Constants.LOGOUT_INTENT);
        Context context = getApplicationContext();

        logoutIntent.setClass(context, FacebookService.class);
        logoutIntent.putExtra(PLUGIN_KEY_PARAMETER, PLUGIN_KEY);

        startService(logoutIntent);
    }

    private void callFacebookLogin(String username, String password) {
        if (username != null && password != null) {
            Intent authenticateIntent = new Intent(AUTHENTICATE_INTENT);
            Context context = getApplicationContext();

            authenticateIntent.putExtra(Constants.USERNAME_EXTRA, username);
            authenticateIntent.putExtra(Constants.PASSWORD_EXTRA, password);
            authenticateIntent.putExtra(PLUGIN_KEY_PARAMETER, PLUGIN_KEY);
            authenticateIntent.setClass(context, FacebookService.class);

            startService(authenticateIntent);
        }
    }

    private void callCancel() {
        finish();
    }

    public void onClick(DialogInterface dialog, int whichButton) {
        callCancel();
    }

    public void onCancel(DialogInterface arg0) {
        callCancel();
    }

    public void onStateChange(final FacebookNotification newState) {
        if (Config.DEBUG) {
            Log.d(TAG, "onStateChange:" + newState.getState());
        }

        Runnable runnable = new Runnable() {
            public void run() {
                if (!isFinishing()) {
                    switch (newState.getState()) {
                        case AUTHENTICATION_FAILED:
                            mErrorMessage = (String) newState.getExtraData();
                            closeDialog(DIALOG_PROGRESS);
                            setInternalState(ActivityStates.ERROR);
                            break;

                        case AUTHENTICATION_BAD_CREDENTIALS:
                            mErrorMessage = (String) newState.getExtraData();
                            closeDialog(DIALOG_PROGRESS);
                            setInternalState(ActivityStates.ERROR);
                            break;

                        case AUTHENTICATION_IN_PROGRESS:
                            showDialog(DIALOG_PROGRESS);
                            break;

                        case AUTHENTICATED:
                            closeDialog(DIALOG_PROGRESS);
                            finish();
                            break;
                    }
                }
            }
        };
        runOnUiThread(runnable);
    }

    /**
     * Indicate that the internal activity states has changed
     * @param state The new state
     */
    private void onActivityStateChanged(ActivityStates state) {
        if (Config.DEBUG) {
            Log.d(TAG, "onActivityStateChanged state: " + state);
        }

        if (state == null) {
            return;
        }

        switch (state) {
            case DISCLAIMER:
                showDialog(DIALOG_DISCLAIMER);
                break;

            case AUTHENTICATE:
                closeDialog(DIALOG_PROGRESS);
                showDialog(DIALOG_LOGIN);
                break;

            case AUTHENTICATED:
                showDialog(DIALOG_LOGOUT);
                break;

            case ERROR:
                Bundle failedDialogParams = new Bundle();
                failedDialogParams.putString(TITLE, getResources().getString(
                        R.string.ts_facebook_authentication_failed_title));
                failedDialogParams.putString(MESSAGE, mErrorMessage);
                mErrorMessage = null;
                showDialog(DIALOG_ERROR, failedDialogParams);
                break;

            default:
                break;
        }
    }

    /**
     * Set a new internal activity state
     * @param state The state to set
     */
    private void setInternalState(ActivityStates state) {
        if (state == null) {
            return;
        }
        mCurrentActivityState = state;
        onActivityStateChanged(state);
    }

    /**
     * Get the current activity state
     * @return The current state
     */
    private ActivityStates getInternalState() {
        return mCurrentActivityState;
    }

    private void closeDialog(int id) {
        try {
            dismissDialog(id);
        } catch (IllegalArgumentException exception) {

        }
    }

    private boolean hasAcceptedDisclaimer() {
        return new Settings(getApplicationContext()).hasAcceptedDisclaimer();
    }

    private State getState() {
        return ((FacebookPluginApplication) getApplication()).getState();
    }
}
