/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

/**
 * @file
 *
 * @author Maria Ekström (maria.ekstrom@sonyericsson.com)
 */

package com.sonyericsson.esdatasourceplugin;

import com.sonyericsson.esdatasourceplugin.EventStreamContentProvider.ConfigState;
import com.sonyericsson.esdatasourceplugin.EventStreamContentProvider.EnabledState;
import com.sonyericsson.esdatasourceplugin.EventStreamContentProvider.StatusSupport;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;

/**
 * Helper class for querying the database
 */
public class DBUtilities {

    private static DBUtilities mDBUtilitiesInstance;

    // Limit for number of events retrieved when querying events for the spline.
    private static final int EVENT_QUERY_LIMIT = 150;

    /**
     * Retrieves an instance of DBUtilities
     *
     * @return DBUtilities instance
     */
    public static DBUtilities getInstance() {
        if (mDBUtilitiesInstance == null) {
            mDBUtilitiesInstance = new DBUtilities();
        }
        return mDBUtilitiesInstance;
    }

    /**
     * Sets an instance of DBUtilities. Used for test purposes
     *
     * @param dbUtilities The instance to set
     */
    public static void setInstance(DBUtilities dbUtilities) {
        mDBUtilitiesInstance = dbUtilities;
    }

    /**
     * Queries the Plug-in table for the plug-ins that support status update
     * and have been configured or does not need configuration
     *
     * @param context The context
     * @return The cursor
     */
    public Cursor queryStatusUpdatePluginCursor(Context context) {
        return context.getContentResolver().query(
                EventStreamContentProvider.Uris.PLUGIN_URI,
                null,
                EventStreamContentProvider.PluginColumns.STATUS_SUPPORT + " = ? AND "
                        + EventStreamContentProvider.PluginColumns.CONFIGURATION_STATE + " != ? ",
                new String[] {
                        String.valueOf(EventStreamContentProvider.StatusSupport.HAS_SUPPORT_TRUE),
                        String.valueOf(EventStreamContentProvider.ConfigState.NOT_CONFIGURED)
                },
                null);
    }

    /**
     * Queries the Source table for all enabled sources
     *
     * @param context The context
     * @return The cursor
     */
    public Cursor queryEnabledSourceCursor(Context context) {
        return context.getContentResolver().query(
                EventStreamContentProvider.Uris.SOURCE_URI,
                null,
                EventStreamContentProvider.SourceColumns.ENABLED + " = ? ",
                new String[] {
                    String.valueOf(EventStreamContentProvider.EnabledState.ENABLED)
                },
                null);
    }

    /**
     * Queries the Plug-in table for the plug-ins that has a configuration
     * activity sorted by name in ascending order
     *
     * @param context The context
     * @return The cursor
     */
    public Cursor querySettingsPluginCursor(Context context) {
        return context.getContentResolver().query(
                EventStreamContentProvider.Uris.PLUGIN_URI,
                null,
                EventStreamContentProvider.PluginColumns.CONFIGURATION_ACTIVITY + " not null ",
                null,
                EventStreamContentProvider.PluginColumns.NAME + " ASC");
    }

    /**
     * Queries the Plug-in table for a plug-in with the input plug-in id that
     * has a configuration activity
     *
     * @param context The context
     * @param pluginId The pluginId of the requested plug-in
     * @return The cursor
     */
    public Cursor querySettingsPluginCursor(Context context, String pluginId) {
        return context.getContentResolver().query(
                EventStreamContentProvider.Uris.PLUGIN_URI,
                null,
                EventStreamContentProvider.PluginColumns.CONFIGURATION_ACTIVITY + " not null AND "
                        + EventStreamContentProvider.ID_COLUMN + " = ? ",
                new String[] {
                    pluginId
                },
                null);
    }

    /**
     * Queries the Plug-in table
     *
     * @param context The context
     * @param projection The requested projection for the query
     * @param selection The requested selection for the query
     * @param selectionArgs The requested selectionArgs for the query
     * @param sortOrder The requested sortOrder for the query
     * @return The cursor
     */
    public Cursor queryPluginCursor(Context context, String[] projection, String selection,
            String[] selectionArgs, String sortOrder) {
        return context.getContentResolver().query(
                EventStreamContentProvider.Uris.PLUGIN_URI,
                projection,
                selection,
                selectionArgs,
                sortOrder);
    }

    /**
     * Queries the Source table for all sources sorted by name in ascending
     * order
     *
     * @param context The context
     * @return The cursor
     */
    public Cursor querySourceCursor(Context context) {
        return context.getContentResolver().query(
                EventStreamContentProvider.Uris.SOURCE_URI,
                null,
                null,
                null,
                EventStreamContentProvider.SourceColumns.NAME + " ASC");
    }

    /**
     * Queries the Event table for all events to be displayed in the spline,
     * joined with the source table
     *
     * @param context The context
     * @param sourceId The sourceId of the requested spline events
     * @return The cursor
     */
    public Cursor querySplineEventCursor(Context context, String sourceId) {
        if (sourceId == null || sourceId.equals("")) {
            return eventSourceViewQuery(context);
        } else {
            return eventSourceViewQuery(context, sourceId);
        }
    }

    /**
     * Queries the Friend table
     *
     * @param context The context
     * @param projection The requested projection for the query
     * @param selection The requested selection for the query
     * @param selectionArgs The requested selectionArgs for the query
     * @return The cursor
     */
    public Cursor queryFriendCursor(Context context, String[] projection, String selection,
            String[] selectionArgs) {
        return context.getContentResolver().query(
                EventStreamContentProvider.Uris.FRIENDS_URI,
                projection,
                selection,
                selectionArgs,
                null);
    }

    /**
     * Performs a join query between event and source with data needed for the
     * main spline
     *
     * @param context The context
     * @return The cursor
     */
    private Cursor eventSourceViewQuery(Context context) {
        String sql = "SELECT "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.ID_COLUMN
                + " AS " + EventStreamContentProvider.ID_COLUMN + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.SOURCE_ID + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.PLUGIN_ID + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.MESSAGE + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.IMAGE_URI + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.PUBLISHED_TIME + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.STATUS_ICON_URI + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.TITLE + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.DISPLAY_NAME + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.FRIEND_KEY + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.EVENT_KEY + ", "
                + EventStreamContentProvider.Tables.SOURCE_TABLE_NAME + "."
                + EventStreamContentProvider.SourceColumns.ICON_URI
                + " FROM " + EventStreamContentProvider.Tables.EVENT_TABLE_NAME
                + " JOIN " + EventStreamContentProvider.Tables.SOURCE_TABLE_NAME + " ON "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.SOURCE_ID + " = "
                + EventStreamContentProvider.Tables.SOURCE_TABLE_NAME + "."
                + EventStreamContentProvider.ID_COLUMN
                + " WHERE " + EventStreamContentProvider.Tables.SOURCE_TABLE_NAME + "."
                + EventStreamContentProvider.SourceColumns.ENABLED + " != 0 "
                + " AND " + EventStreamContentProvider.EventColumns.OUTGOING + " = 0 "
                + " AND " + EventStreamContentProvider.EventColumns.HANDLED_TYPE + " = 0 "
                + " ORDER BY " + EventStreamContentProvider.EventColumns.PUBLISHED_TIME + " DESC"
                + " LIMIT " + EVENT_QUERY_LIMIT + ";";

        return context.getContentResolver().query(
                EventStreamContentProvider.RawQuery.URI,
                null,
                sql,
                null,
                null);
    }

    /**
     * Performs a join query between event and source with data needed for a
     * spline. The sourceId specifies the source of the events.
     *
     * @param context The context
     * @param sourceId The sourceId of the events
     * @return The cursor
     */
    private Cursor eventSourceViewQuery(Context context, String sourceId) {
        String sql = "SELECT "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.ID_COLUMN
                + " AS " + EventStreamContentProvider.ID_COLUMN + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.SOURCE_ID + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.PLUGIN_ID + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.MESSAGE + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.IMAGE_URI + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.PUBLISHED_TIME + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.STATUS_ICON_URI + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.TITLE + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.DISPLAY_NAME + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.FRIEND_KEY + ", "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.EVENT_KEY + ", "
                + EventStreamContentProvider.Tables.SOURCE_TABLE_NAME + "."
                + EventStreamContentProvider.SourceColumns.ICON_URI
                + " FROM " + EventStreamContentProvider.Tables.EVENT_TABLE_NAME
                + " JOIN " + EventStreamContentProvider.Tables.SOURCE_TABLE_NAME + " ON "
                + EventStreamContentProvider.Tables.EVENT_TABLE_NAME + "."
                + EventStreamContentProvider.EventColumns.SOURCE_ID + " = "
                + EventStreamContentProvider.Tables.SOURCE_TABLE_NAME + "."
                + EventStreamContentProvider.ID_COLUMN
                + " WHERE " + EventStreamContentProvider.EventColumns.OUTGOING + " = 0 "
                + " AND " + EventStreamContentProvider.EventColumns.HANDLED_TYPE + " = 0 "
                + " AND " + EventStreamContentProvider.EventColumns.SOURCE_ID + " = ? "
                + " ORDER BY " + EventStreamContentProvider.EventColumns.PUBLISHED_TIME + " DESC"
                + " LIMIT " + EVENT_QUERY_LIMIT + ";";

        return context.getContentResolver().query(
                EventStreamContentProvider.RawQuery.URI,
                null,
                sql,
                new String[] {
                    sourceId
                },
                null);
    }

    /**
     * Updates the enabled value for a specific source
     *
     * @param context The context
     * @param sourceId The id of the source to update
     * @param enabled The enabled value to set
     * @return The amount of values changed
     */
    public int updateSourceEnabledValue(Context context, String sourceId, int enabled) {
        ContentValues values = new ContentValues();
        values.put(EventStreamContentProvider.SourceColumns.ENABLED, enabled);
        return context.getContentResolver().update(
                EventStreamContentProvider.Uris.SOURCE_URI,
                values,
                EventStreamContentProvider.ID_COLUMN + " = ? ",
                new String[] {
                    String.valueOf(sourceId)
                });
    }

    /**
     * Performs a query for the current main spline status
     *
     * @param context The context
     * @return the Cursor containing status text and icon
     */
    public Cursor statusMessageQuery(Context context) {
        // SELECT
        // source.current_status,
        // source.icon_uri
        // FROM
        // source JOIN plugin ON
        // plugin_id=plugin._id
        // WHERE
        // plugin.config_state = 0 AND
        // plugin.status_support = 1 AND
        // source.enabled = 1 AND
        // status_timestamp != 0 AND
        // current_status not null

        String query = "SELECT " + EventStreamContentProvider.Tables.SOURCE_TABLE_NAME + "."
                + EventStreamContentProvider.SourceColumns.CURRENT_STATUS + ", "
                + EventStreamContentProvider.Tables.SOURCE_TABLE_NAME + "."
                + EventStreamContentProvider.SourceColumns.ICON_URI
                + " FROM " + EventStreamContentProvider.Tables.SOURCE_TABLE_NAME
                + " JOIN " + EventStreamContentProvider.Tables.PLUGIN_TABLE_NAME + " ON "
                + EventStreamContentProvider.SourceColumns.PLUGIN_ID + " = "
                + EventStreamContentProvider.Tables.PLUGIN_TABLE_NAME + "."
                + EventStreamContentProvider.PluginColumns.ID
                + " WHERE " + EventStreamContentProvider.Tables.PLUGIN_TABLE_NAME + "."
                + EventStreamContentProvider.PluginColumns.CONFIGURATION_STATE + " = "
                + ConfigState.CONFIGURED
                + " AND " + EventStreamContentProvider.Tables.PLUGIN_TABLE_NAME + "."
                + EventStreamContentProvider.PluginColumns.STATUS_SUPPORT + " = "
                + StatusSupport.HAS_SUPPORT_TRUE
                + " AND " + EventStreamContentProvider.Tables.SOURCE_TABLE_NAME + "."
                + EventStreamContentProvider.SourceColumns.ENABLED + " = "
                + EnabledState.ENABLED
                + " AND " + EventStreamContentProvider.SourceColumns.STATUS_TIMESTAMP + " != 0 "
                + " AND " + EventStreamContentProvider.SourceColumns.CURRENT_STATUS + " not null "
                + " ORDER BY " + EventStreamContentProvider.Tables.SOURCE_TABLE_NAME + "."
                + EventStreamContentProvider.SourceColumns.STATUS_TIMESTAMP + " DESC";

        return context.getContentResolver().query(
                EventStreamContentProvider.RawQuery.URI,
                new String[] {
                        EventStreamContentProvider.Tables.SOURCE_TABLE_NAME + "."
                                + EventStreamContentProvider.SourceColumns.CURRENT_STATUS,
                        EventStreamContentProvider.Tables.SOURCE_TABLE_NAME + "."
                                + EventStreamContentProvider.SourceColumns.ICON_URI
                },
                query,
                null,
                null);
    }
}
