/*********************************************************************
 *  ____                      _____      _                           *
 * / ___|  ___  _ __  _   _  | ____|_ __(_) ___ ___ ___  ___  _ __   *
 * \___ \ / _ \| '_ \| | | | |  _| | '__| |/ __/ __/ __|/ _ \| '_ \  *
 *  ___) | (_) | | | | |_| | | |___| |  | | (__\__ \__ \ (_) | | | | *
 * |____/ \___/|_| |_|\__, | |_____|_|  |_|\___|___/___/\___/|_| |_| *
 *                    |___/                                          *
 *                                                                   *
 *********************************************************************
 * Copyright 2010 Sony Ericsson Mobile Communications AB.            *
 * All rights, including trade secret rights, reserved.              *
 *********************************************************************/

package com.sonyericsson.eventstream.calllogplugin;

import android.content.ContentResolver;
import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.SQLException;
import android.net.Uri;
import android.util.Log;

import com.sonyericsson.eventstream.calllogplugin.PluginConstants.Config;
import com.sonyericsson.eventstream.calllogplugin.PluginConstants.EventStream;
import com.sonyericsson.eventstream.calllogplugin.PluginConstants.EventStream.EventTable;

import java.util.ArrayList;
import java.util.List;

public class EventStreamAdapter {

    static boolean isRegistered(Context context) {
        boolean result;
        Cursor cursor = null;

        try {
            cursor =
                context.getContentResolver().query(
                    EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI,
                    null,
                    null,
                    null,
                    null);

            result =
                cursor != null &&
                cursor.moveToFirst() &&
                cursor.getCount() > 0;
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to query a plugin");
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to query a plugin");
            throw new RuntimeException(exception);
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }

        return result;
    }

    static void register(Context context) {
        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Start registration of plugin.");
        }

        ContentValues values =
            getPluginRegistrationConfiguration(context);

        try {
            context.getContentResolver().insert(
                    EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI,
                    values);
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to register");
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to register");
            throw new RuntimeException(exception);
        }
    }

    static void updateRegistration(Context context) {
        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Updating existing registration.");
        }

        ContentValues values =
            getPluginRegistrationConfiguration(context);

        try {
            context.getContentResolver().update(
                    EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI,
                    values,
                    null,
                    null);
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to update registration");
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to update registration");
            throw new RuntimeException(exception);
        }
    }

    private static ContentValues getPluginRegistrationConfiguration(Context context) {
        String pluginName =
            context.getResources().getString(
                    R.string.calllog_plugin_name);
        String iconUri =
            new Uri.Builder().scheme(
                    ContentResolver.SCHEME_ANDROID_RESOURCE)
                    .authority(context.getPackageName())
                    .appendPath(Integer.toString(
                            R.drawable.icon)).toString();

        ContentValues values = new ContentValues();
        values.put(EventStream.PluginTable.NAME, pluginName);
        values.put(EventStream.PluginTable.ICON_URI, iconUri);
        values.put(EventStream.PluginTable.API_VERSION, 1);
        values.put(EventStream.PluginTable.CONFIGURATION_STATE,
                EventStream.ConfigState.CONFIGURATION_NOT_NEEDED);

        return values;
    }

    private static ContentValues getSourceRegistrationConfiguration(Context context) {
        String pluginName =
            context.getResources().getString(
                    R.string.calllog_source_name);
        String iconUri =
            new Uri.Builder().scheme(
                    ContentResolver.SCHEME_ANDROID_RESOURCE)
                    .authority(context.getPackageName())
                    .appendPath(Integer.toString(
                            R.drawable.icon)).toString();

        ContentValues values = new ContentValues();
        values.putNull(EventStream.SourceTable.CURRENT_STATUS);
        values.put(EventStream.SourceTable.NAME, pluginName);
        values.put(EventStream.SourceTable.ICON_URI, iconUri);

        return values;
    }

  //Update plugin and source names when language has changed
    static void handleLocaleChange(Context context) {

        if (Config.DEBUG) {
            Log.d(Config.LOG_TAG, "Handling locale change.");
        }
        String pluginName = context.getResources().getString(
                    R.string.calllog_plugin_name);

        String sourceName = context.getResources().getString(
                    R.string.calllog_source_name);

        setLocaleColumns(pluginName,sourceName, context);
    }


    static long getSourceId(Context context) {
        long sourceId = -1;
        Cursor cursor = null;

        try {
            cursor = context.getContentResolver().query(
                    EventStream.EVENTSTREAM_SOURCE_PROVIDER_URI,
                    new String[] {EventStream.SourceTable.ID_COLUMN},
                    null, null, null);
            if (cursor != null && cursor.moveToFirst()) {
                sourceId =
                    cursor.getLong(cursor.getColumnIndex(
                            EventStream.SourceTable.ID_COLUMN));
            }
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to query source");
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to query source");
            throw new RuntimeException(exception);
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }

        return sourceId;
    }

    static long registerSource(Context context) {
        long sourceId = -1;
        ContentValues values = getSourceRegistrationConfiguration(context);
        values.put(EventStream.SourceTable.ENABLED, 1);

        try {
            Uri uri = context.getContentResolver().insert(
                    EventStream.EVENTSTREAM_SOURCE_PROVIDER_URI,
                    values);
            sourceId = (int)ContentUris.parseId(uri);
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to create source");
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to create source");
            throw new RuntimeException(exception);
        }

        return sourceId;
    }

    static void updateSource(Context context, long sourceId) {
        ContentValues values = getSourceRegistrationConfiguration(context);

        try {
            int result = context.getContentResolver().update(
                    EventStream.EVENTSTREAM_SOURCE_PROVIDER_URI,
                    values,
                    EventStream.SourceTable.ID_COLUMN + " = " + sourceId,
                    null);

            if (result != 1) {
                Log.e(Config.LOG_TAG, "Failed to update source");
            }
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to update source");
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to update source");
            throw new RuntimeException(exception);
        }
    }

    static Cursor getFriends(Context context) {
        Cursor cursor = null;

        if (context == null) {
            return null;
        }

        try {
            cursor = context.getContentResolver().query(
                    EventStream.EVENTSTREAM_FRIEND_PROVIDER_URI,
                    null,
                    null,
                    null,
                    null);
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to query friends");
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to query friends");
            throw new RuntimeException(exception);
        }

        return cursor;
    }

    static String getFriend(Context context, String phonenumber) {
        String friendKey = null;
        Cursor cursor = null;

        if (context == null || phonenumber == null || phonenumber.length() == 0) {
            return null;
        }

        try {
            cursor =
                context.getContentResolver().query(
                        EventStream.EVENTSTREAM_FRIEND_PROVIDER_URI,
                        new String[] {EventStream.EventTable.FRIEND_KEY},
                        "PHONE_NUMBERS_EQUAL(friend_key, '" + phonenumber + "')",
                        null,
                        null);
            if (cursor != null && cursor.moveToFirst()) {
                friendKey = cursor.getString(cursor.getColumnIndex(
                        EventStream.EventTable.FRIEND_KEY));
            }
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to query friend");
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to query friend");
            throw new RuntimeException(exception);
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }

        return friendKey;
    }

    static void updateFriend(Context context, long friendId,
            String profileImage, String contactReference, String displayName) {
        ContentValues values = new ContentValues();

        if (context == null) {
            return ;
        }

        if (profileImage != null) {
            values.put(EventStream.FriendTable.PROFILE_IMAGE_URI, profileImage);
        } else {
            values.putNull(EventStream.FriendTable.PROFILE_IMAGE_URI);
        }

        if (contactReference != null) {
            values.put(EventStream.FriendTable.CONTACTS_REFERENCE, contactReference);
        } else {
            values.putNull(EventStream.FriendTable.CONTACTS_REFERENCE);
        }

        if (displayName != null) {
            values.put(EventStream.FriendTable.DISPLAY_NAME, displayName);
        } else {
            values.putNull(EventStream.FriendTable.DISPLAY_NAME);
        }

        try {
            int rowsUpdated = context.getContentResolver().update(
                    EventStream.EVENTSTREAM_FRIEND_PROVIDER_URI,
                    values,
                    EventStream.FriendTable.ID_COLUMN + "=" + friendId,
                    null);

            if (rowsUpdated != 1) {
                Log.w(Config.LOG_TAG, "Failed to update friend " + friendId);
            }
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to update friend");
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to update friend");
            throw new RuntimeException(exception);
        }
    }

    static long insertFriend(Context context,
            String profileImage, String contactReference, String displayName,
            String friendKey, long sourceId) {
        long friendId = -1;
        ContentValues values = new ContentValues();

        if (context == null) {
            return friendId;
        }

        if (profileImage != null) {
            values.put(EventStream.FriendTable.PROFILE_IMAGE_URI, profileImage);
        } else {
            values.putNull(EventStream.FriendTable.PROFILE_IMAGE_URI);
        }

        if (contactReference != null) {
            values.put(EventStream.FriendTable.CONTACTS_REFERENCE, contactReference);
        } else {
            values.putNull(EventStream.FriendTable.CONTACTS_REFERENCE);
        }

        if (displayName != null) {
            values.put(EventStream.FriendTable.DISPLAY_NAME, displayName);
        } else {
            values.putNull(EventStream.FriendTable.DISPLAY_NAME);
        }

        values.put(EventStream.FriendTable.FRIEND_KEY, friendKey);
        values.put(EventStream.FriendTable.SOURCE_ID, sourceId);

        try {
            Uri newFriend = context.getContentResolver().insert(
                    EventStream.EVENTSTREAM_FRIEND_PROVIDER_URI,
                    values);
            friendId = (int)ContentUris.parseId(newFriend);
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to create friend");
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to create friend");
            throw new RuntimeException(exception);
        }

        return friendId;
    }

    /**
     * Lookup the call id if the last call we inserted into the events table.
     *
     * @return  the callId
     */
    static long getLastInsertedCallId(Context context) {
        long callId = 0;
        Cursor cursor = null;

        if (context == null) {
            return callId;
        }

        try {
            cursor = context.getContentResolver().query(
                    EventStream.EVENTSTREAM_EVENT_PROVIDER_URI,
                    new String[] {EventTable.EVENT_KEY},
                    null,
                    null,
                    "_id desc limit 1");

            if (cursor != null && cursor.moveToFirst()) {
                callId = cursor.getLong(
                        cursor.getColumnIndex(EventTable.EVENT_KEY));
            }
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to query event");
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to query event");
            throw new RuntimeException(exception);
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }

        return callId;
    }

    static int bulkInsertEvents(Context context, ContentValues[] bulkValues) {
        int nrOfInserts = 0;

        if (context == null) {
            return nrOfInserts;
        }

        try {
            context.getContentResolver().bulkInsert(
                    EventStream.EVENTSTREAM_EVENT_PROVIDER_URI,
                    bulkValues);
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to create events");
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to create events");
            throw new RuntimeException(exception);
        }

        return nrOfInserts;
    }


    static int deleteAllEvents(Context context) {
        int result = 0;
        try {
            result =
                context.getContentResolver().delete(
                        EventStream.EVENTSTREAM_EVENT_PROVIDER_URI, null, null);
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to delete all events");
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to delete all events");
            throw new RuntimeException(exception);
        }

        return result;
    }

    static List<String> getEventKeys(Context context) {
        List<String> eventKeys = new ArrayList<String>();

        Cursor cursor = null;
        try {
            cursor = context.getContentResolver().query(EventStream.EVENTSTREAM_EVENT_PROVIDER_URI,
                    new String[] {
                        EventStream.EventTable.EVENT_KEY
                    }, null, null, null);
            if (cursor != null && cursor.moveToFirst()) {
                int index = cursor.getColumnIndexOrThrow(EventStream.EventTable.EVENT_KEY);
                do {
                    if (!cursor.isNull(index)) {
                        eventKeys.add(cursor.getString(index));
                    }
                } while (cursor.moveToNext());
            }
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to query events");
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to query events");
            throw new RuntimeException(exception);
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }

        return eventKeys;
    }

    static int deleteEvents(Context context, List<String> eventKeys) {
        int result = 0;
        String sql = EventStream.EventTable.EVENT_KEY + " = ?";
        String[] selectionArgs = new String[1];

        try {
            for (int i = 0; i < eventKeys.size(); i++) {
                selectionArgs[0] = eventKeys.get(i);
                context.getContentResolver().delete(
                        EventStream.EVENTSTREAM_EVENT_PROVIDER_URI,
                        sql, selectionArgs);
                result++;
            }

        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to delete events");
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to delete events");
            throw new RuntimeException(exception);
        }

        return result;
    }

    /*
     * Write user-defined values to columns affected by locale change
     */
    private static void setLocaleColumns(String pluginName, String sourceName, Context context) {
        ContentValues values = new ContentValues();

        putStringValueInColumn(EventStream.EVENTSTREAM_PLUGIN_PROVIDER_URI,
                               EventStream.PluginTable.NAME,
                               pluginName,
                               context);


        putStringValueInColumn(EventStream.EVENTSTREAM_SOURCE_PROVIDER_URI,
                               EventStream.SourceTable.NAME,
                               sourceName,
                               context);
    }

    private static void putStringValueInColumn (Uri tableUri, String columnName, String value, Context context) {
        ContentValues values = new ContentValues();
        int result;

        values.put(columnName, value);

        try {
            result = context.getContentResolver().update(
                    tableUri, values, null, null);
            if (result != 1) {
                Log.e(Config.LOG_TAG, "Failed to update column: " + columnName + "in: " + tableUri + "result= " + result);
            }
        } catch (SQLException exception) {
            Log.w(Config.LOG_TAG, "Failed to update column: " + columnName);
            throw new RuntimeException(exception);
        } catch (SecurityException exception) {
            Log.w(Config.LOG_TAG, "Failed to update column: " + columnName);
            throw new RuntimeException(exception);
        }
    }

}
