/*
 *  The MIT License
 *
 *  Copyright 2011 Masahiko, SAWAI <masahiko.sawai@gmail.com>.
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 *  THE SOFTWARE.
 */
package org.routine_work.simple_battery_logger.dropbox_uploader.db;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import java.io.File;
import java.util.Date;
import org.routine_work.util.Log;

public class CsvUploadStatusDBHelper extends SQLiteOpenHelper
	implements DBConstants
{

	public static final String LOG_TAG = "simple-battery-logger-dropbox";

	public CsvUploadStatusDBHelper(Context context)
	{
		super(context, DATABASE_NAME, null, DATABASE_VERSION);
	}

	@Override
	public void onCreate(SQLiteDatabase db)
	{
		Log.d(LOG_TAG, "CREATE_TABLE_SQL : " + CsvUploadStatus.CREATE_TABLE_SQL);
		Log.d(LOG_TAG, "CREATE_DATETIME_INDEX_SQL : " + CsvUploadStatus.CREATE_FILEPATH_INDEX_SQL);
		db.execSQL(CsvUploadStatus.CREATE_TABLE_SQL);
		db.execSQL(CsvUploadStatus.CREATE_FILEPATH_INDEX_SQL);
	}

	@Override
	public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion)
	{
		Log.d(LOG_TAG, "DROP_TABLE_SQL : " + CsvUploadStatus.DROP_TABLE_SQL);
		db.execSQL(CsvUploadStatus.DROP_TABLE_SQL);
		onCreate(db);
	}

	public long insertOrUpdate(SQLiteDatabase db, File file)
	{
		Log.v(LOG_TAG, "Hello");
		long rowID;

		String filePath = file.getAbsolutePath();
		long lastModfied = file.lastModified();
		Log.d(LOG_TAG, "filePath => " + filePath + "lastModified => " + lastModfied);

		ContentValues values = new ContentValues();
		values.put(CsvUploadStatus.Columns.FILEPATH, filePath);
		values.put(CsvUploadStatus.Columns.MODIFIED, lastModfied);

		db.beginTransaction();
		try
		{
			rowID = getIDByFilePath(db, filePath);
			if (rowID == -1)
			{
				rowID = db.insert(CsvUploadStatus.TABLE_NAME, null, values);
			}
			else
			{
				String whereClause = CsvUploadStatus.Columns.ID + " = ? ";
				String[] whereArgs =
				{
					String.valueOf(rowID),
				};
				values.put(CsvUploadStatus.Columns.ID, rowID);
				db.update(CsvUploadStatus.TABLE_NAME, values, whereClause, whereArgs);
			}
			db.setTransactionSuccessful();
		}
		finally
		{
			db.endTransaction();
		}

		Log.v(LOG_TAG, "Bye");
		return rowID;
	}

	public Cursor getCSVFileStatusByFilePath(SQLiteDatabase db, String filePath)
	{
		Log.v(LOG_TAG, "Hello");
		Log.d(LOG_TAG, "filePath => " + filePath);

		String selection = CsvUploadStatus.Columns.FILEPATH + " = ? ";
		String[] selectionArgs =
		{
			filePath,
		};
		Cursor cursor = db.query(CsvUploadStatus.TABLE_NAME, new String[]
		{
			CsvUploadStatus.Columns.ID,
			CsvUploadStatus.Columns.FILEPATH,
			CsvUploadStatus.Columns.MODIFIED,
		}, selection, selectionArgs, null, null, null);

		Log.v(LOG_TAG, "Bye");

		return cursor;
	}

	public boolean checkNecessaryToUpload(SQLiteDatabase db, File file)
	{
		boolean necessaryToUpload = true;
		Log.v(LOG_TAG, "Hello");
		Log.d(LOG_TAG, "file => " + file);

		String filePath = file.getAbsolutePath();
		long lastModified = file.lastModified();
		Cursor cursor = getCSVFileStatusByFilePath(db, filePath);
		if (cursor != null)
		{
			try
			{
				if (cursor.moveToFirst())
				{
					int index = cursor.getColumnIndex(CsvUploadStatus.Columns.MODIFIED);
					long previousUploadedFileModifiedTime = cursor.getLong(index);
					Log.d(LOG_TAG, "previous uploaded lastModified => " + previousUploadedFileModifiedTime + ", lastModified => " + lastModified + ", delta => " + (lastModified - previousUploadedFileModifiedTime));
					Log.d(LOG_TAG, "previous uploaded lastModified date => " + new Date(previousUploadedFileModifiedTime) + ", lastModified date => " + new Date(lastModified));
					if (previousUploadedFileModifiedTime >= lastModified)
					{
						necessaryToUpload = false;
					}
				}
			}
			finally
			{
				cursor.close();
			}
		}

		Log.d(LOG_TAG, "necessaryToUpload => " + necessaryToUpload);
		Log.v(LOG_TAG, "Bye");
		return necessaryToUpload;

	}

	public long getIDByFilePath(SQLiteDatabase db, String filePath)
	{
		long rowID = -1;
		Log.v(LOG_TAG, "Hello");

		Cursor cursor = getCSVFileStatusByFilePath(db, filePath);
		if (cursor != null)
		{
			try
			{
				if (cursor.moveToFirst())
				{
					int index = cursor.getColumnIndex(CsvUploadStatus.Columns.ID);
					rowID = cursor.getLong(index);
				}
			}
			finally
			{
				cursor.close();
			}
		}

		Log.d(LOG_TAG, "rowID => " + rowID);
		Log.v(LOG_TAG, "Bye");
		return rowID;
	}

	public long getModifiedByFilePath(SQLiteDatabase db, String filePath)
	{
		long modified = -1;
		Log.v(LOG_TAG, "Hello");

		Cursor cursor = getCSVFileStatusByFilePath(db, filePath);
		if (cursor != null)
		{
			try
			{
				if (cursor.moveToFirst())
				{
					int index = cursor.getColumnIndex(CsvUploadStatus.Columns.MODIFIED);
					modified = cursor.getLong(index);
				}
			}
			finally
			{
				cursor.close();
			}
		}

		Log.d(LOG_TAG, "modified => " + modified);
		Log.v(LOG_TAG, "Bye");
		return modified;
	}

	public Cursor getAllCSVFileUploadStatusCursor(SQLiteDatabase db)
	{
		final String[] projection =
		{
			CsvUploadStatus.Columns.ID,
			CsvUploadStatus.Columns.FILEPATH,
			CsvUploadStatus.Columns.MODIFIED,
		};
		final String sortOrder =
			CsvUploadStatus.Columns.FILEPATH + " " + SORT_DESCENDING;

		Cursor cursor = db.query(CsvUploadStatus.TABLE_NAME, projection,
			null, null, null, null, sortOrder);

		return cursor;
	}

	public void deleteAll(SQLiteDatabase db)
	{
		db.delete(CsvUploadStatus.TABLE_NAME, null, null);
	}

	public void reindex(SQLiteDatabase db)
	{
		db.execSQL(CsvUploadStatus.REINDEX_SQL);
	}

	public void vaccum(SQLiteDatabase db)
	{
		db.execSQL(CsvUploadStatus.VACCUM_SQL);
	}
}
