/*
 *  The MIT License
 *
 *  Copyright 2013 Masahiko, SAWAI <masahiko.sawai@gmail.com>.
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 *  THE SOFTWARE.
 */
package org.routine_work.simple_battery_logger.dropbox_uploader;

import android.app.IntentService;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.database.sqlite.SQLiteDatabase;
import android.net.Uri;
import com.dropbox.client2.exception.DropboxException;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import org.routine_work.simple_battery_logger.dropbox_uploader.db.CsvUploadStatusDBHelper;
import org.routine_work.util.Log;

/**
 *
 * @author sawai
 */
public class UploadCsvFilesService extends IntentService
{

	private static final String LOG_TAG = "simple-battery-logger-dropbox";
	private static final int NOTIFICATION_ID_UPLOAD_CSV_FILES = 101;
	private static final int UPLOAD_INTERVAL_FILE_COUNT = 16;
	private static final int UPLOAD_INTERVAL_TIME_IN_MILISEC = 10 * 1000;
	private CsvUploadStatusDBHelper dbHelper;
	private SQLiteDatabase db;

	public UploadCsvFilesService(String name)
	{
		super(name);
	}

	public UploadCsvFilesService()
	{
		this("UploadCsvFilesService");
	}

	@Override
	public void onCreate()
	{
		Log.v(LOG_TAG, "Hello");

		super.onCreate();
		dbHelper = new CsvUploadStatusDBHelper(this);
		db = dbHelper.getWritableDatabase();

		Log.v(LOG_TAG, "Bye");
	}

	@Override
	public void onDestroy()
	{
		Log.v(LOG_TAG, "Hello");

		if (db != null)
		{
			db.close();
			db = null;
		}
		super.onDestroy();

		Log.v(LOG_TAG, "Bye");
	}

	@Override
	protected void onHandleIntent(Intent intent)
	{
		Log.i(LOG_TAG, "UploadCsvFilesService  : Start at " + new Date());

		boolean result = false;
		String[] csvFilePaths = intent.getStringArrayExtra(BatteryConstants.EXTRA_CSV_FILE_PATHS);
		if (csvFilePaths != null && csvFilePaths.length > 0)
		{
			result = uploadCsvFiles(csvFilePaths);
			Log.i(LOG_TAG, "UploadCsvFilesService  : result =>  " + result);
		}

		// send broadcast intent
		Intent broadcastIntent = new Intent();
		broadcastIntent.setAction(BatteryConstants.UPLOAD_ALL_CSV_FILE_COMPLETED_ACTION);
		broadcastIntent.putExtra(BatteryConstants.EXTRA_RESULT, result);
		sendBroadcast(broadcastIntent);


		Log.i(LOG_TAG, "UploadCsvFilesService : End at " + new Date());
	}

	private boolean uploadCsvFiles(String[] csvFilePaths)
	{
		boolean result = true;
		Log.v(LOG_TAG, "Hello");

		if (UploadServiceUtils.checkCanUploadCSV(getApplicationContext()))
		{
			Log.i(LOG_TAG, "UploadCsvFilesService : checkCanUploadCSV is OK.");

			List<File> uploadTargetFileList = new ArrayList<File>();
			for (String csvFilePath : csvFilePaths)
			{
				File csvFile = new File(csvFilePath);
				uploadTargetFileList.add(csvFile);
			}

			try
			{
				int uploadedFileCount = 0;
				for (File uploadTargetFile : uploadTargetFileList)
				{
					// Upload
					Log.i(LOG_TAG, "UploadCsvFilesService : CSV file uploading => " + uploadTargetFile);
					nofityUploadCsvFile(uploadTargetFile);
					UploadServiceUtils.uploadOneCsvFile(getApplicationContext(), uploadTargetFile);
					dbHelper.insertOrUpdate(db, uploadTargetFile);

					// Wait
					uploadedFileCount++;
					if ((uploadedFileCount % UPLOAD_INTERVAL_FILE_COUNT) == 0)
					{
						Log.i(LOG_TAG, "UploadCsvFilesService : Upload Waiting... UPLOAD_WAIT_TIME_IN_MILISEC => " + UPLOAD_INTERVAL_TIME_IN_MILISEC);
						uploadInterval();
					}
				}
			}
			catch (DropboxException ex)
			{
				Log.e(LOG_TAG, ex.getLocalizedMessage(), ex);
				result = false;
			}
			catch (IOException ex)
			{
				Log.e(LOG_TAG, ex.getLocalizedMessage(), ex);
				result = false;
			}
			finally
			{
				cancelNotification();
			}
		}

		Log.v(LOG_TAG, "Bye");
		return result;
	}

	private void nofityUploadCsvFile(File csvFile)
	{
		Uri csvDataUri = Uri.fromFile(csvFile);
		Log.d(LOG_TAG, "csvDataUri => " + csvDataUri);
		Intent intent = new Intent(Intent.ACTION_VIEW);
		intent.setClassName("org.routine_work.simple_battery_logger", "org.routine_work.simple_battery_logger.CsvFileViewerActivity");
		intent.setData(csvDataUri);
		intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
		Log.d(LOG_TAG, "intent => " + intent);
		PendingIntent pendingIntent = PendingIntent.getActivity(this, 0, intent, 0);

		String ticker = getString(R.string.upload_csv_file_notification_ticker);
		String eventTitle = getString(R.string.upload_csv_file_notification_event_title);
		String eventText = getString(R.string.upload_csv_file_notification_event_text, csvFile.getName());

		Notification notification = new Notification(
			android.R.drawable.stat_sys_upload,
			ticker,
			System.currentTimeMillis());

		notification.setLatestEventInfo(this,
			eventTitle, eventText, pendingIntent);

		notification.flags = notification.flags
			| Notification.FLAG_ONGOING_EVENT;

		NotificationManager notificationManager = (NotificationManager) getSystemService(Context.NOTIFICATION_SERVICE);
		notificationManager.notify(NOTIFICATION_ID_UPLOAD_CSV_FILES, notification);
	}

	private void cancelNotification()
	{
		NotificationManager notificationManager = (NotificationManager) getSystemService(Context.NOTIFICATION_SERVICE);
		notificationManager.cancel(NOTIFICATION_ID_UPLOAD_CSV_FILES);
	}

	private void uploadInterval()
	{
		try
		{
			Thread.sleep(UPLOAD_INTERVAL_TIME_IN_MILISEC);
		}
		catch (InterruptedException ex)
		{
			Log.w(LOG_TAG, ex.getLocalizedMessage(), ex);
		}
	}
}
