/*
 *  The MIT License
 *
 *  Copyright 2011-2012 Masahiko, SAWAI <masahiko.sawai@gmail.com>.
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 *  THE SOFTWARE.
 */
package org.routine_work.simple_battery_logger;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.content.res.Resources;
import android.os.Bundle;
import android.os.Environment;
import android.preference.EditTextPreference;
import android.preference.ListPreference;
import android.preference.PreferenceActivity;
import android.preference.PreferenceScreen;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.widget.Toast;
import java.io.File;
import java.util.List;
import org.routine_work.simple_battery_logger.db.DBConstants;
import org.routine_work.simple_battery_logger.service.AllCsvFileMediaScanService;
import org.routine_work.simple_battery_logger.service.ServiceUtils;
import org.routine_work.simple_battery_logger.util.PreferenceUtils;
import org.routine_work.util.SystemResourceUtils;
import org.routine_work.util.Log;

/**
 *
 * @author sawai
 */
public class BatteryLoggerPreferenceActivity extends PreferenceActivity
	implements SharedPreferences.OnSharedPreferenceChangeListener
{

	private static final String LOG_TAG = "simple-battery-logger";
	private String loggingServiceEnabledKey;
	private String ignoreVoltageChangeOnlyKey;
	private String dataKeepingPeriodKey;
	private String csvAutoExportEnabledKey;
	private String csvExportDirectoryKey;
	private String csvSortOrderKey;
	private String csvDropboxUploaderKey;
	private BroadcastReceiver deleteAllDataCompletedReceiver = new DeleteAllDataCompletedReceiver();
	private BroadcastReceiver deleteAllCsvFileCompletedReceiver = new DeleteAllCsvFileCompletedReceiver();

	@Override
	protected void onCreate(Bundle savedInstanceState)
	{
		Log.v(LOG_TAG, "onCreate() Hello");
		super.onCreate(savedInstanceState);

		addPreferencesFromResource(R.xml.preferences);
		loggingServiceEnabledKey = getString(R.string.logging_service_enabled_key);
		ignoreVoltageChangeOnlyKey = getString(R.string.ignore_voltage_change_only_key);
		dataKeepingPeriodKey = getString(R.string.data_keeping_period_key);
		csvAutoExportEnabledKey = getString(R.string.csv_auto_export_enabled_key);
		csvExportDirectoryKey = getString(R.string.csv_export_directory_key);
		csvSortOrderKey = getString(R.string.csv_sort_order_key);
		csvDropboxUploaderKey = getString(R.string.csv_dropbox_uploader_key);

		Log.v(LOG_TAG, "onCreate() Bye");
	}

	@Override
	protected void onResume()
	{
		super.onResume();

		// register preference listener
		SharedPreferences preferences = getPreferenceScreen().getSharedPreferences();
		preferences.registerOnSharedPreferenceChangeListener(this);

		// register broadcast receiver
		IntentFilter intentFilter;
		intentFilter = new IntentFilter();
		intentFilter.addAction(BatteryConstants.DELETE_ALL_DATA_COMPLETED_ACTION);
		registerReceiver(deleteAllDataCompletedReceiver, intentFilter);

		intentFilter = new IntentFilter();
		intentFilter.addAction(BatteryConstants.DELETE_ALL_CSV_FILE_COMPLETED_ACTION);
		registerReceiver(deleteAllCsvFileCompletedReceiver, intentFilter);

		updateAllSummary();
		updateDropboxUploaderPreference();

	}

	@Override
	protected void onPause()
	{
		// unregister broadcast receiver
		unregisterReceiver(deleteAllDataCompletedReceiver);
		unregisterReceiver(deleteAllCsvFileCompletedReceiver);

		// unregister preference listener
		SharedPreferences preferences = getPreferenceScreen().getSharedPreferences();
		preferences.unregisterOnSharedPreferenceChangeListener(this);

		super.onPause();
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu)
	{
		MenuInflater menuInflater = getMenuInflater();
		menuInflater.inflate(R.menu.quit_menu, menu);

		return true;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item)
	{
		boolean result = true;

		int itemId = item.getItemId();
		switch (itemId)
		{
			case R.id.quit_menuitem:
				Log.d(LOG_TAG, "quit_menuitem is clicked.");
				DashboardActivity.quitApplication(this);
				break;
			default:
				if (itemId == SystemResourceUtils.getResourceId("android.R$id.home"))
				{
					Log.d(LOG_TAG, "home is clicked.");
					DashboardActivity.goDashboardActivity(this);
				}
				else
				{
					result = super.onOptionsItemSelected(item);
				}
				break;
		}
		return result;
	}

	public void onSharedPreferenceChanged(SharedPreferences prefs, String key)
	{
		Log.v(LOG_TAG, "onSharedPreferenceChanged(" + key + ") Hello");
		if (loggingServiceEnabledKey.equals(key)
			|| ignoreVoltageChangeOnlyKey.equals(key))
		{
			ServiceUtils.initializeLoggingService(this);
		}
		else if (dataKeepingPeriodKey.equals(key))
		{
			ServiceUtils.initializeDeleteOldDataTask(this);
		}
		else if (csvAutoExportEnabledKey.equals(key))
		{
			ServiceUtils.initializeExportCsvFileTask(this);
		}
		else if (csvExportDirectoryKey.equals(key))
		{
			PreferenceUtils.exportCsvExportDirectoryName(this);

			// Scan new export csv directory
			PreferenceUtils.putCsvMediaScanCompleted(this, false);
			Intent intent = new Intent(getApplicationContext(), AllCsvFileMediaScanService.class);
			startService(intent);
		}

		updateSummary(key);

		Log.v(LOG_TAG, "onSharedPreferenceChanged() Bye");
	}

	private void updateAllSummary()
	{
		Log.v(LOG_TAG, "Hello");

		updateSummary(null);

		Log.v(LOG_TAG, "Bye");
	}

	private void updateSummary(String key)
	{
		Log.v(LOG_TAG, "Hello");

		if (key == null || key.equals(dataKeepingPeriodKey))
		{
			ListPreference dataKeepingPeriodPreference = (ListPreference) getPreferenceScreen().findPreference(dataKeepingPeriodKey);
			CharSequence entry = dataKeepingPeriodPreference.getEntry();
			dataKeepingPeriodPreference.setSummary(entry);
		}

		if (key == null || key.equals(csvExportDirectoryKey))
		{
			EditTextPreference csvExportDirectoryPreference = (EditTextPreference) getPreferenceScreen().findPreference(csvExportDirectoryKey);
			String csvExportDirectory = csvExportDirectoryPreference.getText();
			File externalStorageDirectory = Environment.getExternalStorageDirectory();
			File csvExportDirectoryPath = new File(externalStorageDirectory, csvExportDirectory);
			csvExportDirectoryPreference.setSummary(csvExportDirectoryPath.getAbsolutePath());
		}

		if (key == null || key.equals(csvSortOrderKey))
		{
			ListPreference csvSortOrderListPreference = (ListPreference) getPreferenceScreen().findPreference(csvSortOrderKey);
			CharSequence sortOrder = csvSortOrderListPreference.getValue();
			String summary = null;
			if (DBConstants.SORT_ASCENDING.equals(sortOrder))
			{
				summary = getString(R.string.csv_sort_order_summary_ascending);
			}
			else if (DBConstants.SORT_DESCENDING.equals(sortOrder))
			{
				summary = getString(R.string.csv_sort_order_summary_descending);
			}
			csvSortOrderListPreference.setSummary(summary);
		}

		Log.v(LOG_TAG, "Bye");
	}

	private void updateDropboxUploaderPreference()
	{
		Log.v(LOG_TAG, "Hello");

		PreferenceScreen preference = (PreferenceScreen) getPreferenceScreen().findPreference(csvDropboxUploaderKey);
		Log.d(LOG_TAG, "preference => " + preference);

		Intent targetIntent = preference.getIntent();
		PackageManager pm = getPackageManager();
		List<ResolveInfo> activities = pm.queryIntentActivities(targetIntent,
			PackageManager.MATCH_DEFAULT_ONLY);
		Log.d(LOG_TAG, "activities => " + activities);
		if (activities != null && activities.isEmpty() == false)
		{
			preference.setEnabled(true);
		}
		else
		{
			preference.setEnabled(false);
		}

		Log.v(LOG_TAG, "Bye");
	}

	class DeleteAllDataCompletedReceiver extends BroadcastReceiver
	{

		@Override
		public void onReceive(Context context, Intent intent)
		{
			boolean result = intent.getBooleanExtra(BatteryConstants.EXTRA_RESULT, true);

			Resources resources = context.getResources();
			String message;
			if (result)
				message = resources.getString(R.string.delete_data_succeeded);
			else
				message = resources.getString(R.string.delete_data_failed);

			Toast.makeText(context, message, Toast.LENGTH_LONG).show();

		}
	}

	class DeleteAllCsvFileCompletedReceiver extends BroadcastReceiver
	{

		@Override
		public void onReceive(Context context, Intent intent)
		{
			boolean result = intent.getBooleanExtra(BatteryConstants.EXTRA_RESULT, true);

			Resources resources = context.getResources();
			String message;
			if (result)
				message = resources.getString(R.string.delete_csv_file_succeeded);
			else
				message = resources.getString(R.string.delete_csv_file_failed);

			Toast.makeText(context, message, Toast.LENGTH_LONG).show();

		}
	}
}