/*
 *  The MIT License
 *
 *  Copyright 2011-2012 Masahiko, SAWAI <masahiko.sawai@gmail.com>.
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 *  THE SOFTWARE.
 */
package org.routine_work.simple_battery_logger;

import android.app.Activity;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.BatteryManager;
import android.os.Bundle;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.widget.ImageView;
import android.widget.TextView;
import org.routine_work.util.Log;
import org.routine_work.util.SystemResourceUtils;

public class BatteryStatusActivity extends Activity
{

	public static final String LOG_TAG = "simple-battery-logger";
	private BatteryChangeReceiver batteryChangeReceiver;

	/**
	 * Called when the activity is first created.
	 */
	@Override
	public void onCreate(Bundle savedInstanceState)
	{
		super.onCreate(savedInstanceState);
		setContentView(R.layout.battery_status_activity);

		batteryChangeReceiver = new BatteryChangeReceiver();
	}

	@Override
	protected void onPause()
	{
		unregisterReceiver(batteryChangeReceiver);
		Log.d(LOG_TAG, "unregisterReceiver BatteryChangeReceiver");
		super.onPause();
	}

	@Override
	protected void onResume()
	{
		super.onResume();

		IntentFilter filter = new IntentFilter(Intent.ACTION_BATTERY_CHANGED);
		registerReceiver(batteryChangeReceiver, filter);
		Log.d(LOG_TAG, "registerReceiver BatteryChangeReceiver");
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu)
	{
		MenuInflater menuInflater = getMenuInflater();
		menuInflater.inflate(R.menu.preference_menu, menu);
		menuInflater.inflate(R.menu.quit_menu, menu);

		return true;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item)
	{
		boolean result = true;

		int itemId = item.getItemId();
		switch (itemId)
		{
			case R.id.preference_menuitem:
				Log.d(LOG_TAG, "preference_menuitem is clicked.");
				Intent prefIntent = new Intent(this, BatteryLoggerPreferenceActivity.class);
				startActivity(prefIntent);
				break;
			case R.id.quit_menuitem:
				Log.d(LOG_TAG, "quit_menuitem is clicked.");
				DashboardActivity.quitApplication(this);
				break;
			default:
				if (itemId == SystemResourceUtils.getResourceId("android.R$id.home"))
				{
					Log.d(LOG_TAG, "home is clicked.");
					DashboardActivity.goDashboardActivity(this);
				}
				else
				{
					result = super.onOptionsItemSelected(item);
				}
				break;
		}

		return result;
	}

	class BatteryChangeReceiver extends BroadcastReceiver
	{

		@Override
		public void onReceive(Context context, Intent intent)
		{
			int statusCode = intent.getIntExtra(BatteryConstants.EXTRA_STATUS, BatteryManager.BATTERY_STATUS_UNKNOWN);
			String statusText = BatteryUtils.getStatusText(context, statusCode);
			TextView statusTextView = (TextView) findViewById(R.id.status_textview);
			statusTextView.setText(statusText);

			int healthCode = intent.getIntExtra(BatteryConstants.EXTRA_HEALTH, BatteryManager.BATTERY_HEALTH_UNKNOWN);
			String healthText = BatteryUtils.getHealthText(context, healthCode);
			TextView healthTextView = (TextView) findViewById(R.id.health_textview);
			healthTextView.setText(healthText);

			int pluggedCode = intent.getIntExtra(BatteryConstants.EXTRA_PLUGGED, 0);
			String pluggedText = BatteryUtils.getPluggedText(context, pluggedCode);
			TextView pluggedTextView = (TextView) findViewById(R.id.plugged_textview);
			pluggedTextView.setText(pluggedText);

			int statusIconId = intent.getIntExtra(BatteryConstants.EXTRA_ICON_SMALL, 0);
			ImageView statusIconImageView = (ImageView) findViewById(R.id.status_icon_imageview);
			statusIconImageView.setImageResource(statusIconId);

			int level = intent.getIntExtra(BatteryConstants.EXTRA_LEVEL, 0);
			TextView levelTextView = (TextView) findViewById(R.id.level_textview);
			String levelText = String.valueOf(level);
			levelTextView.setText(levelText);

			int scale = intent.getIntExtra(BatteryConstants.EXTRA_SCALE, 100);
			TextView scaleTextView = (TextView) findViewById(R.id.scale_textview);
			String scaleText = String.valueOf(scale);
			scaleTextView.setText(scaleText);

			String levelPercentText = String.format("%d", level * 100 / scale);
			TextView levelPercentTextView = (TextView) findViewById(R.id.level_percent_textview);
			levelPercentTextView.setText(levelPercentText);

			int voltage = intent.getIntExtra(BatteryConstants.EXTRA_VOLTAGE, 0);
			String voltageText = String.valueOf(voltage);
			TextView voltageTextView = (TextView) findViewById(R.id.voltage_textview);
			voltageTextView.setText(voltageText);

			int temperature = intent.getIntExtra(BatteryConstants.EXTRA_TEMPERATURE, 0);
			String temperatureText = String.format("%.1f", (float) temperature / 10);
			TextView temperatureTextView = (TextView) findViewById(R.id.temperature_textview);
			temperatureTextView.setText(temperatureText);

			String technologyText = intent.getStringExtra(BatteryConstants.EXTRA_TECHNOLOGY);
			TextView technologyTextView = (TextView) findViewById(R.id.technology_textview);
			technologyTextView.setText(technologyText);

		}
	}
}
