unit RoundData;

interface

uses
	Windows, Messages, SysUtils, Classes,
	GikoSystem, BoardGroup;

type
	TGikoRoundType = (grtBoard, grtItem);
	TRoundItem = class;

	TRoundList = class(TObject)
	private
		FBoardList: TList;
		FItemList: TList;
		function GetCount(RoundType: TGikoRoundType): Integer;
		function GetRoundItem(Index: Integer; RoundType: TGikoRoundType): TRoundItem;
		function ParseRoundLine(Line: string; RoundType: TGikoRoundType): TRoundItem;
	public
		RoundNameList: TStringList;

		constructor Create;
		destructor Destroy; override;
		function Add(Board: TBoard): Integer; overload;
		function Add(ThreadItem: TThreadItem): Integer; overload;
		procedure Delete(Board: TBoard); overload;
		procedure Delete(ThreadItem: TThreadItem); overload;
		procedure Clear;
		function Find(Board: TBoard): Integer; overload;
		function Find(ThreadItem: TThreadItem): Integer; overload;
		property Count[RoundType: TGikoRoundType]: Integer read GetCount;
		property Items[Index: integer; RoundType: TGikoRoundType]: TRoundItem read GetRoundItem;
		procedure SetRoundName(Board: TBoard; RoundName: string); overload;
		procedure SetRoundName(ThreadItem: TThreadItem; RoundName: string); overload;

		procedure LoadRoundFile;
		procedure SaveRoundFile;
	end;

	TRoundItem = class(TObject)
	private
//		FBBSType: TGikoBBSType;
		FRoundName: string;
		FRoundType: TGikoRoundType;
		FBBSID: string;
		FBoardTitle: string;
		FThreadTitle: string;
		FFileName: string;
		FBoolData: Boolean;		//낢g傣
	public
//		property BBSType: TGikoBBSType read FBBSType write FBBSType;
		property RoundName: string read FRoundName write FRoundName;
		property RoundType: TGikoRoundType read FRoundType write FRoundType;
		property BBSID: string read FBBSID write FBBSID;
		property BoardTitle: string read FBoardTitle write FBoardTitle;
		property ThreadTitle: string read FThreadTitle write FThreadTitle;
		property FileName: string read FFileName write FFileName;
		property BoolData: Boolean read FBoolData write FBoolData;
	end;

var
	RoundList: TRoundList;

implementation
const
	ROUND_BOARD_FILENAME: string = 'RoundBoard.2ch';	//ƂBoardGroupֈړ
	ROUND_ITEM_FILENAME: string  = 'RoundItem.2ch';		//
	ROUND_INDEX_VERSION: string = '1.00';

constructor TRoundList.Create;
begin
	inherited;
	FBoardList := TList.Create;
	FItemList := TList.Create;
	RoundNameList := TStringList.Create;
	RoundNameList.Sorted := True;
	RoundNameList.Duplicates := dupIgnore;
end;

destructor TRoundList.Destroy;
begin
	RoundNameList.Free;
	Clear;
	FBoardList.Free;
	FItemList.Free;
	inherited;
end;

function TRoundList.Add(Board: TBoard): Integer;
var
	idx: Integer;
	Item: TRoundItem;
begin
	idx := Find(Board);
	if idx = -1 then begin
		Item := TRoundItem.Create;
//		Item.BBSType := gbt2ch;	//Ƃ肠
		Item.RoundType := grtBoard;
		Item.BBSID := Board.BBSID;
		Item.BoardTitle := Board.Title;
		Item.ThreadTitle := '';
		Item.FileName := '';
		Item.RoundName := Board.RoundName;
		FBoardList.Add(Item);
	end;
end;

function TRoundList.Add(ThreadItem: TThreadItem): Integer;
var
	idx: Integer;
	Item: TRoundItem;
begin
	idx := Find(ThreadItem);
	if idx = -1 then begin
		Item := TRoundItem.Create;
//		Item.BBSType := gbt2ch;	//Ƃ肠
		Item.RoundType := grtItem;
		Item.BBSID := ThreadItem.ParentBoard.BBSID;
		Item.BoardTitle := ThreadItem.ParentBoard.Title;
		Item.ThreadTitle := ThreadItem.Title;
		Item.FileName := ThreadItem.FileName;
		Item.RoundName := ThreadItem.RoundName;
		FItemList.Add(Item);
	end;
end;

procedure TRoundList.Delete(Board: TBoard);
var
	idx: Integer;
	Item: TRoundItem;
begin
	idx := Find(Board);
	if idx <> -1 then begin
		Item := TRoundItem(FBoardList[idx]);
		Item.Free;
		FBoardList.Delete(idx);
	end;
end;

procedure TRoundList.Delete(ThreadItem: TThreadItem);
var
	idx: Integer;
	Item: TRoundItem;
begin
	idx := Find(ThreadItem);
	if idx <> -1 then begin
		Item := TRoundItem(FItemList[idx]);
		Item.Free;
		FItemList.Delete(idx);
	end;
end;

procedure TRoundList.Clear;
var
	i: Integer;
begin
	for i := FBoardList.Count - 1 downto 0 do begin
		TRoundItem(FBoardList[i]).Free;
		FBoardList.Delete(i);
	end;
	for i := FItemList.Count - 1 downto 0 do begin
		TRoundItem(FItemList[i]).Free;
		FItemList.Delete(i);
	end;
end;

function TRoundList.Find(Board: TBoard): Integer;
var
	i: Integer;
	Item: TRoundItem;
begin
	Result := -1;
	for i := 0 to FBoardList.Count - 1 do begin
		Item := TRoundItem(FBoardList[i]);
		if Item.FRoundType <> grtBoard then Continue;
		if Item.FBBSID = Board.BBSID then begin
			Result := i;
			Exit;
		end;
	end;
end;

function TRoundList.Find(ThreadItem: TThreadItem): Integer;
var
	i: Integer;
	Item: TRoundItem;
begin
	Result := -1;
	for i := 0 to FItemList.Count - 1 do begin
		Item := TRoundItem(FItemList[i]);
		if Item.FRoundType <> grtItem then Continue;
		if (Item.FBBSID = ThreadItem.ParentBoard.BBSID) and (Item.FFileName = ThreadItem.FileName) then begin
			Result := i;
			Exit;
		end;
	end;
end;

procedure TRoundList.SetRoundName(Board: TBoard; RoundName: string);
var
	idx: Integer;
	Item: TRoundItem;
begin
	idx := Find(Board);
	if idx <> -1 then begin
		Item := TRoundItem(FBoardList[idx]);
		Item.RoundName := RoundName;
	end;
end;

procedure TRoundList.SetRoundName(ThreadItem: TThreadItem; RoundName: string);
var
	idx: Integer;
	Item: TRoundItem;
begin
	idx := Find(ThreadItem);
	if idx <> -1 then begin
		Item := TRoundItem(FItemList[idx]);
		Item.RoundName := RoundName;
	end;
end;

function TRoundList.GetCount(RoundType: TGikoRoundType): Integer;
begin
	Result := 0;
	if RoundType = grtBoard then
		Result := FBoardList.Count
	else if RoundType = grtItem then
		Result := FItemList.Count;
end;

function TRoundList.GetRoundItem(Index: Integer; RoundType: TGikoRoundType): TRoundItem;
begin
	Result := nil;
	if RoundType = grtBoard then begin
		if (Index >= 0) and (Index < FBoardList.Count) then
			Result := TRoundItem(FBoardList[Index]);
	end else if RoundType = grtItem then begin
		if (Index >= 0) and (Index < FItemList.Count) then
			Result := TRoundItem(FItemList[Index]);
	end;
end;

procedure TRoundList.LoadRoundFile;
var
	i: Integer;
	sl: TStringList;
	FileName: string;
	Item: TRoundItem;
begin
	sl := TStringList.Create;
	try
		//{[ht@Cǂݍ
		FileName := GikoSys.GetConfigDir + ROUND_BOARD_FILENAME;
		if FileExists(FileName) then begin
			sl.LoadFromFile(FileName);
			//Psڂ̓o[WȂ̂Ŗ
			for i := 1 to sl.Count - 1 do begin
				Item := ParseRoundLine(sl[i], grtBoard);
				FBoardList.Add(Item);
				RoundNameList.Add(Item.RoundName);
			end;
		end;
		//Xt@Cǂݍ
		FileName := GikoSys.GetConfigDir + ROUND_ITEM_FILENAME;
		if FileExists(FileName) then begin
			sl.LoadFromFile(FileName);
			//Psڂ̓o[WȂ̂Ŗ
			for i := 1 to sl.Count - 1 do begin
				Item := ParseRoundLine(sl[i], grtItem);
				FItemList.Add(Item);
				RoundNameList.Add(Item.RoundName);
			end;
		end;
	finally
		sl.Free;
	end;
end;

procedure TRoundList.SaveRoundFile;
var
	i: integer;
	FileName: string;
	sl: TStringList;
	s: string;
	Item: TRoundItem;
begin
	GikoSys.ForceDirectoriesEx(GikoSys.GetConfigDir);

	sl := TStringList.Create;
	try
		FileName := GikoSys.GetConfigDir + ROUND_BOARD_FILENAME;
		sl.Add(ROUND_INDEX_VERSION);
		for i := 0 to FBoardList.Count - 1 do begin
			Item := TRoundItem(FBoardList[i]);
			s := Item.BBSID + #1
				 + Item.BoardTitle + #1
				 + Item.RoundName;
			sl.Add(s);
		end;
		sl.SaveToFile(FileName);
		sl.Clear;
		FileName := GikoSys.GetConfigDir + ROUND_ITEM_FILENAME;
		sl.Add(ROUND_INDEX_VERSION);
		for i := 0 to FItemList.Count - 1 do begin
			Item := TRoundItem(FItemList[i]);
			s := Item.BBSID + #1
				 + Item.BoardTitle + #1
				 + Item.FileName + #1
				 + Item.ThreadTitle + #1
				 + Item.RoundName;
			sl.Add(s);
		end;
		sl.SaveToFile(FileName);
	finally
		sl.Free;
	end;
end;

function TRoundList.ParseRoundLine(Line: string; RoundType: TGikoRoundType): TRoundItem;
var
	s: string;
	i: Integer;
begin
	Result := TRoundItem.Create;
	if RoundType = grtBoard then begin
		Result.ThreadTitle := '';
		Result.FileName := '';
		Result.RoundType := grtBoard;
		for i := 0 to 2 do begin
			s := GikoSys.GetTokenIndex(Line, #1, i);
			case i of
				0: Result.BBSID := s;
				1: Result.BoardTitle := s;
				2: Result.RoundName := s;
			end;
		end;
	end else if RoundType = grtItem then begin
		Result.RoundType := grtItem;
		for i := 0 to 4 do begin
			s := GikoSys.GetTokenIndex(Line, #1, i);
			case i of
				0: Result.BBSID := s;
				1: Result.BoardTitle := s;
				2: Result.FileName := s;
				3: Result.ThreadTitle := s;
				4: Result.RoundName := s;
			end;
		end;
	end;
end;

end.
