﻿/**
FormulaDebugger.DebuggerForm

Copyright (c) 2015 Shigeyuki Horimoto

This software is released under the MIT License.
http://opensource.org/licenses/mit-license.php
*/
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using Formula;
using Formula.Node.Impl;
using Formula.Node;

namespace FormulaDebugger
{
    public partial class DebuggerForm : Form
    {
        public event EventHandler OnSave;
        
        public string DllPath { get; set; }
        public string Formula { get; set; }
        NodeTreeBuilder DebugTreeBuilder { get; set; }
        VariableManager VarManager { get; set; }
        /// <summary>
        /// 呼び出し元オブジェクト
        /// 保存実行時、返されます
        /// </summary>
        public Object CalledObject { get; set; }
        public HashSet<string> VariableNameSet { get; set; }
        //NodeTreeBuilder Tree
        public int LastUpdateId { get; set; }
        public DebuggerForm()
        {
            InitializeComponent();

            this.textBoxFormula.OnEndEval += _EndEvalEvent;
        }
        /// <summary>
        /// 初期化します
        /// </summary>
        public void Initialize()
        {
            this.DebugTreeBuilder = new NodeTreeBuilder(this.DllPath);
            this.DebugTreeBuilder.Initalize();

            this.textBoxFormula.builder = this.DebugTreeBuilder;
            this.textBoxFormula.Initialize();

            ToolStripMenuItemSave.Enabled = OnSave != null;
            //説明用
            //FunctionNode operatorTree = new FunctionNode() { Name = "Operator", Children = new List<FunctionNode>() };
            //FunctionNode functionTree = new FunctionNode() { Name = "Function", Children = new List<FunctionNode>() };
            List<FunctionNode> operatorList = new List<FunctionNode>();
            List<FunctionNode> methodList = new List<FunctionNode>();
            #region 演算子
            foreach (KeyValuePair<string, List< NodeInfo>> elems in this.DebugTreeBuilder.Parser.OperatorTable.OrderBy(a => a.Key))
            {
                foreach (NodeInfo elem in elems.Value)
                {
                    IFunctionNode instance = (IFunctionNode)elem.assembly.CreateInstance(elem.type.FullName);
                    List<string> args = new List<string>();
                    foreach (Type arg in instance.ArgCheckList)
                        args.Add(arg.Name);
                    operatorList.Add(new FunctionNode(elems.Key)
                    {
                        FunctionName = elems.Key
                        ,
                        DllPath = elem.assembly.Location
                        ,
                        //Note = string.Format(elem.type.FullName)
                        Note = ((IFunctionNode)elem.sample).Note
                        ,
                        Argument = string.Format("{0} = {2} {1} {3}"
                            , instance.ReturnNode.Name
                            , string.Join("", instance.Signature)
                            , instance.ArgCheckList.Count() > 0 ? instance.ArgCheckList[0].Name : ""
                            , instance.ArgCheckList.Count() > 1 ? instance.ArgCheckList[1].Name : "")
                    });
                }
            }
            #endregion
            #region 関数
            foreach (KeyValuePair<string, NodeInfo> elem in this.DebugTreeBuilder.Parser.MethodTable.OrderBy(a => a.Key))
            {
                IFunctionNode instance = (IFunctionNode)elem.Value.assembly.CreateInstance(elem.Value.type.FullName);
                List<string> args = new List<string>();
                foreach (Type arg in instance.ArgCheckList)
                    args.Add(arg.Name);
                methodList.Add(new FunctionNode(elem.Key)
                {
                    FunctionName = elem.Key
                    ,DllPath = elem.Value.assembly.Location
                    //,Note = elem.Value.type.FullName
                    ,Note = ((IFunctionNode)elem.Value.sample).Note
                    ,Argument = string.Format("{0} = {1} ( {2} )", instance.ReturnNode.Name, string.Join("", instance.Signature), string.Join(" , ", args))
                });
            }
            #endregion

            FunctionNode operatorNode = new FunctionNode("演算子" , operatorList.ToArray());
            FunctionNode methodNode = new FunctionNode("関数", methodList.ToArray());

            this.TreeViewFunction.Nodes.Add(operatorNode);
            this.TreeViewFunction.Nodes.Add(methodNode);

            this.textBoxFormula.Text = this.Formula ?? @"(1+(2+3)*4)*変数";
        }

        /// <summary>
        /// 式を外部から設定する
        /// </summary>
        /// <param name="formula"></param>
        public void setFormula(string formula)
        {
            this.Formula = formula;
            this.textBoxFormula.Text = formula;
        }

        /// <summary>
        /// イベント：テキストボックス内で式の評価が終了した
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void _EndEvalEvent(object sender , EventArgs e)
        {
            FormulaTextBox.EndEvalEventArgs ev = (FormulaTextBox.EndEvalEventArgs)e;
            this.toolStripStatusLabel.Text = null;
            if (ev.IsError)
                this.toolStripStatusLabel.Text = ev.ErrorMessage;
            else if(this.VariableNameSet != null)
            {
                List<string> vars = new List<string>();
                foreach(string varName in ev.VariableNameSet)
                {
                    if (!this.VariableNameSet.Contains(varName))
                        vars.Add(varName);
                }
                if(vars.Count > 0)
                {
                    this.toolStripStatusLabel.Text = string.Format("使用できない変数が宣言されています。{0}", string.Join(",", vars));
                }
            }
        }


        /// <summary>
        /// フォーム表示する
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void DebuggerForm_Shown(object sender, EventArgs e)
        {
            this.Initialize();
        }

        /// <summary>
        /// フォーム閉じる
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void DebuggerForm_FormClosed(object sender, FormClosedEventArgs e)
        {
            this.DebugTreeBuilder.Dispose();
        }

        /// <summary>
        /// ボタン：反映
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void buttonReflect_Click(object sender, EventArgs e)
        {
            if (string.IsNullOrWhiteSpace(this.textBoxFormula.Text)) return;

            List<Tuple<string, string>> before = new List<Tuple<string, string>>();
            //foreach (Variable var in this.data.VariableList)
            //    before.Add(var);
            this.dataGridViewVariable.Rows.Clear();

            this.DebugTreeBuilder.analyzeFormula(this.textBoxFormula.Text,false);

            this.VarManager = new VariableManager(this.DebugTreeBuilder.Tree);
            this.VarManager.loadVariable();
            
            //this.DebugTreeBuilder.Variable.loadVariable();
            //this.data.treeBuilder.Variable.variableDictionary.to
            //HashSet <string> variableNameSet = this.data.treeBuilder.Variable.loadVariable();
            //if (variableNameSet != null)
            foreach (string name in this.VarManager.variableDictionary.Keys)
            {
                var val = (from a in before
                           where a.Item1 == name
                           select a.Item2).SingleOrDefault();
                int index = this.dataGridViewVariable.Rows.Add();
                this.dataGridViewVariable.Rows[index].Cells["VariableName"].Value = name;
                this.dataGridViewVariable.Rows[index].Cells["VariableValue"].Value = val;
            }
            this.LastUpdateId = -1;
            NodeSplitterVisitor visitor = new NodeSplitterVisitor();
            this.textBoxDebuggerFormat.Text = visitor.visit(this.VarManager.Node);
            NodeSplitterOriginalVisitor originalVisitor = new NodeSplitterOriginalVisitor();
            this.textBoxDebuggerOriginal.Text = originalVisitor.visit(this.VarManager.Node);

        }

        /// <summary>
        /// ボタン：実行
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void buttonExecute_Click(object sender, EventArgs e)
        {
            
            if (this.VarManager == null) return;
            this.VarManager.variableDictionary.Clear();
            //foreach (Variable v in this.data.VariableList)
            foreach(DataGridViewRow row in this.dataGridViewVariable.Rows)
            {
                WordNode node = new WordNode();
                node.Value = (string)row.Cells["VariableValue"].Value;
                this.VarManager.variableDictionary.Add(
                    (string)row.Cells["VariableName"].Value
                    , node);
            }
            this.VarManager.setVariable();
            this.LastUpdateId = -1;

            //this.data.NodeString = string.Empty;
            NodeSplitterVisitor visitor = new NodeSplitterVisitor();
            this.textBoxDebuggerFormat.Text = visitor.visit(this.VarManager.Node.eval());
            NodeSplitterOriginalVisitor nodeOriginalVisitor = new NodeSplitterOriginalVisitor();
            this.textBoxDebuggerOriginal.Text = nodeOriginalVisitor.visit(this.VarManager.Node.eval());
        }

        /// <summary>
        /// イベント：ステップ実行ボタン
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void buttonStepExecute_Click(object sender, EventArgs e)
        {
            StepExecuteVisitor visitor = new StepExecuteVisitor();
            if ( this.VarManager == null) return;

            
            this.VarManager.variableDictionary.Clear();
            foreach (DataGridViewRow row in this.dataGridViewVariable.Rows)
            {
                WordNode node = new WordNode();
                node.Value = (string)row.Cells["VariableValue"].Value;
                this.VarManager.variableDictionary.Add(
                    (string)row.Cells["VariableName"].Value
                    , node);
            }
            this.VarManager.setVariable();

            StepExecuteVisitor stepVisitor = new StepExecuteVisitor();
            stepVisitor.Execute(this.VarManager.Node);
            this.LastUpdateId = stepVisitor.replaceToId;
            this.VarManager.Node = stepVisitor.currentNode;

            //this.data.NodeString = string.Empty;
            NodeSplitterVisitor nodeVisitor = new NodeSplitterVisitor();
            this.textBoxDebuggerFormat.Text = nodeVisitor.visit(this.VarManager.Node);
            NodeSplitterOriginalVisitor nodeOriginalVisitor = new NodeSplitterOriginalVisitor();
            this.textBoxDebuggerOriginal.Text = nodeOriginalVisitor.visit(this.VarManager.Node);
        }

        /// <summary>
        /// イベント:説明のツリー選択
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void TreeViewFunction_NodeMouseClick(object sender, TreeNodeMouseClickEventArgs e)
        {
            FunctionNode node = (FunctionNode)e.Node;
            this.labelFunctionDllPath.Text = node.DllPath;
            this.labelFunctionName.Text = node.FunctionName;
            this.labelFunctionArgument.Text = node.Argument;
            this.labelFunctionNote.Text = node.Note;
        }

        /// <summary>
        /// メニュー：反映
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ToolStripMenuItemReflect_Click(object sender, EventArgs e)
        {
            this.buttonReflect.PerformClick();
        }

        /// <summary>
        /// メニュー：実行
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ToolStripMenuItemExecute_Click(object sender, EventArgs e)
        {
            this.buttonExecute.PerformClick();
        }

        /// <summary>
        /// メニュー：ステップ実行
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ToolStripMenuItemStepExecute_Click(object sender, EventArgs e)
        {
            this.buttonStepExecute.PerformClick();
        }

        /// <summary>
        /// メニュー：デバッガ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ToolStripMenuItemDebugger_Click(object sender, EventArgs e)
        {
            this.tabControl1.SelectedIndex = 0;
        }

        /// <summary>
        /// メニュー：説明
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ToolStripMenuItemHelp_Click(object sender, EventArgs e)
        {
            this.tabControl1.SelectedIndex = 1;
        }

        /// <summary>
        /// メニュー：閉じる
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ToolStripMenuItemQuit_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        /// <summary>
        /// メニュー：反映する
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ToolStripMenuItemSave_Click(object sender, EventArgs e)
        {
            if (OnSave != null)
                OnSave(this, new DebuggerEventArgs() { Formula = this.textBoxFormula.Text , CalledObject = this.CalledObject});
        }

        public class DebuggerEventArgs : EventArgs{
            public string Formula { get; set; }
            public object CalledObject { get; set; }
        }

        
    }
}
