﻿/**
Formula.NodeTreeBuilder

Copyright (c) 2015 Shigeyuki Horimoto

This software is released under the MIT License.
http://opensource.org/licenses/mit-license.php
*/
using Formula.Node;
using Formula.Node.Impl;
using System;
using System.Collections.Generic;
using System.Reflection;

namespace Formula
{
    /// <summary>
    /// 式のパースから処理まで一連の処理を取りまとめるクラスです
    /// </summary>
    public class NodeTreeBuilder : IDisposable
    {
        /// <summary>
        /// DLLの読み出しパス
        /// </summary>
        public string DllPath { get; set; }
        
        public FormulaReader Reader { get; set; }
        public FormulaParser Parser { get; set; }
        //public VariableManager Variable { get; set; }
        public FormulaCheckVisitor.ErrorInformation ErrorInfo { get; set; }

        /// <summary>
        /// 式を解釈する途中の段階で呼ばれます。
        /// エラーチェックされる前に呼ばれます。
        /// 式の解釈は初期構築後の1回と、
        /// 優先順の組み換えが行われた際にN回呼ばれます。
        /// 優先順の組み換えは有無は式により異なる為、1回も呼ばれないこともあります。
        /// </summary>
        public event EventHandler OnStepParceFormula;

        /// <summary>
        /// 構成された構文木
        /// </summary>
        public INode Tree { get; set; }

        /// <summary>
        /// 構文木を構築します
        /// </summary>
        /// <param name="formula">処理対象となる式</param>
        public bool analyzeFormula(string formula , bool isCheckError = true )
        {
            //this.Variable = null;
            this.Tree = null;
            ObjectIdUtility.getInstance().reset();
            this.Reader.Reset();
            this.Reader.formula = formula;
            INode rootNode = this.Parser.parse();

            if (OnStepParceFormula != null)
                OnStepParceFormula(this, new ParceStepEventArgs() { Formula = formula, Node = rootNode });
            
            //FormulaCheckVisitor checkVisitor = null;
            //if (isCheckError)
            //{
            //    checkVisitor = new FormulaCheckVisitor();
            //    checkVisitor.TargetNode = rootNode;
            //    if (checkVisitor.isError())
            //    {
            //        this.ErrorInfo = checkVisitor.ErrorInfo;
            //        return false;
            //    }
            //}

            while (true)
            {
                ParentSetVisitor parentSetVisitor = new ParentSetVisitor();
                parentSetVisitor.currentNode = rootNode;
                parentSetVisitor.visit();

                PriorityChangeVisitor visitor = new PriorityChangeVisitor();
                visitor.currentNode = rootNode;
                if (!visitor.visit()) break;
                rootNode = rootNode.getRootNode();

                if (OnStepParceFormula != null)
                    OnStepParceFormula(this, new ParceStepEventArgs() { Formula = formula, Node = rootNode });
                
                //if (isCheckError)
                //{
                //    checkVisitor.TargetNode = rootNode;
                //    if (checkVisitor.isError())
                //    {
                //        this.ErrorInfo = checkVisitor.ErrorInfo;
                //        return false;
                //    }
                //}
            }

            PositionSetVisitor.updatePosition(rootNode);

            this.Tree = rootNode;
            if (isCheckError)
            {
                FormulaCheckVisitor checkVisitor = new FormulaCheckVisitor();
                checkVisitor.TargetNode = rootNode;
                if (checkVisitor.isError())
                {
                    this.ErrorInfo = checkVisitor.ErrorInfo;
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="dllPath"></param>
        public NodeTreeBuilder(string dllPath)
        {
            this.DllPath = dllPath;
        }
        /// <summary>
        /// 終了処理
        /// </summary>
        public void Dispose()
        {
            if(this.Reader != null)
                this.Reader.Dispose();
        }

        /// <summary>
        /// 初期化を行います
        /// </summary>
        public void Initalize()
        {
            //DLLの読み込み
            DllLoader loader = new DllLoader();
            loader.DllPath = this.DllPath;
            loader.Load();

            Console.Write(loader.ToString());

            //初期化
            this.Reader = new FormulaReader();
            this.Reader.reserved.Add(" ");

            BlockReader b = new BlockReader();
            b.target = this.Reader;
            b.escapeSequence = this.Reader.escapeSequence;
            b.blockSignature = this.Reader.blockSignature;


            this.Parser = new FormulaParser();
            this.Parser.Formula = b;

            foreach (KeyValuePair<Type, Assembly> a in loader.DllTable)
            {
                if(a.Key.GetInterface(typeof(IValueNode).FullName) != null && !a.Key.IsAbstract)
                {
                    if (a.Key == typeof(VariableNode)
                        || a.Key == typeof(ErrorNode)
                        || a.Key == typeof(WordNode)
                        || a.Key == typeof(BlankNode)) continue;

                    IValueNode instance = (IValueNode)a.Value.CreateInstance(a.Key.FullName);

                    this.Parser.ValueTable.Add(instance.Regex, new NodeInfo()
                    {
                        assembly = a.Value
                        ,
                        sample = instance
                        ,
                        type = a.Key
                    });
                }
                else if (a.Key.GetInterface(typeof(IFunctionNode).FullName) != null
                    && !a.Key.IsAbstract)
                {

                    IFunctionNode instance = (IFunctionNode)a.Value.CreateInstance(a.Key.FullName);

                    if (instance is IOperatorNode)
                    {
                        foreach (string signature in instance.Signature)
                            this.Reader.reserved.Add(signature.ToLower());
                    }

                    if (a.Key.GetInterface(typeof(IMethodNode).FullName) != null)
                    {
                        foreach (string signature in instance.Signature)
                        {
                            this.Parser.MethodTable.Add(signature.ToLower(), new NodeInfo()
                            {
                                type = a.Key
                                ,
                                assembly = a.Value
                                ,
                                sample = instance
                            });
                        }
                    }
                    else if (a.Key.GetInterface(typeof(IOperatorNode).FullName) != null)
                    {
                        foreach (string signature in instance.Signature)
                        {
                            if (!this.Parser.OperatorTable.ContainsKey(signature.ToLower()))
                                this.Parser.OperatorTable.Add(signature.ToLower(), new List<NodeInfo>());
                            this.Parser.OperatorTable[signature.ToLower()].Add( new NodeInfo()
                            {
                                type = a.Key
                                ,assembly = a.Value
                                ,sample = instance
                            });
                        }
                    }
                    else continue;

                    
                }
            }
            this.Reader.initialize();

        }

        /// <summary>
        /// ノードが整理される途中、もしくは終了時状態
        /// </summary>
        public class ParceStepEventArgs : EventArgs {
            public string Formula { get; set; }
            public INode Node { get; set; }
        }
    }
}
