﻿/**
FormulaDebugger.FormulaTextBox

Copyright (c) 2015 Shigeyuki Horimoto

This software is released under the MIT License.
http://opensource.org/licenses/mit-license.php
*/
using Formula;
using Formula.Node;
using FormulaDebugger.Utility;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace FormulaDebugger
{
    /// <summary>
    /// シンタックスハイライトの処理を自動で行うテキストボックスです。
    /// 式入力後自動で色分けが行われます。
    /// </summary>
    /// <remarks>
    /// パース処理は内部生成した別テキストボックスで行い、結果を書き戻す動作をします。
    /// これはちらつき防止のためです。
    /// </remarks>
    /// <remarks>
    /// シンタックスハイライトの処理は下記2モードいずれかで動作します。
    /// ・SyntaxMode=Timer
    /// 　入力後、{TimerInterval}ミリ秒の間未入力状態が続いたときに動作
    /// ・SyntaxMode=Every
    /// 　入力後、すぐに動作
    /// シンタックスハイライトの処理は同じスレッドで動作します。
    /// SyntaxMode=Every時、長文を入力した際は性能次第で重くなります。
    /// </remarks>
    public class FormulaTextBox : RichTextBox
    {
        
        /// <summary>
        /// シンタックスハイライト処理の動作モード
        /// </summary>
        [Description("シンタックスハイライト処理の動作モードを指定します。")]
        public SyntaxhighlighterMode SyntaxMode { get; set; }
        /// <summary>
        /// シンタックスハイライト処理が走るまでの間隔。
        /// syntaxMode=TImer時に利用されます。
        /// </summary>
        [Description("シンタックスハイライト処理の動作モードがtimerの際の間隔を指定します。(ミリ秒で指定)")]
        public int TimerInterval { get; set; }
        /// <summary>
        /// デフォルトフォント
        /// </summary>
        private Font _DefaultFont { get; set; }
        /// <summary>
        /// デフォルト文字色
        /// </summary>
        private Color _DefaultForeColor { get; set; }
        /// <summary>
        /// デフォルト背景色
        /// </summary>
        private Color _DefaultBackColor { get; set; }
        /// <summary>
        /// 式評価用ビルダ
        /// </summary>
        [Browsable(false)]
        public NodeTreeBuilder builder { get; set; }
        /// <summary>
        /// イベント：式の評価が終了した
        /// エラー発生の有無確認用を想定
        /// </summary>
        public event EventHandler OnEndEval;

        
        /// <summary>
        /// シンタックスハイライト動作用タイマー
        /// </summary>
        private Timer timer;

        /// <summary>
        /// 式の解釈・書式付テキスト構築用ダミーテキストボックス
        /// </summary>
        private RichTextBox dummyTextBox {get;set;}

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FormulaTextBox() : base()
        {
            this.dummyTextBox = new RichTextBox();
            this.dummyTextBox.TextChanged += dummyTextBox_TextChanged;
            this.TextChanged += _TextChanged;
            this.timer = new Timer();
            this.timer.Tick += _TimerTick;
        }

        /// <summary>
        /// クラスの初期化
        /// ・デフォルトのフォントを確保
        /// </summary>
        public void Initialize()
        {
            this._DefaultForeColor = this.ForeColor;
            this._DefaultBackColor = this.BackColor;
            this._DefaultFont = this.Font;
        }

        /// <summary>
        /// シンタックスハイライト処理を行います。
        /// </summary>
        private void Syntaxhighlight()
        {
            int selectPos = this.SelectionStart;
            this.dummyTextBox.SelectionStart = 0;
            this.dummyTextBox.SelectionLength = this.dummyTextBox.Text.Length + 1;
            this.dummyTextBox.ForeColor = this._DefaultForeColor;
            this.dummyTextBox.BackColor = this._DefaultBackColor;
            this.dummyTextBox.Font = this._DefaultFont;

            this.ForeColor = this._DefaultForeColor;
            this.BackColor = this._DefaultBackColor;
            this.Font = this._DefaultFont;
            //処理開始
            this.dummyTextBox.Text = this.Text;
            this.Rtf = this.dummyTextBox.Rtf;
            this.SelectionStart = selectPos;
        }


        /// <summary>
        /// イベント：式が変更された
        /// syntaxMode=timer用
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void _TimerTick(object sender, EventArgs e)
        {
            this.timer.Stop();
            this.TextChanged -= _TextChanged;
            this.Syntaxhighlight();
            this.TextChanged += _TextChanged;
            
        }

        /// <summary>
        /// イベント：式が変更された
        /// syntaxMode=every用
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void _TextChanged(object sender, EventArgs e)
        {
            this.timer.Stop();
            if (this.SyntaxMode == SyntaxhighlighterMode.Every)
            {
                this.TextChanged -= _TextChanged;
                this.Syntaxhighlight();
                this.TextChanged += _TextChanged;
            }
            else if (this.SyntaxMode == SyntaxhighlighterMode.Timer)
            {
                this.timer.Interval = this.TimerInterval;
                this.timer.Start();
            }
        }

        /// <summary>
        /// 式のパース用ダミーテキストボックス
        /// イベント：式が変更された
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void dummyTextBox_TextChanged(object sender, EventArgs e)
        {
            if (this.builder == null) return;

            //着色用オブジェクト初期化
            ColorUtility.getInstance().Reset();
            RichTextBox c = (RichTextBox)sender;

            //========================================================
            //式をパース、
            //構文木を走査しながらその内容をテキストボックスに反映する
            //エラーがある場合は赤で着色する。
            //========================================================
            string formula = c.Text;
            //構文木を組み立てる
            bool isOk = this.builder.analyzeFormula(formula , true);
            INode rootNode = this.builder.Tree;
            //構文木の内容からrichテキストボックスの該当箇所を着色する
            ColorSetVisitor.visit(c, rootNode);
            //構文エラーがある場合、赤で着色する
            FormulaCheckVisitor.ErrorInformation errorInfo = this.builder.ErrorInfo;
            if (!isOk)
            {
                c.SelectionStart = errorInfo.ErrorNode.startPosition;
                c.SelectionLength = errorInfo.ErrorNode.endPosition
                    - errorInfo.ErrorNode.startPosition
                    + 1;
                c.ForeColor = Color.Red;
            }
            //=================================
            //式で使用されている変数を読みだし、
            //イベント[OnEndEval]を処理する
            //=================================
            VariableManager man = new VariableManager(rootNode);
            man.loadVariable();
            List<string> varNameList = man.variableDictionary.Keys.ToList();
            HashSet<string> varNameSet = new HashSet<string>();
            foreach (string varName in varNameList)
                varNameSet.Add(varName);

            if (this.OnEndEval != null)
            {
                OnEndEval(this, new EndEvalEventArgs()
                {
                    IsError = !isOk
                    , ErrorMessage = errorInfo == null ? null : errorInfo.ErrorMessage
                    , ErrorNode = errorInfo == null ? null : errorInfo.ErrorNode
                    , Formula = formula
                    , VariableNameSet = varNameSet
                    , Node = rootNode
                });
            }
        }


        public class EndEvalEventArgs : EventArgs
        {
            /// <summary>
            /// エラーの有無
            /// </summary>
            public bool IsError { get; set; }
            /// <summary>
            /// エラーが発生してる場合、エラーが発生した箇所
            /// (エラーなしの場合、null)
            /// </summary>
            public INode ErrorNode { get; set; }
            /// <summary>
            /// エラーの内容
            /// (エラーなしの場合、null)
            /// </summary>
            public string ErrorMessage { get; set; }
            /// <summary>
            /// 評価された式
            /// </summary>
            public string Formula { get; set; }
            /// <summary>
            /// エラーが発生していない場合、
            /// パースされた変数のリスト
            /// </summary>
            public HashSet<string> VariableNameSet { get; set; }
            /// <summary>
            /// パースされた構文木
            /// （エラー発生時は最後まで構築されてないので注意）
            /// </summary>
            public INode Node { get; set; }
        }


    }
}
