﻿/**
Formula.Optimizer.VariableCashOptimizer

Copyright (c) 2016 Shigeyuki Horimoto

This software is released under the MIT License.
http://opensource.org/licenses/mit-license.php
*/
using Formula.Node;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Formula.Optimizer
{
    /// <summary>
    /// 式は途中変更されない事を前提として
    /// 使用されている変数のパターンとその結果ををキャッシュするクラスです。
    /// </summary>
    /// <remarks>
    /// このクラスは
    /// ・式単位(=変数抽出済みVariableManager)でインスタンスが必要です
    /// ・同じ変数値の組み合わせを多数処理する際に有効です。
    /// ・一つの式を使いまわす回数が多いことを前提としています。
    /// ・使用される値がランダムに近いほどメモリと時間を消費します。
    /// </remarks>
    public class VariableCashOptimizer
    {
        
        /// <summary>
        /// 読み込み済み変数設定クラス
        /// </summary>
        public VariableManager variableManager{ get; set; }

        /// <summary>
        /// 内部保持するキャッシュ
        /// </summary>
        private IDictionary<String, INode> cash;

        /// <summary>
        /// キャッシュ操作用同期オブジェクト
        /// </summary>
        private Object l = new object();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="manager"></param>
        public VariableCashOptimizer(VariableManager manager)
        {
            this.variableManager = manager;
            this.cash = new ConcurrentDictionary<string, INode>();
        }
        
        /// <summary>
        /// キャッシュの仕組みを利用せず単純実行します
        /// </summary>
        /// <returns></returns>
        public INode EvalNonCash()
        {
            this.variableManager.setVariable();
            return this.variableManager.Node.eval();
        }

        /// <summary>
        /// キャッシュの仕組みを利用して実行します。
        /// 式に含まれている変数のパターンが既に実行済みの場合、結果をキャッシュから返します。
        /// </summary>
        /// <returns></returns>
        public INode Eval()
        {
            IDictionary<string, IValueNode> dictionary = this.variableManager.variableDictionary;
            List<INode> varNodeList = this.variableManager.variableNode;
            StringBuilder msg = new StringBuilder();
            List<String> args = new List<String>();
            foreach (INode elem in varNodeList)
                msg.AppendLine(((IValueNode)dictionary[((IValueNode)elem).Value.ToString()]).Value.ToString());
            String key = msg.ToString();
            if (this.cash.ContainsKey(key))
                return this.cash[key];
            INode result = null;
            lock (this.l)
            {
                result = this.EvalNonCash();
                this.cash.Add(key, result);
            }
            return result;
        }
    }
}
