﻿/**
Formula.Optimizer.InnerMethodOptimizer

Copyright (c) 2016 Shigeyuki Horimoto

This software is released under the MIT License.
http://opensource.org/licenses/mit-license.php
*/

using Formula.Node;
using Formula.Node.Impl;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Formula.Optimizer.InnerMethodOptimizer
{
    public class InnerMethodOptimizer
    {

        /// <summary>
        /// 構文木の中から同じ式の箇所を見つけ、
        /// ダミーメソッドを作成し、
        /// 該当箇所をダミーメソッドで置き換えます。
        /// 引数で渡された式を直接加工します。
        /// 構文木を土台に判断するため、
        /// 単純な式（＝構文木に変換した際、直列になる）な場合、対象とはなりません。
        /// ※内部メソッド化することにより子ノードから親ノードへのリンクが切れます。
        /// ※ダミーメソッドは結果を保持するので実行前に必ずリセットする必要があります
        /// </summary>
        /// <param name="targetFormula">置換対象式</param>
        /// <returns>true：置き換えた false:置き換える箇所がない</returns>
        public static bool parce(INode targetFormula) {

            //===============================================
            // 構文木中出てくる式を直列化し、リストにまとめる
            //===============================================
            //演算子を抽出
            List<INode> nodeList = new List<INode>();
            NodeVisitor visitor = new NodeVisitor();
            visitor.NodeList = nodeList;
            visitor.eval(targetFormula);

            //抽出した演算子を含め、子ノードを直列化したものをリストにまとめる
            //（元の式をNodeでもって再現、リストに詰める）
            List<List<INode>> nodeConnectList = new List<List<INode>>();
            foreach (INode node in nodeList)
            {
                if (node is InnerMethodNode) continue;
                if (!(node is IFunctionNode)) continue;
                List<INode> nl = new List<INode>();
                NodeConnectVisitor.eval(nl, node);
                nodeConnectList.Add(nl);
            }
            //=================================
            // 同じ式構造を持つ箇所をまとめます
            //=================================
            //構文木の一覧をソート
            NodeComparator comparator = new NodeComparator();
            List<INode>[] nodeConnectArray = nodeConnectList.ToArray();
            Array.Sort(nodeConnectArray, comparator);
            //前後で同じ式の場合、メソッド化する箇所としてまとめる
            Dictionary<int, List<INode>> table = new Dictionary<int, List<INode>>();
            int id = 0;
            for (int i = 1; i < nodeConnectArray.Length; i++)
            {
                List<INode> before = nodeConnectArray[i - 1];
                List<INode> current = nodeConnectArray[i];
                if (comparator.Compare(before , current) != 0)
                {
                    id++;
                    continue;
                }
                //同じ式である場合、まとめて保持しておく
                if (!table.ContainsKey(id))
                {
                    table.Add(id, new List<INode>());
                    table[id].Add(before[0]);
                }
                table[id].Add(current[0]);
            }

            if (table.Count <= 0) return false;

            //================================================================
            // 同じ式が出てくる箇所にダミーメソッドを示すInnerMethodNodeを挟む
            //================================================================
            foreach (KeyValuePair<int, List<INode>> kv in table)
            {
                InnerMethodNode innerMethod = new InnerMethodNode();
                innerMethod.Args = new List<INode>() { kv.Value[0] };
                foreach(INode n in kv.Value)
                {
                    //構文木から該当するノードを見つけ、作成したダミーメソッドと置き換える
                    NodeReplaceVisitor replaceVisitor = new NodeReplaceVisitor();
                    replaceVisitor.TargetNode = targetFormula;
                    replaceVisitor.eval(n, innerMethod);
                }
            }
            
            return true;
        }


    }
}
