﻿/**
Formula.FormulaCheckVisitor

Copyright (c) 2015 Shigeyuki Horimoto

This software is released under the MIT License.
http://opensource.org/licenses/mit-license.php
*/

using Formula.Node;
using Formula.Node.Impl;
using System;
using System.Collections.Generic;
using System.Linq;

namespace Formula
{
    /// <summary>
    /// 構文木のエラー有無を確認するためのクラスです
    /// 確認は関数ノードに定義された引数・帰り値をもとに行います。
    /// </summary>
    public class FormulaCheckVisitor
    {
        /// <summary>
        /// 処理対象ノード
        /// </summary>
        public INode TargetNode;

        /// <summary>
        /// エラーの情報
        /// </summary>
        public ErrorInformation ErrorInfo;

        /// <summary>
        /// 処理対象ノードが親に持つIDリスト
        /// </summary>
        public int[] NodeIdArray = new int[] {  };

        /// <summary>
        /// エラー有無を確認します。
        /// </summary>
        /// <returns>True:エラーあり False:エラーなし</returns>
        public bool isError()
        {
            bool ret = false;
            int argCount = 0;

            if (this.TargetNode != null
                && this.NodeIdArray.Contains(this.TargetNode.nodeId))
            {
                this.ErrorInfo = new ErrorInformation()
                {
                    ErrorNode = this.TargetNode
                    , ErrorMessage = "文法が不正です。"
                };
                return true;
            }


            if (TargetNode is IOperatorNode)
            {
                foreach (INode n in ((IOperatorNode)TargetNode).Args)
                {
                    FormulaCheckVisitor v = new FormulaCheckVisitor();
                    v.TargetNode = n;
                    Array.Resize(ref v.NodeIdArray, this.NodeIdArray.Length + 1);
                    if (this.NodeIdArray.Length > 0)
                        Array.Copy(this.NodeIdArray, v.NodeIdArray, this.NodeIdArray.Length);
                    v.NodeIdArray[v.NodeIdArray.Length - 1] = this.TargetNode.nodeId;
                    if (v.isError()) {
                        this.ErrorInfo = v.ErrorInfo;
                        return true;
                    }
                }

                //引数として要求している型と、
                //実際に引数に設定されてる演算子/関数が返すであろう型の比較
                foreach (Type t in ((IOperatorNode)this.TargetNode).ArgCheckList)
                {
                    if (t != null)
                    {
                        INode n = ((IOperatorNode)this.TargetNode).Args[argCount];
                        if(n is IBracketNode && ((IBracketNode)n).Args.Count > 0)
                            n = ((IBracketNode)n).Args[0];
                        
                        if (argCount == 0 && ((IOperatorNode)TargetNode).IsPermitLeftNodeBlank)
                        {
                            if (n == null || n is BlankNode) continue;
                        }
                        else
                        {
                            if (n == null || n is BlankNode)
                            {
                                this.ErrorInfo = new ErrorInformation()
                                {
                                    ErrorNode = this.TargetNode
                                    , ErrorMessage = "左辺、もしくは右辺が指定されていません。"
                                };
                                return true;
                            }
                        }

                        Type resultType = n.GetType();
                        if (n is IFunctionNode)
                            resultType = ((IFunctionNode)n).ReturnNode;

                        if (!this.isExtends(resultType, t))
                        {
                            this.ErrorInfo = new ErrorInformation()
                            {
                                ErrorNode = this.TargetNode
                                , ErrorMessage = "左辺、もしくは右辺の型が異なります。"
                            };
                            return true;
                        }
                    }
                    argCount++;
                }
            }
            else if (TargetNode is IMethodNode)
            {
                foreach (INode n in ((IMethodNode)TargetNode).FormatArgs)
                {
                    FormulaCheckVisitor v = new FormulaCheckVisitor();
                    v.TargetNode = n;
                    Array.Resize(ref v.NodeIdArray, this.NodeIdArray.Length + 1);
                    Array.Copy(this.NodeIdArray, v.NodeIdArray, this.NodeIdArray.Length);
                    v.NodeIdArray[v.NodeIdArray.Length - 1] = this.TargetNode.nodeId;
                    if (v.isError())
                    {
                        this.ErrorInfo = v.ErrorInfo;
                        return true;
                    }
                }
                //引数として要求している型と、
                //実際に引数に設定されてる演算子/関数が返すであろう型の比較
                foreach (Type t in ((IMethodNode)this.TargetNode).ArgCheckList)
                {
                    if (((IMethodNode)this.TargetNode).FormatArgs.Count <= argCount) break;
                    if (t.IsArray)
                    {
                        #region 配列の場合
                        Type tm = t.GetElementType();
                        for (; argCount < ((IMethodNode)this.TargetNode).FormatArgs.Count; argCount++)
                        {
                            if (((IMethodNode)this.TargetNode).FormatArgs[argCount] is IFunctionNode)
                            {
                                if (!this.isExtends(((IFunctionNode)(((IMethodNode)TargetNode).FormatArgs[argCount])).ReturnNode, tm))
                                {
                                    this.ErrorInfo = new ErrorInformation()
                                    {
                                        ErrorNode = this.TargetNode
                                        ,
                                        ErrorMessage = "引数の型が異なります。"
                                    };
                                    return true;
                                }
                            }
                            else
                            {
                                if (((IMethodNode)this.TargetNode).FormatArgs[argCount] == null
                                    || ((IMethodNode)this.TargetNode).FormatArgs[argCount] is BlankNode)
                                {
                                    this.ErrorInfo = new ErrorInformation()
                                    {
                                        ErrorNode = this.TargetNode
                                        ,
                                        ErrorMessage = "引数が指定されていません。"
                                    };
                                    return true;
                                }
                                if (!this.isExtends((((IMethodNode)this.TargetNode).FormatArgs[argCount]).GetType(), tm))
                                {
                                    this.ErrorInfo = new ErrorInformation()
                                    {
                                        ErrorNode = this.TargetNode
                                        ,
                                        ErrorMessage = "引数の型が異なります。"
                                    };
                                    return true;
                                }
                            }
                        }
                        #endregion
                    }
                    else if(((IMethodNode)this.TargetNode).FormatArgs.Count > 0)
                    {
                        if (((IMethodNode)this.TargetNode).FormatArgs[argCount] is IFunctionNode)
                        {
                            if (!this.isExtends(((IFunctionNode)(((IMethodNode)this.TargetNode).FormatArgs[argCount])).ReturnNode, t))
                            {
                                this.ErrorInfo = new ErrorInformation()
                                {
                                    ErrorNode = this.TargetNode
                                    ,ErrorMessage = "引数の型が異なります。"
                                };
                                return true;
                            }
                        }
                        else
                        {
                            if (((IMethodNode)this.TargetNode).FormatArgs[argCount] == null) continue;
                            if (!this.isExtends((((IMethodNode)this.TargetNode).FormatArgs[argCount]).GetType(), t))
                            {
                                this.ErrorInfo = new ErrorInformation()
                                {
                                    ErrorNode = this.TargetNode
                                    ,ErrorMessage = "引数の型が異なります。"
                                };
                                return true;
                            }
                        }
                    }
                    argCount++;
                }
            }

            return false;
        }


        /// <summary>
        /// {arg1}が{arg2}を継承なりしているかどうかを返す。
        /// </summary>
        /// <param name="arg1">確認対象</param>
        /// <param name="arg2">確認する継承物</param>
        /// <returns>True:{arg1}は{arg2}と同じ、もしくは継承している False:継承してない</returns>
        private bool isExtends(Type arg1 , Type arg2)
        {
            if (arg1 == typeof(VariableNode)) return true;
            return arg1 == arg2 || arg1.IsSubclassOf(arg2) || arg1.GetInterfaces().Contains(arg2);
        }


        public class ErrorInformation
        {
            public INode ErrorNode { get; set; }
            public string ErrorMessage { get; set; }

            public override string ToString()
            {
                return string.Format("{0} {1}", this.ErrorNode, this.ErrorMessage);
            }
        }
    }
}
