﻿/**
Formula.BlockReader

Copyright (c) 2015 Shigeyuki Horimoto

This software is released under the MIT License.
http://opensource.org/licenses/mit-license.php
*/
using System.Collections.Generic;
using System.Text;
using System.Collections;

namespace Formula
{
    /// <summary>
    /// シグネチャによる取りまとめを行う。
    /// エスケープシーケンスが来た場合は続く一文字を演算子ではなく文字として扱う。
    /// ブロック（ダブルクォート）の認識を行う。
    /// </summary>
    public class BlockReader : IEnumerator<Word>, IEnumerable<Word>
    {
        /// <summary>
        /// 予約語であらかじめ分割されたリスト。
        /// エスケープシーケンス、ダブルクォートはそれぞれこのリストの一要素であること。
        /// </summary>
        public IEnumerator<string> target { get; set; }

        /// <summary>
        /// ひとまとめにした読み出し文字列
        /// </summary>
        private Word current;
        public Word Current
        {
            get { return this.current; }
        }

        /// <summary>
        /// １文節を構成するためのシグネチャ（通常ダブルクォート）
        /// </summary>
        public string blockSignature { get; set; }

        /// <summary>
        /// エスケープシーケンス
        /// </summary>
        public string escapeSequence { get; set; }

        /// <summary>
        /// 読みだした文字要素
        /// </summary>
        object IEnumerator.Current
        {
            get { return this.current; }
        }

        /// <summary>
        /// 次の要素を読みだす
        /// </summary>
        /// <returns>True:読み出し成功 False:すでに末尾まで読み込み済み</returns>
        public bool MoveNext()
        {
            bool ret = false;
            bool isin = false;
            StringBuilder value = new StringBuilder();
            int startPosition = -1;
            int endPosition = 0;
            bool isBlock = false;
            while (target.MoveNext())
            {
                //EscapeSequenceの場合、次を読み込む。
                bool isEscape = false;
                if (target.Current == escapeSequence)
                {
                    isEscape = true;
                    target.MoveNext();
                }
                //エスケープされたシグネチャではないばあい、
                //ブロックの内と外を切り替える。
                //以外の場合は結果文字列に足し込む
                if (!isEscape && target.Current == blockSignature)
                {
                    isin = !isin;
                    isBlock = true;
                    if(isin) startPosition = ((FormulaReader)target).Position;
                    else endPosition = ((FormulaReader)target).Position;
                }
                else
                {
                    if (startPosition < 0 && this.target is FormulaReader)
                        startPosition = ((FormulaReader)target).Position - target.Current.Length + 1;
                    value.Append(target.Current);
                }
                if (this.target is FormulaReader && endPosition < ((FormulaReader)target).Position)
                    endPosition = ((FormulaReader)target).Position;
                //if (!isin & target.Current == blockSignature)
                //    endPosition -= 1;
                ret = true;
                if (!isin) break; 
            }

            this.current = new Word();
            this.current.isBlock = isBlock;
            this.current.startPosition = startPosition;
            this.current.endPosition = endPosition;
            this.current.word = value.ToString();

            return ret;
        }

        /// <summary>
        /// 読み出し位置を初期化する
        /// </summary>
        public void Reset()
        {
            if (this.target != null)
                this.target.Reset();
        }

        public IEnumerator<Word> GetEnumerator()
        {
            return this;
        }


        IEnumerator IEnumerable.GetEnumerator()
        {
            return this;
        }

        public void Dispose(){ }
    }
}
