// -*- C++ -*-
/*!
 * @file  PublisherNew.h
 * @brief PublisherNew class
 * @date  $Date: 2007-12-31 03:08:06 $
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2006-2009
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id: PublisherNew.h 1799 2010-01-27 02:20:24Z fsi-katami $
 *
 */

#ifndef RTC_PUBLISHERNEW_H
#define RTC_PUBLISHERNEW_H

#include <coil/Task.h>
#include <coil/Mutex.h>
#include <coil/Condition.h>
#include <coil/PeriodicTask.h>

#include <rtm/RTC.h>
#include <rtm/PublisherBase.h>
#include <rtm/CdrBufferBase.h>
#include <rtm/DataPortStatus.h>
#include <rtm/SystemLogger.h>
#include <rtm/ConnectorBase.h>
#include <rtm/ConnectorListener.h>

namespace coil
{
  class Properties;
};

namespace RTC
{
  class InPortConsumer;
  /*!
   * @if jp
   * @class PublisherNew
   * @brief PublisherNew 饹
   *
   * Хåե˿ǡǼ줿ߥ󥰤ǡοǡ롣
   * ǡХߥ󥰤Ԥĥ󥷥塼ޤФ¦Ȥϰۤʤ륹åɤ
   * ư˻ѡ
   * PublisherζưϡǡФΥߥ󥰤ˤʤޤǥ֥å졢
   * Хߥ󥰤Τȡ¨¤˥󥷥塼ޤнƤӽФ
   *
   * @else
   * @class PublisherNew
   * @brief PublisherNew class
   *
   * Send new data at timing of when it is stored into the buffer.
   * This class is used when operating Consumer that waits for the data send
   * timing in different thread from one of the send side.
   * Publisher's driven is blocked until the data send timing reaches, if the
   * send timing notification is received, the Consumer's send processing will
   * be invoked immediately.
   *
   * @endif
   */

  class PublisherNew
    : public PublisherBase
  {
  public:
    typedef coil::Mutex Mutex;
    typedef coil::Condition<Mutex> Condition;
    typedef coil::Guard<coil::Mutex> Guard;
    DATAPORTSTATUS_ENUM
    
    /*!
     * @if jp
     * @brief 󥹥ȥ饯
     *
     * 󥹥ȥ饯
     *  Publisher ѿåɤ롣
     *
     * @param consumer ǡФԤĥ󥷥塼
     * @param property PublisherζưꤷProperty֥
     *                 (PublisherǤ̤)
     * @else
     * @brief Constructor
     *
     * Constructor
     * Create new thread for this Publisher.
     *
     * @param consumer Consumer that waits for the data sending
     * @param property Property object that is configured this Publisher's
     *                 control information.(Unused in this Publisher)
     * @endif
     */
    PublisherNew();
    
    /*!
     * @if jp
     * @brief ǥȥ饯
     *
     * ǥȥ饯
     *
     * @else
     * @brief Destructor
     *
     * Destructor
     *
     * @endif
     */
    virtual ~PublisherNew(void);

    /*!
     * @if jp
     * @brief 
     * @else
     * @brief initialization
     * @endif
     */
    virtual ReturnCode init(coil::Properties& prop);
    /*!
     * @if jp
     * @brief InPort󥷥塼ޤΥå
     *
     * @else
     * @brief Store InPort consumer
     * @endif
     */
    virtual ReturnCode setConsumer(InPortConsumer* consumer);
    /*!
     * @if jp
     * @brief ХåեΥå
     * @param buffer CDRХåե
     * @return PORT_OK      ｪλ
     *         INVALID_ARGS ʰ
     * @else
     * @brief Setting buffer pointer
     * @param buffer CDR buffer
     * @return PORT_OK      ｪλ
     *         INVALID_ARGS ʰ
     * @endif
     */
    virtual ReturnCode setBuffer(CdrBufferBase* buffer);
    /*!
     * @if jp
     * @brief ꥹʤꤹ롣
     * @param info ConnectorInfo
     * @param listeners ConnectorListeners 
     * @return PORT_OK      ｪλ
     *         INVALID_ARGS ʰ
     * @else
     * @brief Set the listener. 
     * @param info ConnectorInfo
     * @param listeners ConnectorListeners 
     * @return PORT_OK      Normal return
     *         INVALID_ARGS Invalid arguments
     * @endif
     */
    virtual ReturnCode setListener(ConnectorInfo& info,
                                   ConnectorListeners* listeners);
    /*!
     * @if jp
     * @brief ǡ񤭹
     * @param data 񤭹ǡ 
     * @param sec ॢȻ
     * @param nsec ॢȻ
     * @return ꥿󥳡
     * @else
     * @brief Write data 
     * @param data Data
     * @param sec Timeout period
     * @param nsec Timeout period
     * @return Return code
     * @endif
     */
    virtual ReturnCode write(const cdrMemoryStream& data,
                             unsigned long sec,
                             unsigned long usec);
    /*!
     * @if jp
     *
     * @brief ƥֲǧ
     * 
     * ƥֲƤ뤫ǧ롣
     *
     * @return ֳǧ(ƥ־:true󥢥ƥ־:false)
     *
     * @else
     *
     * @brief Confirm to activate
     * 
     * Confirm that has been activated.
     *
     * @return Result of state confirmation
     *         (Active state:true, Inactive state:false)
     *
     * @endif
     */
    virtual bool isActive();
    /*!
     * @if jp
     * @brief ƥֲ
     * @return ꥿󥳡
     *
     * @else
     *
     * @brief activation
     * @return Return code
     *
     * @endif
     */
    virtual ReturnCode activate();
    /*!
     * @if jp
     * @brief 󥢥ƥֲ
     * @return ꥿󥳡
     *
     * @else
     *
     * @brief deactivation
     * @return Return code
     *
     * @endif
     */
    virtual ReturnCode deactivate();
    
    /*!
     * @if jp
     * @brief åɼ¹Դؿ
     *
     * ACE_Task::svc() ΥС饤
     * ХåեΥǡޤǥåɤԵ롣
     *
     * @return ¹Է
     *
     * @else
     * @brief Thread execution function
     *
     * ACE_Task::svc() override function.
     * Make thread stand by until data in the buffer is updated.
     *
     * @return Execution result
     *
     * @endif
     */
    virtual int svc(void);
    
    /*!
     * @if jp
     * @brief 
     *
     * ACE_Task::open() ΥС饤
     *  publisher ѿåɤ롣
     *
     * @param args åѰ(PublisherǤ̤)
     *
     * @return ¹Է
     *
     * @else
     * @brief Task start function
     *
     * ACE_Task::open() override function.
     * Create newly thread for this Publisher.
     *
     * @param args Thread creation arguments (Unused in this Publisher)
     *
     * @return Execution result
     *
     * @endif
     */
    //    virtual int open(void *args);
    
    /*!
     * @if jp
     * @brief λؿ
     *
     * ACE_Task::release() ΥС饤
     * ưե饰falseꤷ Publisher ζưߤ롣
     * ưåɤ֥åƤˤϡ
     * 磱󥳥󥷥塼ޤнƤӽФ礬롣
     *
     * @else
     * @brief Task terminate function
     *
     * ACE_Task::release() override function.
     * Set 2driven flag to false, and terminate this Publisher's operation.
     * However, if the driven thread is blocked, Consumer's send
     * processing may be invoked maximum once.
     *
     * @endif
     */
    //    virtual void release();
    
  protected:
    enum Policy
      {
        ALL,
        FIFO,
        SKIP,
        NEW
      };

    /*!
     * @if jp
     * @brief PushPolicy 
     * @else
     * @brief Setting PushPolicy
     * @endif
     */
    void setPushPolicy(const coil::Properties& prop);

    /*!
     * @if jp
     * @brief Task 
     * @else
     * @brief Setting Task
     * @endif
     */
    bool createTask(const coil::Properties& prop);

    /*!
     * @brief push "all" policy
     */
    ReturnCode pushAll();

    /*!
     * @brief push "fifo" policy
     */
    ReturnCode pushFifo();

    /*!
     * @brief push "skip" policy
     */
    ReturnCode pushSkip();

    /*!
     * @brief push "new" policy
     */
    ReturnCode pushNew();

    /*!
     * @brief return code conversion (BufferStatus -> DataPortStatus)
     *
     *
     */
    ReturnCode convertReturn(BufferStatus::Enum status,
                             const cdrMemoryStream& data);

    /*!
     * @if jp
     * @brief DataPortStatus˽äƥꥹʤΤؿƤӽФ
     * @param status DataPortStatus
     * @param data cdrMemoryStream
     * @return ꥿󥳡
     *
     * @else
     * @brief This method calls the function notified the listener 
     *        according to DataPortStatus.  
     * @param status DataPortStatus
     * @param data cdrMemoryStream
     * @return Return code
     *
     * @endif
     */
    ReturnCode invokeListener(DataPortStatus::Enum status,
                              const cdrMemoryStream& data);
    
    /*!
     * @brief Connector data listener functions
     */
    /*!
     * @if jp
     * @brief ON_BUFFER_WRITEΥꥹʤΤ롣 
     * @param data cdrMemoryStream
     * @else
     * @brief This method is notified to listeners of ON_BUFFER_WRITE. 
     * @param data cdrMemoryStream
     * @endif
     */
    inline void onBufferWrite(const cdrMemoryStream& data)
    {
      m_listeners->
        connectorData_[ON_BUFFER_WRITE].notify(m_profile, data);
    }

    /*!
     * @if jp
     * @brief ON_BUFFER_FULLΥꥹʤΤ롣 
     * @param data cdrMemoryStream
     * @else
     * @brief This method is notified to listeners of ON_BUFFER_FULL. 
     * @param data cdrMemoryStream
     * @endif
     */
    inline void onBufferFull(const cdrMemoryStream& data)
    {
      m_listeners->
        connectorData_[ON_BUFFER_FULL].notify(m_profile, data);
    }

    /*!
     * @if jp
     * @brief ON_BUFFER_WRITE_TIMEOUTΥꥹʤΤ롣 
     * @param data cdrMemoryStream
     * @else
     * @brief This method is notified to listeners of ON_BUFFER_WRITE_TIMEOUT. 
     * @param data cdrMemoryStream
     * @endif
     */
    inline void onBufferWriteTimeout(const cdrMemoryStream& data)
    {
      m_listeners->
        connectorData_[ON_BUFFER_WRITE_TIMEOUT].notify(m_profile, data);
    }

    /*!
     * @if jp
     * @brief ON_BUFFER_OVERWRITEΥꥹʤΤ롣 
     * @param data cdrMemoryStream
     * @else
     * @brief This method is notified to listeners of ON_BUFFER_OVERWRITE.
     * @param data cdrMemoryStream
     * @endif
     */
    inline void onBufferWriteOverwrite(const cdrMemoryStream& data)
    {
      m_listeners->
        connectorData_[ON_BUFFER_OVERWRITE].notify(m_profile, data);
    }

    /*!
     * @if jp
     * @brief ON_BUFFER_READΥꥹʤΤ롣 
     * @param data cdrMemoryStream
     * @else
     * @brief This method is notified to listeners of ON_BUFFER_READ.
     * @param data cdrMemoryStream
     * @endif
     */
    inline void onBufferRead(const cdrMemoryStream& data)
    {
      m_listeners->
        connectorData_[ON_BUFFER_READ].notify(m_profile, data);
    }

    /*!
     * @if jp
     * @brief ON_SENDΥꥹʤΤ롣 
     * @param data cdrMemoryStream
     * @else
     * @brief This method is notified to listeners of ON_SEND. 
     * @param data cdrMemoryStream
     * @endif
     */
    inline void onSend(const cdrMemoryStream& data)
    {
      m_listeners->
        connectorData_[ON_SEND].notify(m_profile, data);
    }

    /*!
     * @if jp
     * @brief ON_RECEIVEDΥꥹʤΤ롣 
     * @param data cdrMemoryStream
     * @else
     * @brief This method is notified to listeners of ON_RECEIVED. 
     * @param data cdrMemoryStream
     * @endif
     */
    inline void onReceived(const cdrMemoryStream& data)
    {
      m_listeners->
        connectorData_[ON_RECEIVED].notify(m_profile, data);
    }

    /*!
     * @if jp
     * @brief ON_RECEIVER_FULLΥꥹʤΤ롣 
     * @param data cdrMemoryStream
     * @else
     * @brief This method is notified to listeners of ON_RECEIVER_FULL. 
     * @param data cdrMemoryStream
     * @endif
     */
    inline void onReceiverFull(const cdrMemoryStream& data)
    {
      m_listeners->
        connectorData_[ON_RECEIVER_FULL].notify(m_profile, data);
    }

    /*!
     * @if jp
     * @brief ON_RECEIVER_TIMEOUTΥꥹʤΤ롣 
     * @param data cdrMemoryStream
     * @else
     * @brief This method is notified to listeners of ON_RECEIVER_TIMEOUT. 
     * @param data cdrMemoryStream
     * @endif
     */
    inline void onReceiverTimeout(const cdrMemoryStream& data)
    {
      m_listeners->
        connectorData_[ON_RECEIVER_TIMEOUT].notify(m_profile, data);
    }

    /*!
     * @if jp
     * @brief ON_RECEIVER_ERRORΥꥹʤΤ롣 
     * @param data cdrMemoryStream
     * @else
     * @brief This method is notified to listeners of ON_RECEIVER_ERROR.
     * @param data cdrMemoryStream
     * @endif
     */
    inline void onReceiverError(const cdrMemoryStream& data)
    {
      m_listeners->
        connectorData_[ON_RECEIVER_ERROR].notify(m_profile, data);
    }

    /*!
     * @brief Connector listener functions
     */
//    inline void onBufferEmpty()
//    {
//      m_listeners->
//        connector_[ON_BUFFER_EMPTY].notify(m_profile);
//    }

//    inline void onBufferReadTimeout()
//    {
//      m_listeners->
//        connector_[ON_BUFFER_READ_TIMEOUT].notify(m_profile);
//    }

//    inline void onSenderEmpty()
//    {
//      m_listeners->
//        connector_[ON_SENDER_EMPTY].notify(m_profile);
//    }

//    inline void onSenderTimeout()
//    {
//      m_listeners->
//        connector_[ON_SENDER_TIMEOUT].notify(m_profile);
//    }

    /*!
     * @if jp
     * @brief ON_SENDER_ERRORΥꥹʤΤ롣 
     * @param data cdrMemoryStream
     * @else
     * @brief This method is notified to listeners of ON_SENDER_ERROR.
     * @param data cdrMemoryStream
     * @endif
     */
    inline void onSenderError()
    {
      m_listeners->
        connector_[ON_SENDER_ERROR].notify(m_profile);
    }


  private:
    Logger rtclog;
    InPortConsumer* m_consumer;
    CdrBufferBase* m_buffer;
    ConnectorInfo m_profile;
    coil::PeriodicTaskBase* m_task;
    ConnectorListeners* m_listeners;
    ReturnCode m_retcode;
    Mutex m_retmutex;
    Policy m_pushPolicy;
    int m_skipn;
    bool m_active;
    int m_leftskip;
  };
};     // namespace RTC

extern "C"
{
  void DLL_EXPORT PublisherNewInit();
};

#endif // RTC_PUBLISHERNEW_H

