// -*- C++ -*-
/*!
 * @file StringUtil.h
 * @brief String operation utility
 * @date $Date: 2007-12-31 03:08:07 $
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2003-2008
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id: StringUtil.h 826 2008-08-26 08:13:39Z n-ando $
 *
 */

#ifndef COIL_STRINGUTIL_H
#define COIL_STRINGUTIL_H

#include <string>
#include <vector>
#include <sstream>

namespace coil
{
  typedef std::vector<std::string> vstring;

  /*!
   * @if jp
   * @brief string  wstring ؤѴ
   *
   * Ϳ줿 string ʸ wstring ʸѴ
   *
   * @param str std::string ʸ
   * @return std::wstring νʸ
   *
   * @else
   * @brief string to wstring conversion
   *
   * This function convert from a string to a wstring.
   * 
   * @param str The input std::string type string
   * @return Converted std::wstring type string
   *
   * @endif
   */
  std::wstring string2wstring(std::string str);

  /*!
   * @if jp
   * @brief wstring  string ؤѴ
   *
   * Ϳ줿 wstring ʸ string ʸѴ
   *
   * @param str std::wstring ʸ
   * @return std::string νʸ
   *
   * @else
   * @brief wstring to string conversion
   *
   * This function convert from a wstring to a string.
   * 
   * @param str The input std::wstring type string
   * @return Converted std::string type string
   *
   * @endif
   */
  std::string wstring2string(std::wstring wstr);

  /*!
   * @if jp
   * @brief ʸؤѴ
   *
   * Ϳ줿ʸʸѴ
   *
   * @param str ʸ
   *
   * @else
   * @brief Uppercase String Transformation
   *
   * This function transforms a given string to uppercase letters
   * 
   * @param str The input string
   *
   * @endif
   */
  void toUpper(std::string& str);

  /*!
   * @if jp
   * @brief ʸؤѴ
   *
   * Ϳ줿ʸʸѴ
   *
   * @param str ʸ
   *
   * @else
   * @brief Lowercase String Transformation
   *
   * This function transforms a given string to lowercase letters
   * 
   * @param str The input string
   *
   * @endif
   */
  void toLower(std::string& str);

  /*!
   * @if jp
   * @brief ϥȥ꡼फ1ɤ߹
   *
   * ϥȥ꡼फ1ɤ߹ࡣ
   * ϥȥ꡼ϤβԥɤUNIX, WindowsβԥɤΤ줫
   * ⤷ϺߤƤƤ褤
   *
   * @param istr ϥȥ꡼
   * @param line ɤ߹ʸǼѿ
   *
   * @return ʸɤ߹ʸĹ
   *
   * @else
   * @brief Read a line from input stream
   *
   * This function reads a line from input stream.
   * UNIX, Windows or mixed line feed code is acceptable.
   *
   * @param istr The input stream.
   * @param line The output variable to store string to be read.
   *
   * @return The length of read string except line feed character.
   *
   * @endif
   */
  int getlinePortable(std::istream& istr, std::string& line);
  
  /*!
   * @if jp
   * @brief ʸ󤬥פƤ뤫ȽǤ
   *
   * ꤵ줿ʸפƤ뤫ɤȽǤ롣
   *
   * @param str פƤ뤫ɤȽǤʸޤʸ
   * @param pos פƤ뤫ɤȽǤʸΰ
   *
   * @return ꤷʸפƤ true, ʳ false
   *
   * @else
   * @brief Check whether the character is escaped or not
   *
   * Check whether the specified character is escaped or not
   *
   * @param str The string that includes the character to be investigated.
   * @param pos The position of the character to be investigated.
   *
   * @return True if the specified character is escaped, else False.
   *
   * @endif
   */
  bool isEscaped(const std::string& str, std::string::size_type pos);
  
  /*!
   * @if jp
   * @brief ʸ򥨥פ
   *
   * ʸ򥨥ץ󥹤Ѵ롣<br>
   * HT -> "\t" <br>
   * LF -> "\n" <br>
   * CR -> "\r" <br>
   * FF -> "\f" <br>
   * 󥰥륯ȡ֥륯ȤˤĤƤϤȤ˽Ϥʤ
   *
   * @param str ׽оʸ
   *
   * @return ׽ʸ
   *
   * @else
   *
   * @brief Escape string
   *
   * The following characters are converted. <br>
   * HT -> "\t" <br>
   * LF -> "\n" <br>
   * CR -> "\r" <br>
   * FF -> "\f" <br>
   * Single quote and double quote are not processed.
   *
   * @param str The target string for the escape
   *
   * @return Result string of the escape
   *
   * @endif
   */
  std::string escape(const std::string str);
  
  /*!
   * @if jp
   * @brief ʸΥפ᤹
   *
   * Υץ󥹤ʸѴ롣<br>
   * "\t" -> HT <br>
   * "\n" -> LF <br>
   * "\r" -> CR <br>
   * "\f" -> FF <br>
   * "\"" -> "  <br>
   * "\'" -> '  <br>
   * ׽δʵѴˤϤʤäƤʤᡢդɬס
   *
   * @param str 󥨥׽оʸ
   *
   * @return 󥨥׽ʸ
   *
   * @else
   *
   * @brief Unescape string
   *
   * The following characters are converted. <br>
   * "\t" -> HT <br>
   * "\n" -> LF <br>
   * "\r" -> CR <br>
   * "\f" -> FF <br>
   * "\"" -> "  <br>
   * "\'" -> '  <br>
   * Note: This is not complete inversion of the escape processing.
   *
   * @param str The target string for the unescape
   *
   * @return Result string of the unescape
   *
   * @endif
   */
  std::string unescape(const std::string str);

  /*!
   * @if jp
   * @brief ʸζʸ
   *
   * Ϳ줿ʸζʸ롣
   * ʸȤưΤ' '(ڡ)'\\t'()
   *
   * @param str ʸʸ
   *
   * @else
   * @brief Erase blank characters of string
   *
   * Erase blank characters that exist at the head of the given string.
   * Space ' 'and tab '\\t' are supported as the blank character.
   *
   * @param str The target blank characters of string for the erase
   *
   * @endif
   */
  void eraseBlank(std::string& str);

  /*!
   * @if jp
   * @brief ʸƬζʸ
   *
   * Ϳ줿ʸƬ¸ߤʸ롣
   * ʸȤưΤ' '(ڡ)'\\t'()
   *
   * @param str Ƭʸʸ
   *
   * @else
   * @brief Erase the head blank characters of string
   *
   * Erase the blank characters that exist at the head of the given string.
   * Space ' 'and tab '\\t' are supported as the blank character.
   *
   * @param str The target head blank characters of string for the erase
   *
   * @endif
   */
  void eraseHeadBlank(std::string& str);
  
  /*!
   * @if jp
   * @brief ʸζʸ
   *
   * Ϳ줿ʸ¸ߤʸ롣
   * ʸȤưΤ' '(ڡ)'\\t'()
   *
   * @param str ʸʸ
   *
   * @else
   * @brief Erase the tail blank characters of string
   *
   * Erase the blank characters that exist at the tail of the given
   * string.  Space ' 'and tab '\\t' are supported as the blank
   * character.
   *
   * @param str The target tail blank characters of string for the erase
   *
   * @endif
   */
  void eraseTailBlank(std::string& str);

  /*!
   * @if jp
   * @brief ʸƬζʸ
   *
   * Ϳ줿ʸƬ¸ߤʸ롣
   * ʸȤưΤ' '(ڡ)'\\t'()
   *
   * @param str Ƭʸʸ
   *
   * @else
   * @brief Erase the head blank and the tail blank characters of string
   *
   * Erase the head blank characters and the blank characters that
   * exist at the tail of the given string.  Space ' 'and tab '\\t'
   * are supported as the blank character.
   *
   * @param str The target tail blank characters of string for the erase
   *
   * @endif
   */
  void eraseBothEndsBlank(std::string& str);

  /*!
   * @if jp
   * @brief ʸΤ
   *
   * Ϳ줿ʸƬ¸ߤʸ
   * ѻ򤹤٤ƾʸѴ롣
   *
   * @param str оʸ
   *
   * @else
   * @brief Erase the head/tail blank and replace upper case to lower case
   *
   * Erase the head blank characters and the blank characters that
   * exist at the tail of the given string.  Space ' 'and tab '\\t'
   * are supported as the blank character.
   * And all upper case cahracters are converted into lower case.
   *
   * @param str The target string for the erase
   *
   * @endif
   */
  std::string normalize(std::string& str);
  
  /*!
   * @if jp
   * @brief ʸ֤
   *
   * Ϳ줿ʸФơꤷʸ֤Ԥ
   *
   * @param str ֤оʸ
   * @param from ִʸ
   * @param to ִʸ
   *
   * @else
   * @brief Replace string
   *
   * Replace the given string with the specified characters.
   *
   * @param str The target characters of string for replacement processing
   * @param from Characters of replacement source
   * @param to Characters of replacement destination
   *
   * @endif
   */
  void replaceString(std::string& str, const std::string from,
                     const std::string to);
  
  /*!
   * @if jp
   * @brief ʸʬʸʬ䤹
   * 
   * ꤵ줿ʸͿ줿ǥߥʬ䤹롣
   *
   * @param input ʬоʸ
   * @param delimiter ʬʸ(ǥߥ)
   *
   * @return ʸʬ̥ꥹ
   *
   * @else
   * @brief Split string by delimiter
   * 
   * Split the set string by the given delimiter
   *
   * @param input The target characters of string for split
   * @param delimiter Split string (delimiter)
   *
   * @return Split string result list
   *
   * @endif
   */
  vstring split(const std::string& input,
                const std::string& delimiter,
                bool ignore_empty = false);
  
  /*!
   * @if jp
   * @brief Ϳ줿ʸboolͤѴ
   * 
   * ꤵ줿ʸtrueɽʸfalseɽʸӤη̤
   * boolͤȤ֤
   * Ӥη̡trueɽʸfalseɽʸΤɤȤפʤϡ
   * Ϳ줿ǥե֤ͤ
   *
   * @param str Ƚоʸ
   * @param yes trueɽʸ
   * @param no falseɽʸ
   * @param default_value ǥե(ǥե:true)
   * @else
   * @brief Convert given string into bool value
   * 
   * Compare the specified string with the true representation string and
   * the false representation string, and return the result as bool value.
   * If it matches neither the true representation string nor the false
   * representation string as a result of the comparison, the given default
   * value will be return.
   *
   * @param str The target string for investigation
   * @param yes The true representation string
   * @param no The false representation string
   * @param default_value The default value (The default value:true)
   * @endif
   */
  bool toBool(std::string str, std::string yes, std::string no, 
              bool default_value = true);
  /*!
   * @if jp
   * @brief ʸꥹˤʸ󤬴ޤޤ뤫ɤ
   * 
   * 1˥޶ڤΥꥹȤ2õоʸꤷ
   * ʸ1˴ޤޤ뤫ȽǤ롣
   *
   * @param list оݥꥹ
   * @param value õʸ
   * @return true: ޤޤ롢false: ޤޤʤ
   *
   * @else
   * @brief Include if a string is included in string list
   * 
   * if the second argument is included in the comma separated string
   * list of the first argument, This operation returns "true value".
   *
   * @param list The target comma separated string
   * @param value The searched string
   * @return true: included, false: not included
   *
   * @endif
   */
  bool includes(const vstring& list, std::string value,
                bool ignore_case = true);
  bool includes(const std::string& list, std::string value,
                bool ignore_case = true);
  
  /*!
   * @if jp
   * @brief Ϳ줿ʸХѥɤȽǤ
   *
   * Ϳ줿ʸХѥɽǤ뤫ɤȽǤ롣
   * ʸ󤬰ʲξˤХѥȤȽǤ롣
   *  - Ƭʸ'/' (UNIXξ)
   *  - Ƭʸե٥åȡ'/''\\' (Windowsξ)
   *  - Ƭʸ'\\\\' (Windowsͥåȥѥξ)
   *
   * @param str Ƚоʸ
   *
   * @return ХѥȽ
   *
   * @else
   * @brief Investigate whether the given string is absolute path or not
   *
   * Investigate whether the given string is absolute path or not.
   * Investigate it as an absolute path, if the string is as follows:
   *  - The first character '/' (UNIX)
   *  - The first 3 characters are alphabet '/''\\' (Windows)
   *  - The first 2 characters are '\\\\' (Windows network path)
   *
   * @param str The target string for the investigation
   *
   * @return Investigation result of absolute path
   *
   * @endif
   */
  bool isAbsolutePath(const std::string& str);
  
  /*!
   * @if jp
   * @brief Ϳ줿ʸURLɤȽǤ
   *
   * Ϳ줿ʸURLɽɤȽǤ롣
   * Ϳ줿ʸˡ'://'Ȥʸ󤬴ޤޤƤˤ
   * URLɽȤȽǤ롣
   *
   * @param str Ƚоʸ
   *
   * @return URLȽ
   *
   * @else
   * @brief Investigate whether the given string is URL or not
   *
   * Investigate whether the given string is URL or not.
   * When the string '://' is included in the given character string,
   * make it of URL representation.
   *
   * @param str The target string for investigation
   *
   * @return URL investigation result
   *
   * @endif
   */
  bool isURL(const std::string& str);
  
  /*!
   * @if jp
   * @brief Ϳ줿֥Ȥstd::stringѴ
   *
   * ǻꤵ줿֥ȤʸѴ롣
   *
   * @param n Ѵоݥ֥
   *
   * @return ʸѴ
   *
   * @else
   * @brief Convert the given object to std::string
   *
   * Convert the object specified by the argument to string.
   *
   * @param n The target object for conversion
   *
   * @return String conversion result
   *
   * @endif
   */
  template <class Printable>
  std::string otos(Printable n)
  {
    std::stringstream str_stream;
    str_stream << n;
    return str_stream.str();
  };
  
  /*!
   * @if jp
   * @brief Ϳ줿std::string򥪥֥ȤѴ
   *
   * Ϳ줿ʸꤵ줿֥ȤѴ롣
   *
   * @param val Ѵ襪֥
   * @param str Ѵʸ
   *
   * @return Ѵ¹Է
   *
   * @else
   * @brief Convert the given std::string to object.
   *
   * Convert string given by the argument to specified object.
   *
   * @param val The object of conversion destination
   * @param str String of conversion source
   *
   * @return Conversion processing result
   *
   * @endif
   */
  template <typename To>
  bool stringTo(To& val, const char* str)
  {
    if (str == 0) { return false; }

    std::stringstream s;
    if ((s << str).fail()) { return false; }
    if ((s >> val).fail()) { return false; }
    return true;
  }
  
  template<>
  bool stringTo<std::string>(std::string& val, const char* str);

  /*!
   * @if jp
   * @brief Ϳ줿ʸꥹȤʣ
   *
   * Ϳ줿ʸꥹȤʣꥹȤ롣
   *
   * @param sv ǧʸꥹ
   *
   * @return ʣ̥ꥹ
   *
   * @else
   * @brief Eliminate duplication from the given string list
   *
   * Create a list of eliminating duplication from the string list
   * given by the argument.
   *
   * @param sv The string list for confirmation source
   *
   * @return Eliminating duplication result list
   *
   * @endif
   */
  vstring unique_sv(vstring sv);
  
  /*!
   * @if jp
   * @brief Ϳ줿ʸꥹȤCSV
   *
   * Ϳ줿ʸꥹȤγǤ¤٤CSV롣
   * ʸꥹȤξˤ϶ʸ֤
   *
   * @param sv CSVѴоʸꥹ
   *
   * @return CSVѴʸ
   *
   * @else
   * @brief Create CSV file from the given string list
   *
   * Create CSV that arranged each element of the character string
   * list given by the argument.
   * If the string list is empty, the null will be returned.
   *
   * @param sv The target string list for creating CSV
   *
   * @return String of CSV creating result
   *
   * @endif
   */
  std::string flatten(vstring sv);
  
  /*!
   * @if jp
   * @brief Ϳ줿ʸꥹȤꥹȤѴ
   *
   * Ϳ줿ʸꥹȤγ'\\0'äꥹ
   * Ѵ롣
   *
   * @param args Ѵоʸꥹ
   *
   * @return Ѵʸ
   *
   * @else
   * @brief Convert the given string list into the argument list
   *
   * Convert the string list into the argument list by adding '\\0' to
   * each element at the end of the string list given by the argument
   *
   * @param args The target string list for conversion
   *
   * @return String of argument conversion result
   *
   * @endif
   */
  char** toArgv(const vstring& args); 


  std::string sprintf(char const * __restrict fmt, ...);
  
}; // namepsace coil
#endif // COIL_STRINGUTIL_H
