package littlemaid.inventory;

import littlemaid.LittleMaidCore;
import littlemaid.entity.EntityLittleMaid;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.IInventory;
import net.minecraft.item.ItemArmor;
import net.minecraft.item.ItemStack;
import net.minecraft.util.ChatComponentTranslation;
import net.minecraft.util.IChatComponent;

public class InventoryLittleMaid implements IInventory {

	/** An array of 18 item stacks indicating the maid inventory. */
	private ItemStack[] mainInventory = new ItemStack[18];
	/** An array of 4 item stacks containing the currently worn armor pieces. */
	private ItemStack[] armorInventory = new ItemStack[4];
	/** The maid whose inventory this is. */
	private EntityLittleMaid maid;
	/** インベントリの変更を検知 */
    private boolean inventoryChanged;

	public InventoryLittleMaid(EntityLittleMaid maidIn) {
		this.maid = maidIn;
	}

	@Override
	public String getName() {
		return LittleMaidCore.MODID + ".inventory.insideSkirt";
	}

	@Override
	public boolean hasCustomName() {
		return false;
	}

	@Override
	public IChatComponent getDisplayName() {
		return new ChatComponentTranslation(this.getName(), new Object[0]);
	}

	@Override
	public int getSizeInventory() {
		return this.mainInventory.length + this.armorInventory.length;
	}

	/**
	 * アイテムを取得する。
	 * @param index アイテムスロット番号 0-17:インベントリ、18-21:防具（頭、胴、レギンス、靴）
	 * @return 指定されたスロットのItemStackインスタンス
	 */
	@Override
	public ItemStack getStackInSlot(int index) {
		ItemStack[] aitemstack = this.mainInventory;

		// 0-17:インベントリ、18-21:防具（18は使わない？）
		if (index >= aitemstack.length)
		{
			index -= aitemstack.length;
			aitemstack = this.armorInventory;
		}
		return aitemstack[index];
	}

	/**
	 * アイテムのスタック数を減らす。
	 * @param index アイテムスロット番号
	 * @param count 減らす数量
	 * @return 減らしたアイテムと数量が設定されたItemStackインスタンス。アイテムがない場合はnull。
	 */
	@Override
	public ItemStack decrStackSize(int index, int count) {
		ItemStack[] aitemstack = this.mainInventory;

		if (index >= aitemstack.length)
		{
			index -= aitemstack.length;
			aitemstack = this.armorInventory;
		}

		if (aitemstack[index] != null)
		{
			ItemStack itemstack;

			if (aitemstack[index].stackSize <= count)
			{
				itemstack = aitemstack[index];
				aitemstack[index] = null;
			}
			else
			{
				itemstack = aitemstack[index].splitStack(count);
			}

			return itemstack;
		}
		else
		{
			return null;
		}
	}

	/**
	 * インベントリを閉じた時にドロップするアイテムを取得する。
	 * ワークベンチとか、途中で閉じた場合にドバっと出すやつ。
	 * @param index アイテムスロット番号
	 * @return ドロップするアイテムと数量が入ったItemStackインスタンス。ドロップがない場合はnullでいいぽい。
	 */
	@Override
	public ItemStack getStackInSlotOnClosing(int index) {
		return null;
	}

	/**
	 * アイテムスロットにアイテムを設定する。
	 * @param index アイテムスロット番号
	 * @param stack 設定するItemStackインスタンス。
	 */
	@Override
	public void setInventorySlotContents(int index, ItemStack stack) {
		ItemStack[] aitemstack = this.mainInventory;

		if (index >= aitemstack.length)
		{
			index -= aitemstack.length;
			aitemstack = this.armorInventory;
		}
		aitemstack[index] = stack;
	}

	@Override
	public void clear() {
		for (int i = 0; i < this.mainInventory.length; i++){
			this.mainInventory[i] = null;
		}
		for (int i = 0; i < this.armorInventory.length; i++){
			this.armorInventory[i] = null;
		}
	}

	@Override
	public int getInventoryStackLimit() {
		return 64;
	}

	@Override
	public void markDirty() {
		// TileEntityを継承するインスタンスを内包している場合、そのmarkDirty()を呼び出す。
	}

	@Override
	public boolean isUseableByPlayer(EntityPlayer player) {
		// このインベントリをプレイヤーがつつけるかどうか
        return this.maid.isDead ? false : player.getDistanceSqToEntity(this.maid) <= 64.0D;
	}

	@Override
	public void openInventory(EntityPlayer player) {
		//LittleMaidCore.getLogger().info("InventoryLittleMaid_openInventory");
	}

	@Override
	public void closeInventory(EntityPlayer player) {
		//LittleMaidCore.getLogger().info("InventoryLittleMaid_closeInventory");

		// 装備品を同期する
		this.maid.syncEquip();
	}

    /**
     * Damages armor in each slot by the specified amount.
     */
    public void damageArmor(float amount)
    {
    	amount /= 4.0F;

        if (amount < 1.0F)
        {
        	amount = 1.0F;
        }

        for (int i = 0; i < this.armorInventory.length; ++i)
        {
            if (this.armorInventory[i] != null && this.armorInventory[i].getItem() instanceof ItemArmor)
            {
                this.armorInventory[i].damageItem((int)amount, this.maid);

                if (this.armorInventory[i].stackSize == 0)
                {
                    this.armorInventory[i] = null;
                }
            }
        }
    }

    /**
     * スロットによって入れられるアイテムが限定されるときに使う？
     */
	@Override
	public boolean isItemValidForSlot(int index, ItemStack stack) {
		return true;
	}

	/**
	 * インベントリが保持しておくべき情報を出し入れするためのところ
	 * ex.かまどの燃焼残り時間とか、錬成時間とか
	 */
	@Override
	public int getField(int id) {
		return 0;
	}

	@Override
	public void setField(int id, int value) {}

	@Override
	public int getFieldCount() {
		return 0;
	}

}
