/* Yet another re-write of the show-the-character-list standard:
// But this one shows the characters embedded in #defines.
// Prints the characters of the old Japanese 8-bit standard including 8-bit katakana.
// Began September 2000, Takino, Kato, Hyougo, Japan, by Joel Matthew Rees.
//    joel_rees@sannet.ne.jp
// Finally found time to add the kana and kana punctuation in December 2000 and January 2001.
//
-- What I really wanted first was a program to print out the values of all valid characters 
-- in the 8-bit range, in a readable form.
-- Second was a program to print out names that can be compiled as constants or whatever.
-- Third was a program to print out a list of defines or enumerations or constants.
-- And then I wanted to repeat the steps in the 16-bit range.
-- But there is no time for me to do that.
//
// Copyright 2000, 2001 Joel Matthew Rees.
//   All rights reserved.
//
// Assignment of Stewardship, or Terms of Use: 
//
// The author grants permission to use and/or redistribute the code in this 
// file, in either source or translated form, under the following conditions:
// 1. When redistributing the source code, the copyright notices and terms of 
//    use must be neither removed nor modified. 
// 2. When redistributing in a form not generally read by humans, the 
//    copyright notices and terms of use, with proper indication of elements 
//    covered, must be reproduced in the accompanying documentation and/or 
//    other materials provided with the redistribution. In addition, if the 
//    source includes statements designed to compile a copyright notice 
//    into the output object code, the redistributor is required to take 
//    such steps as necessary to preserve the notice in the translated 
//    object code.
// 3. Modifications must be annotated, with attribution, including the name(s) 
//    of the author(s) and the contributor(s) thereof, the conditions for 
//    distribution of the modification, and full indication of the date(s) 
//    and scope of the modification. Rights to the modification itself 
//    shall necessarily be retained by the author(s) thereof.
// 4. These grants shall not be construed as an assignment or assumption of 
//    liability of any sort or to any degree. Neither shall these grants be 
//    construed as endorsement or represented as such. Any party using this 
//    code in any way does so under the agreement to entirely indemnify the 
//    author and any contributors concerning the code and any use thereof. 
//    Specifically, THIS SOFTWARE IS PROVIDED AT NO COST, AS IT IS, WITHOUT 
//    ANY EXPRESS OR IMPLIED WARRANTY OF ANY SORT, INCLUDING, BUT NOT LIMITED 
//    TO, WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
//    UNDER NO CIRCUMSTANCES SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR 
//    ANY DAMAGES WHATSOEVER ARISING FROM ITS USE OR MISUSE, EVEN IF ADVISED 
//    OF THE EXISTENCE OF THE POSSIBILITY OF SUCH DAMAGE.
// 5. This code should not be used for any illegal or immoral purpose, 
//    including, but not limited to, the theft of property or services, 
//    deliberate communication of false information, the distribution of drugs 
//    for purposes other than medical, the distribution of pornography, the 
//    provision of illicit sexual services, the maintenance of oppressive 
//    governments or organizations, or the imposture of false religion and 
//    false science. 
//    Any illegal or immoral use incurs natural and legal penalties, which the 
//    author invokes in full force upon the heads of those who so use it.
// 6. Alternative redistribution arrangements:
//    a. If the above conditions are unacceptable, redistribution under the 
//       following commonly used public licenses is expressly permitted:
//       i.   The GNU General Public License (GPL) of the Free Software 
//            Foundation.
//       ii.  The Perl Artistic License, only as a part of Perl.
//       iii. The Apple Public Source License, only as a part of Darwin or 
//            a Macintosh Operating System using Darwin.
//    b. No other alternative redistribution arrangement is permitted.
//       (The original author reserves the right to add to this list.)
//    c. When redistributing this code under an alternative license, the 
//       specific license being invoked shall be noted immediately beneath 
//       the body of the terms of use. The terms of the license so specified 
//       shall apply only to the redistribution of the source so noted. 
// 7. In no case shall the rights of the original author to the original work 
//    be impaired by any distribution or redistribution arrangement.
//
// End of the Assignment of Stewardship, or terms of use.
//
// License invoked: Assignment of Stewardship.
// Notes concerning license:
//    Compiler directives are strongly encouraged as a means of meeting 
//    the attribution requirements in the Assignment of Stewardship.
*/

/* One must be careful to make the mental distinction between the concepts of "hankaku" and 
// 8-bit. "Hankaku" (or half-width) should be a matter of choice of font, 
// but the Japanese have a tradition that dates back to around 1970 that fails 
// to properly distinguish between bit-width and displayed glyph width in this sense. 
// This is partly due to the tendency they have of adjusting the size of characters to fit 
// the space, rather than adjusting the space to fit the characters, which can probably be 
// meaningfully ascribed to the influence of the non-linear flow of their writing system. 
//
// The proper solution to the half-width issue is to be able to select a peace of text and 
// squeeze it in a particular direction, essentially generating a new font rendering on-the-fly. 
// But I am guessing that getting the Japanese to give up their hankaku will be as easy as 
// convincing them that the really do, after all, write by the radical.
// 
// This is not the place to preach radicals, either.
*/

#include <stdio.h>
#include <stdlib.h>


/* A tautology seems nice, but I think I prefer 
// to show and at least some of the steps by not including "sjisranges.h".
// Come to think of it, it would probably be a good thing 
// to make this as stand-alone as possible, thus, define ubute directly.
*/


#include <limits.h>


#if CHAR_BIT != 8
#	error "Need to adjust for variant definition of char!"
/* Nothing I can think of to encapsulate this issue until C provides something like 
// typedef unsigned sjisByte: 8;
*/
#endif


/* This will conflict with some poorly encapsulated programming environments.
// (Pre-compiled clots of headers probably do not promote clean programming habits, btw.)
*/
typedef unsigned char ubyte;
typedef signed char sbyte;


#define b7_ASCIIPrintLo	'\x20'	/* ' ' */
#define b7_punct1Lo	'\x21'	/* '!' */
#define b7_punct1Hi	'\x2f'	/* '/' */
#define b7_digitLo	'\x30'	/* '0' */
#define b7_digitHi	'\x39'	/* '9' */
#define b7_punct2Lo	'\x3a'	/* ':' */
#define b7_punct2Hi	'\x40'	/* '@' */
#define b7_upperLo	'\x41'	/* 'A' */
#define b7_upperHi	'\x5a'	/* 'Z' */
#define b7_punct3Lo	'\x5b'	/* '[' */
#define b7_punct3Hi	'\x60'	/* '`' */
#define b7_lowerLo	'\x61'	/* 'a' */
#define b7_lowerHi	'\x7a'	/* 'z' */
#define b7_punct4Lo	'\x7b'	/* '{' */
#define b7_punct4Hi	'\x7e'	/* '~' */
#define b7_ASCIIPrintHi	b7_punct4Hi
#define b7_ASCIIHi	'\x7f'	/* delete */


#define b8_GAP1low	( (ubyte) '\x80' )	/* bottom of the first gap */

/* There are no traditional 8-bit hiragana in SJIS.
// We must be careful to distinguish between 8-bit and hankaku (half-width).
// The duplication of certain characters in the 8-bit and 16-bit "ranges" has 
// helped the proliferation of the unfortunate practice of giving half-width 
// and full-width characters separate codes. 
// (Width is properly a function of typeface/font.)
// For the katakana, the 8-bit codes were naturally usurped as the half-width codes.
*/
#define b8_JISLo	( (ubyte) '\xa1' )	/* '' (kuten) */
#define b8_JISHi	( (ubyte) '\xdf' )	/* '' (handakuten) */
/* */
#define b8_JISPunctLo	( (ubyte) '\xa1' )	/* '' (kuten) */
#define b8_JISPunctHi	( (ubyte) '\xa5' )	/* '' (chuten) */
/* */
#define b8_katakanaLo	( (ubyte) '\xa6' )	/* '' (wo) */
#define b8_katakanaHi	( (ubyte) '\xdf' )	/* '' (handakuten) */
/* */
#define b8_katakana50Lo	( (ubyte) '\xa6' )	/* '' (wo) (The theoretically 50 syllables.) */
#define b8_katakana50Hi	( (ubyte) '\xdd' )	/* '' ('n) */
/* */
#define b8_katakanaSubLo	( (ubyte) '\xa7' )	/* '' (a) */
#define b8_katakanaSubHi	( (ubyte) '\xaf' )	/* '' (tu) */
/* */


#include "port.h"


/* #define INTERMEDIATE_STUFF */


typedef struct charName_s
{	char * name;
	char * comment;
} charName_s;


charName_s charNames[] =
{	{	"b7_NUL",	"0x00: Null character"	},
	{	"b7_SOH",	"0x01: start of heading"	},
	{	"b7_STX",	"0x02: start of text"	},
	{	"b7_ETX",	"0x03: end of text"	},
	{	"b7_EOT",	"0x04: end of transmission"	},
	{	"b7_ENQ",	"0x05: enquiry"	},
	{	"b7_ACK",	"0x06: acknowledge"	},
	{	"b7_BEL",	"0x07: ring that bell"	},
	{	"b7_BS",	"0x08: backspace, usually destructive"	},
	{	"b7_HT",	"0x09: horizontal tabulation"	},
	{	"b7_LF",	"0x0a: line feed in output"	},
	{	"b7_VT",	"0x0b: vertical tabulation"	},
	{	"b7_FF",	"0x0c: form (page) feed in output"	},
	{	"b7_CR",	"0x0d: carriage return"	},
	{	"b7_SO",	"0x0e: shift out"	},
	{	"b7_SI",	"0x0f: shift in"	},
	{	"b7_DLE",	"0x10: data link escape"	},
	{	"b7_DC1",	"0x11: device control 1"	},
	{	"b7_DC2",	"0x12: device control 2"	},
	{	"b7_DC3",	"0x13: device control 3"	},
	{	"b7_DC4",	"0x14: device control 4"	},
	{	"b7_NAK",	"0x15: negate acknowledge"	},
	{	"b7_SYN",	"0x16: synchronous idle"	},
	{	"b7_ETB",	"0x17: end of transmission block"	},
	{	"b7_CAN",	"0x18: cancel"	},
	{	"b7_EM",	"0x19: end of medium"	},
	{	"b7_SUB",	"0x1a: substitute character"	},
	{	"b7_ESC",	"0x1b: escape character"	},
	{	"b7_FS",	"0x1c: file separator"	},
	{	"b7_GS",	"0x1d: group separator"	},
	{	"b7_RS",	"0x1e: record separator"	},
	{	"b7_US",	"0x1f: unit separator"	},
	{	"b7_SP", 	"0x20: (for visibility)"	},
	{	"b7_EXCLAIM",	""	},
	{	"b7_SECONDS",	"also inches or non-directed double-quote"	},
	{	"b7_HASH",	"also pound or sharp"	},
	{	"b7_DOLLAR",	""	},
	{	"b7_PERCENT",	""	},
	{	"b7_AMPERSAND",	""	},
	{	"b7_MINUTES",	"also feet, or non-directed/right single-quote, or acute accent"	},
	{	"b7_LEFTPAREN",	""	},
	{	"b7_RIGHTPAREN",	""	},
	{	"b7_ASTERISK",	""	},
	{	"b7_PLUS",	""	},
	{	"b7_COMMA",	""	},
	{	"b7_HYPHEN",	"also minus or dash"	},
	{	"b7_PERIOD",	""	},
	{	"b7_SLASH",	"Did they call this terminus? (No, and I don't care!)"	},
	{	"b7_ZERO",	""	},
	{	"b7_ONE",	""	},
	{	"b7_TWO",	""	},
	{	"b7_THREE",	""	},
	{	"b7_FOUR",	""	},
	{	"b7_FIVE",	""	},
	{	"b7_SIX",	""	},
	{	"b7_SEVEN",	""	},
	{	"b7_EIGHT",	""	},
	{	"b7_NINE",	""	},
	{	"b7_COLON",	""	},
	{	"b7_SEMICOLON",	""	},
	{	"b7_LESSTHAN",	"also left angle bracket"	},
	{	"b7_EQUAL",	"but which equal?"	},
	{	"b7_GREATERTHAN",	"also right angle bracket"	},
	{	"b7_QUESTIONMARK",	""	},
	{	"b7_ATEACH",	"at or each"	},
	{	"b7_A",	""	},
	{	"b7_B",	""	},
	{	"b7_C",	"(upper)"	},
	{	"b7_D",	""	},
	{	"b7_E",	""	},
	{	"b7_F",	""	},
	{	"b7_G",	""	},
	{	"b7_H",	""	},
	{	"b7_I",	"(upper case i)"	},
	{	"b7_J",	""	},
	{	"b7_K",	""	},
	{	"b7_L",	"(upper case L)"	},
	{	"b7_M",	""	},
	{	"b7_N",	""	},
	{	"b7_O",	"(upper case oh)"	},
	{	"b7_P",	"(upper)"	},
	{	"b7_Q",	""	},
	{	"b7_R",	""	},
	{	"b7_S",	"(upper)"	},
	{	"b7_T",	""	},
	{	"b7_U",	"(upper)"	},
	{	"b7_V",	"(upper)"	},
	{	"b7_W",	"(upper)"	},
	{	"b7_X",	"(upper)"	},
	{	"b7_Y",	"(upper)"	},
	{	"b7_Z",	"(upper)"	},
	{	"b7_LEFTBRACKET",	""	},
	{	"b7_BACKSLASH",	"also the UNIX, C line/char escape character, Japanese yen"	},
	{	"b7_RIGHTBRACKET",	""	},
	{	"b7_CARET",	""	},
	{	"b7_UNDERSCORE",	""	},
	{	"b7_ACCENTGRAVE",	"or the left single quote"	},
	{	"b7_a",	""	},
	{	"b7_b",	""	},
	{	"b7_c",	"(lower)"	},
	{	"b7_d",	""	},
	{	"b7_e",	""	},
	{	"b7_f",	""	},
	{	"b7_g",	""	},
	{	"b7_h",	""	},
	{	"b7_i",	"(lower case i)"	},
	{	"b7_j",	""	},
	{	"b7_k",	""	},
	{	"b7_l",	"(lower case L)"	},
	{	"b7_m",	""	},
	{	"b7_n",	""	},
	{	"b7_o",	"(lower case oh)"	},
	{	"b7_p",	"(lower)"	},
	{	"b7_q",	""	},
	{	"b7_r",	""	},
	{	"b7_s",	"(lower)"	},
	{	"b7_t",	""	},
	{	"b7_u",	"(lower)"	},
	{	"b7_v",	"(lower)(lower)"	},
	{	"b7_w",	""	},
	{	"b7_x",	"(lower)"	},
	{	"b7_y",	"(lower)"	},
	{	"b7_z",	"(lower)"	},
	{	"b7_LEFTBRACE",	""	},
	{	"b7_VERTICALBAR",	""	},
	{	"b7_RIGHTBRACE",	""	},
	{	"b7_TILDE",	"and sometimes IBM overscore"	},
	{	"b7_DEL",	"The infamous delete character that is not backspace."	}
};

char kanaVowels[] = "aiueo";	/* Make sure there are 5 in each list. */
char yKana8Vowels[] = "a\0u\0o";	/* At some point in history, certain sounds disappeared. */
typedef struct kana8Syllable_s
{	char consonant;
	char * vowelList;
} kana8Syllable_s;
kana8Syllable_s kanaSyllables[] = 
{	{	'w', "\0\0\0\0o"	},	/* Not really a subscripting kana, just out of position. */
	{	'\0', kanaVowels	},	/* subscripting post-fix modifiers */
	{	'y', yKana8Vowels	},	/* subscripting post-fix modifiers */
	{	't', "\0\0u\0\0"	},	/* pre-fixed glottal stop modifier */
	{	'\0', kanaVowels	},	/* the actual "50" sounds from here */
	{	'k', kanaVowels	},
	{	's', kanaVowels	},
	{	't', kanaVowels	},
	{	'n', kanaVowels	},
	{	'h', kanaVowels	},
	{	'm', kanaVowels	},
	{	'y', yKana8Vowels	},
	{	'r', kanaVowels	},
	{	'w', "a\0\0\0\0"	},	/* Why did they put wo in front in JIS X 209-1276? */
	{	'n', "g\0\0\0\0"	},	/* The nasal fits in algorithmically, if not meaningfully. */
	{	'\0', NULL	}
};


int main( int argc, char * argv[] )
{	int ch;
	int vowel, consonant;
/* */
	commandLine( &argc, &argv );
	for ( ch = 0; ch <= b7_ASCIIHi; ++ch )
	{
		printf( "#define\t%s\t", charNames[ ch ].name );
		if ( ch < b7_ASCIIPrintLo || ch > b7_ASCIIPrintHi )
			printf( "'\\x%02x'\t/* %s */\n", ch, charNames[ ch ].comment );
		else if ( ch == b7_ASCIIPrintLo )	/* '\x20'	 ' ' */
			printf( "'\\x%02x'\t/* '%c'==%s */\n", ch, ch, charNames[ ch ].comment );
		else 
		{	if ( ch == '\'' || ch == '\\' )
				printf( "'\\%c'", ch );
			else
				printf( "'%c'", ch );
			if ( charNames[ ch ].comment[ 0 ] != '\0' )
				printf( "\t/* %s */", charNames[ ch ].comment );
			putchar( '\n' );
		}
	}
	putchar( '\n' );

	/* This is also a kludge. 
	// But I don't seem to have time to go back and print without the #defines, either.
	*/
	printf( "#define\tb8_Kuten\t( (ubyte) '\\x%02x' )\t/* '%c' (like period) */\n", 
			0xa1, 0xa1 );
	printf( "#define\tb8_LKagiKakko\t( (ubyte) '\\x%02x' )\t/* '%c' (left \"key\" bracket) */\n", 
			0xa2, 0xa2 );
	printf( "#define\tb8_RKagiKakko\t( (ubyte) '\\x%02x' )\t/* '%c' (right \"key\" bracket) */\n",
			0xa3, 0xa3 );
	printf( "#define\tb8_ToTen\t( (ubyte) '\\x%02x' )\t/* '%c' (like comma) */\n",
			0xa4, 0xa4 );
	printf( "#define\tb8_ChuTen\t( (ubyte) '\\x%02x' )\t/* '%c' (central dot, but used sort of like hyphen and/or slash) */\n",
			0xa5, 0xa5 );
	ch = b8_katakana50Lo;
	for ( consonant = 0; 
		  kanaSyllables[ consonant ].vowelList != NULL && ch <= b8_katakana50Hi; 
		  ++consonant )
	{	for ( vowel = 0; vowel < 5 && ch <= b8_katakana50Hi; ++vowel )
		{	if ( kanaSyllables[ consonant ].vowelList[ vowel ] != '\0' )
			{	fputs( "#define\t", stdout );
				if ( ch >= b8_katakanaSubLo && ch <= b8_katakanaSubHi )
					fputs( "b8_katakanaSub_", stdout );
				else if ( ch == b8_katakanaSubHi + 1 )
				{	fputs( "b8_ChoOn", stdout );	/* Maybe should have made separate loops? */
					/* This is a terrible kludge! */
					--vowel;	/* good thing vowel is signed. */
				}
				else
					fputs( "b8_katakana_", stdout );
				if ( ch != b8_katakanaSubHi + 1 )
				{	if ( kanaSyllables[ consonant ].consonant != '\0' )
						putchar( kanaSyllables[ consonant ].consonant );
					putchar( kanaSyllables[ consonant ].vowelList[ vowel ] );
				}
				printf( "\t( (ubyte) '\\x%02x' )\t/* '%c' */\n", ch, ch );
				++ch;
			}
		}
	}
	printf( "#define\tb8_DakuTen\t( (ubyte) '\\x%02x' )\t/* '%c' (muddying==voicing symbol, shaped like double quote mark) */\n",
			0xde, 0xde );
	printf( "#define\tb8_HandakuTen\t( (ubyte) '\\x%02x' )\t/* '%c' (half-muddying==plosive symbol, shaped like degree symbol) */\n",
			0xdf, 0xdf );

	putchar( '\n' );
	return EXIT_SUCCESS;
}


#if defined INTERMEDIATE_STUFF
char * intermediateCharNames[] =
{	"b7_NUL",	/* '\x00' Null character */
	"b7_SOH",	/* '\x01' start of heading */
	"b7_STX",	/* '\x02' start of text */
	"b7_ETX",	/* '\x03' end of text */
	"b7_EOT",	/* '\x04' end of transmission */
	"b7_ENQ",	/* '\x05' enquiry */
	"b7_ACK",	/* '\x06' acknowledge */
	"b7_BEL",	/* '\x07' ring that bell */
	"b7_BS",	/* '\x08' backspace, usually destructive */
	"b7_HT",	/* '\x09' horizontal tabulation */
	"b7_LF",	/* '\x0a' line feed in output */
	"b7_VT",	/* '\x0b' vertical tabulation */
	"b7_FF",	/* '\x0c' form (page) feed in output */
	"b7_CR",	/* '\x0d' carriage return */
	"b7_SO",	/* '\x0e' shift out */
	"b7_SI",	/* '\x0f' shift in */
	"b7_DLE",	/* '\x10' data link escape */
	"b7_DC1",	/* '\x11' device control 1 */
	"b7_DC2",	/* '\x12' device control 2 */
	"b7_DC3",	/* '\x13' device control 3 */
	"b7_DC4",	/* '\x14' device control 4 */
	"b7_NAK",	/* '\x15' negate acknowledge  */
	"b7_SYN",	/* '\x16' synchronous idle */
	"b7_ETB",	/* '\x17' end of transmission block */
	"b7_CAN",	/* '\x18' cancel */
	"b7_EM",	/* '\x19' end of medium */
	"b7_SUB",	/* '\x1a' substitute character */
	"b7_ESC",	/* '\x1b' excape character */
	"b7_FS",	/* '\x1c' file separator */
	"b7_GS",	/* '\x1d' group separator */
	"b7_RS",	/* '\x1e' record separator */
	"b7_US",	/* '\x1f' unit separator */
	"b7_SP", 	/* '\x20' for completeness */
    "b7_!",
    "b7_\"",
    "b7_#",
    "b7_$",
    "b7_%",
    "b7_&",
    "b7_'",
    "b7_(",
    "b7_)",
    "b7_*",
    "b7_+",
    "b7_,",
    "b7_-",
    "b7_.",
    "b7_/",
    "b7_0",
    "b7_1",
    "b7_2",
    "b7_3",
    "b7_4",
    "b7_5",
    "b7_6",
    "b7_7",
    "b7_8",
    "b7_9",
    "b7_:",
    "b7_;",
    "b7_<",
    "b7_=",
    "b7_>",
    "b7_?",
    "b7_@",
    "b7_A",
    "b7_B",
    "b7_C",
    "b7_D",
    "b7_E",
    "b7_F",
    "b7_G",
    "b7_H",
    "b7_I",
    "b7_J",
    "b7_K",
    "b7_L",
    "b7_M",
    "b7_N",
    "b7_O",
    "b7_P",
    "b7_Q",
    "b7_R",
    "b7_S",
    "b7_T",
    "b7_U",
    "b7_V",
    "b7_W",
    "b7_X",
    "b7_Y",
    "b7_Z",
    "b7_[",
    "b7_\\",
    "b7_]",
    "b7_^",
    "b7__",
    "b7_`",
    "b7_a",
    "b7_b",
    "b7_c",
    "b7_d",
    "b7_e",
    "b7_f",
    "b7_g",
    "b7_h",
    "b7_i",
    "b7_j",
    "b7_k",
    "b7_l",
    "b7_m",
    "b7_n",
    "b7_o",
    "b7_p",
    "b7_q",
    "b7_r",
    "b7_s",
    "b7_t",
    "b7_u",
    "b7_v",
    "b7_w",
    "b7_x",
    "b7_y",
    "b7_z",
    "b7_{",
    "b7_|",
    "b7_}",
    "b7_~",
    "b7_DEL"
};
#endif /* defined INTERMEDIATE_STUFF */

#ifdef INTERMEDIATE_STUFF
char * controlCharNames[] =
{	"b7_NUL",	/* '\x00' Null character */
	"b7_SOH",	/* '\x01' start of heading */
	"b7_STX",	/* '\x02' start of text */
	"b7_ETX",	/* '\x03' end of text */
	"b7_EOT",	/* '\x04' end of transmission */
	"b7_ENQ",	/* '\x05' enquiry */
	"b7_ACK",	/* '\x06' acknowledge */
	"b7_BEL",	/* '\x07' ring that bell */
	"b7_BS",	/* '\x08' backspace, usually destructive */
	"b7_HT",	/* '\x09' horizontal tabulation */
	"b7_LF",	/* '\x0a' line feed in output */
	"b7_VT",	/* '\x0b' vertical tabulation */
	"b7_FF",	/* '\x0c' form (page) feed in output */
	"b7_CR",	/* '\x0d' carriage return */
	"b7_SO",	/* '\x0e' shift out */
	"b7_SI",	/* '\x0f' shift in */
	"b7_DLE",	/* '\x10' data link escape */
	"b7_DC1",	/* '\x11' device control 1 */
	"b7_DC2",	/* '\x12' device control 2 */
	"b7_DC3",	/* '\x13' device control 3 */
	"b7_DC4",	/* '\x14' device control 4 */
	"b7_NAK",	/* '\x15' negate acknowledge  */
	"b7_SYN",	/* '\x16' synchronous idle */
	"b7_ETB",	/* '\x17' end of transmission block */
	"b7_CAN",	/* '\x18' cancel */
	"b7_EM",	/* '\x19' end of medium */
	"b7_SUB",	/* '\x1a' substitute character */
	"b7_ESC",	/* '\x1b' excape character */
	"b7_FS",	/* '\x1c' file separator */
	"b7_GS",	/* '\x1d' group separator */
	"b7_RS",	/* '\x1e' record separator */
	"b7_US",	/* '\x1f' unit separator */
	"b7_SP" 	/* '\x20' for completeness */
};
char deleteCharName[]= "b7_DEL";	/* '\x7f' delete  */
#endif /* INTERMEDIATE_STUFF */

#ifdef INTERMEDIATE_STUFF
int main(void)
{	int ch;
/* */
	for ( ch = 0; ch <= b7_ASCIIHi; ++ch )
	{	printf( "%3d (0x%02X): %s\n", ch, ch, charNames[ ch ].name );
	}

	/* I could have left a more complete trace, but I don't want to back up at this point. */
	/* The gaps help in redundancy check, so I don't want to eliminate the gaps. */
	for ( ch = 0; ch <= b7_ASCIIHi; ++ch )
	{	printf( "%3d (0x%02X): %s\n", ch, ch, intermediateCharNames[ ch ] );
	}

	for ( ch = b7_SP + 1; ch < b7_DEL; ++ch )
	{	if ( ch == '"' || ch == '\\' )
			printf( "\t\"b7_\\%c\",\n", ch );
		else
			printf( "\t\"b7_%c\",\n", ch );
	}

	for ( ch = 0; ch <= b7_ASCIIHi; ++ch )
	{	if ( ch < b7_ASCIIPrintLo )
			printf( "%3d (0x%02X): %s\n", ch, ch, controlCharNames[ ch ] );
		else if ( ch == b7_DEL )
			printf( "%3d (0x%02X): %s\n", ch, ch, deleteCharName );
		else 
			printf( "%3d (0x%02X): '%c'\n", ch, ch, ch );
	}
	return EXIT_SUCCESS;
}
#endif /* INTERMEDIATE_STUFF */

