        PAGEWIDTH 128
        HEADER 'Turn Signal Switch by Joel Matthew Rees'

* Electronic Turn Signal and Emergency Flasher Switch/Timer
*       Version 1.00
*       Implemented on the 68HC705K1
*       Assembled and tested on the M68HC705KICS
*
*       Copyright 1993 Joel Matthew Rees
*
*       Permission granted in advance for strictly non-profit
* educational use.  All other rights retained by the author.

* Authored by Joel Matthew Rees, June to November 1993
*   of                                  also of
*   South Salt Lake City                Amagasaki
*   Utah                                Japan
*   



        SUBHEADER 'Purpose, Description, and Design'
        PAGE
* Design:

*       When the turn signal and emergency flasher switch assembly on my
* parents' 1971 Colt fell apart some years back, rather than go to the
* nearest Dodge dealer in the next city to order the part, I designed a
* substitute switch using a pair of double-pole, double-throw relays in
* set-reset latch configuration.  Momentary-closed pushbuttons (mounted
* to the steering column cover) set the turn signals, and a momentary
* open pushbutton (mounted between the set buttons) or reed switch
* (mounted inside the cover housing so a magnet mounted to the steering
* column would actuate it) shut them off.  I usurped the emergency
* flasher unit and took power from the battery lead on the ignition
* switch, so when I needed emergency flashers, I could just turn both
* turn signals on. Mom had trouble getting used to this arrangement, but
* it passed Texas inspection in the seventies.

*       Examining the 68HC05K series MCUs brought that turn signal
* switch back to mind.  I thought ten bits of I/O, internal timers,
* and the STOP mode would be sufficient to replicate the logic of that
* switch assembly, adding directional turn signal return switches and
* flasher and turn sgnal return logic as well.  It turned out to be a
* tight squeeze.

*       The intent of this document is to focus on the problems of
* decoding inputs, directionalizing the turn signal return, eliminating
* the flasher units, and minimizing power consumption using the STOP
* mode and interrupts.  Hardware issues such as power supply design,
* electro-static discharge protection (very important in automotive
* applications), and details of lamp driver circuits are mostly set
* aside.  The logic has been tested with the M68HC705KICS simulator
* program, but has not been tested in a physical mock up or real
* vehicle.  (I ran out of time for this project.)

******* Potential problems: ******************************************

* I have not dealt with the problems that occur when the turn signals
* are turned on while the return sensors are within range of their
* actuator fields.  Two more bits of port B would have allowed complete
* separation of the two functions.  A little extra code might allow
* suppressing the turn signal inputs while both return switches are
* active, to reduce the operator frustration factor.

* I have also mostly ignored the problems presented by shorted switches.
* Having the turn signal functions on port B is handy, because a week
* turn signal spring will not keep the circuit from staying in the STOP
* instruction.  Another 50 bytes of ROM might have allowed tracking time
* of no change for every sensor, which could allow shutting out a
* sensor that has been active more than n minutes.

**********************************************************************

*       One of the intents of this design was to replace mechanical
* parts (that tend to flow in hot climates and become brittle in cold)
* with logic.  In order to eliminate the mechanical latching lever with
* spring loaded return, I have (perhaps arbitrarily) enforced a new "user
* interface" involving momentary contact switches for the turn signals.
* Two possible physical arrangements for the turn signal present
* themselves to my mind:  In the one, a spring-loaded, non-latching
* signal rod (the latch/return mechanism was the weakest part) actuates
* momentary contact switches in two axes; the rod can be pushed up or
* down to activate the turn signals, or it can be pushed in to actuate
* the manual cancel.  In another arrangement, the signal rod is fixed,
* and three pushbuttons are provided on a pad at its end.  (Cancel in
* the center would seem logical.)

*       The emergency flasher switch is a momentary on pushbutton, and
* it cancels itself in software -- push on, push off.  I think the audio
* feedback and dash panel indicators should be sufficient user feedback.

*       The turn signal return mechanism is a pair of reed relays
* mounted inside the bottom of the steering column housing about 11.25
* degrees apart, with one to four magnetic actuators distributed around
* the steering shaft.  The shape, size, and position of the magnets and
* reeds should be such that the reeds close only once as a magnet
* passes, and such that the reeds have about 22.5 degrees of total
* actuation. The overlap in actuation is used to determine the direction
* the steering wheel is rotating.  Minor changes to the design should
* allow the use of optical sensors, but I have the impression that
* magnetic devices are more immune to extremes of climate.  (Maybe?)

        SUBHEADER 'Model Schematic'
        PAGE
*       An idealized schematic follows:


* PA7>---FRNT_LEFT_DRVR---+---FL_SGNL_LAMP---GND
*                         |
*                         +---LEFT_INDCTR---GND
*                         |
*                         +---LEFT_AUDIO---GND

* PA6>---FRNT_RGHT_DRVR---+---FR_SGNL_LAMP---GND
*                         |
*                         +---RGHT_INDCTR---GND
*                         |
*                         +---RGHT_AUDIO---GND

* PA5>---REAR_LEFT_DRVR----RL_SGNL_LAMP---GND
*
* PA4>---REAR_RGHT_DRVR----RR_SGNL_LAMP---GND

* PA3<---+---160K_OHM---------+---BRK_SW---12V
*        |                    |
*        +---5V_ZENER---GND   +---CENTER_BRK_LGHT---GND

* PA2<---+---160K_OHM---ENGINE_ON_SW---12V
*        |
*        +---5V_ZENER---GND

* PA1>---DRVRS_ON_CNTRL

* IRQ<---+---100K_OHM---Vdd
*        |
*        +---EMG_SW---GND

* PB1<>--+----------LEFT_RETURN_SW---+
*        |                           |
*        +---LEFT_TURN_SW---+        |
*                           |        |
*          Vdd---100K_OHM---+        |
*                           |        |
*        +---RGHT_TURN_SW---+        |
*        |                           |
* PB0<>--+----------RGHT_RETURN_SW---+
*                                    |
* PA0<---+---------------------------+
*        |
*        +---CAN_SW---100K_OHM---Vdd

* RESET<---100K_OHM---Vdd

* IF THIS IS IMPLEMENTED IN A REAL CIRCUIT, ESD PROTECTION MUST BE ADDED!

        SUBHEADER 'Circuit Explication'
        PAGE
* Explanatory notes:

*       PA1 allows the signal drivers to be shut down to preserve power.
* I assume a suitable digital switch with power down is available.  If
* optical sensors are used for the turn signal return, the optical
* sources would also be turned on by this signal.  I assume it will be
* active high.

*       I arbitrarily assume the signal lamp drivers will be active low.
* Who could say why?  It should be easy to find all references to the
* driver lines with a text editor and reverse the active states of the
* outputs.

*       Hanging the indicators and the audio feedback on the front lamp
* drivers separates them from brake activity.  Two audio feedback
* devices are used here because there were not enough MCU outputs to
* drive a single cheap clicking device.  Stereo audio feedback might be
* a nice touch anyway, allowing a person with binaural hearing to
* determine by sound which signal is on.  (If the ROM were a little
* larger, a separate audio feedback could be modulated with different
* sounds for left, right, and emergency flashers.)

*       The center brake light is directly switched to eliminate one of
* the separate high-power outputs. Fortunately, its state is not
* dependent on any other inputs.

*       The engine-on input (PA2) is necessary so that the MCU knows
* when it can shut itself down to conserve power.  It is an interrupting
* input so that the MCU STOP mode can be used.  Brake lights and
* emergency flashers must be accessible when the engine is off, so those
* inputs are also interrupting.  The engine-on and brake inputs are
* naturally positive-going signals, but the emergency flasher input is
* not naturally biased.  So the the emergency flasher seemed a more
* natural match for IRQ.

*       The following crude diagram illustrates the actuation in a left
* rotation (restoring from a right turn or turning left).  The carots
* indicate the position of the reed switches, the asterisks indicate the
* actuation field of a magnet.

*    Actuation Field ********
*         Left Return Switch ^  ^ Right Return Switch
*                      ********
*                            ^  ^
*                        ********
*                            ^  ^
*                          ********
*                            ^  ^
*                            ********
*                            ^  ^
*                              ********
*                            ^  ^
*                                ********
*                            ^  ^

*       Referring to the above diagram, the right turn signal return
* switch is positioned about 5 5/8 degrees right of bottom center below
* the steering column and fixed to the steering column housing, and the
* left return is positioned the same distance to the left.  In this way,
* when the steering wheel rotates, the return switch for the direction
* of rotation actuates first, then the opposite direction switch closes,
* then the same direction switch opens, and then the opposite direction
* switch opens.  Thus, when restoring from a turn, the return switch for
* the direction of the turn closes after the other switch.  In other
* words, if you turn right, the steering wheel will rotate to the left
* when you return to straight, and the switches actuate from left to
* right during the restore.  (I say this so many times because it takes
* repeating to convince myself of it.)

*       When both turn signals are off, both turn signal switch ports
* are set as inputs.  When a turn signal switch input senses a closure,
* its port is made an output and driven high.  Now, in addition to
* manual cancel, the cancel port can sense steering wheel rotation.  In
* same direction rotation, there is a lead period when the cancel input
* does not remain high if the turn signal input ports' output states are
* inverted.  The debounce routine suppresses cancel if this lead time is
* sensed.

*       Since the opposite direction turn signal switch port remains an
* input, it can still be sensed.  The operator can change turn signals
* directly, without hitting the cancel button.

*       The separation of the switches and the actuation arc length I
* specify are based on my assumption that the steering wheel will rotate
* at a maximum speed of three revolutions per second, and that the four
* to six mS (1 MHz crystal) required debounce period requires a minimum
* 10 milli-second separate actuation period.  The actual result was 10.9
* degrees separate actuation (21.8 total), which I arbitrarily rounded
* up to a power of two fraction of the circle: 2*PI/16 total actuation
* and 2*PI/32 separate actuation.

*       A peculiar characteristic of this design is the apparent lack of
* use for the RESET input.  Since power-on reset is built in to the
* K-series MCUs, RESET should probably be tied high.  However, I suppose
* it could make a good manual cancel-all input, if such is necessary.

*       This assembler appears not to have a bit-not function for
* operand expressions.  So, I have in several places resorted to the
* obtuse expedient of exclusive-orring bits to be reset with $FF,
* ergo,                 AND #{$FF ^ F_OPC ^ F_MNC}
* instead of            AND #~(F_OPC | F_MNC)
* which I think would have been clearer.

*       One more assembler quirk -- I have used $FF in a couple of
* places I would normally have used -1 to indicate all bits set.  The
* assembler apparently turns {-1} into 1 in operands.

        SUBHEADER 'Obligatory Mnemonics'
        PAGE
$base   $0A
$include "68HC05K.EQU"

        SUBHEADER 'Program Mnemonics'
        PAGE
* physical I/O definitions
* PORTA
F_LDRV  equ B7          ; four lamp drivers
F_LDRV_ equ B7_
F_RDRV  equ B6
F_RDRV_ equ B6_
R_LDRV  equ B5
R_LDRV_ equ B5_
R_RDRV  equ B4
R_RDRV_ equ B4_
DRVLMPS equ {F_LDRV | F_RDRV | R_LDRV | R_RDRV}
LFTDRV  equ {F_LDRV | R_LDRV}
RGTDRV  equ {F_RDRV | R_RDRV}
FRNTDRV equ {F_LDRV | F_RDRV}
REARDRV equ {R_LDRV | R_RDRV}
BRK_SW  equ B3
BRK_SW_ equ B3_
NGN_SW  equ B2          ; engine (ENG and EMG could be confusing)
NGN_SW_ equ B2_
DRVDRV  equ B1
DRVDRV_ equ B1_
EMG_BTF equ DRVDRV      ; where to fold EMG_BT
EMG_BTF_        equ DRVDRV_
CAN_BT  equ B0
CAN_BT_ equ B0_
* PORTB
LFT_BT  equ B1  ; a handy data dependency! DO NOT CHANGE
LFT_BT_ equ B1_
RGT_BT  equ B0  ; another handy data dependency!
RGT_BT_ equ B0_
* IRQ input (inverted) is the real EMG_BT


* logical input definitions for debounce and states
* bits are remapped for debounce and speed
* six bits of physical input, high two bits borrowed for cancel logic
F_LFT   equ LFT_BT      ; another handy data dependency!
F_LFT_  equ LFT_BT_
F_RGT   equ RGT_BT      ; another handy data dependency!
F_RGT_  equ RGT_BT_
TURNSIG equ {F_LFT | F_RGT}
F_BRK   equ {BRK_SW < 2}        ; physical output bit
F_BRK_  equ {BRK_SW_ + 2}
F_NGN   equ {NGN_SW < 2}        ; physical output bit
F_NGN_  equ {NGN_SW_ + 2}
F_EMG   equ {DRVDRV < 2}        ; fold onto output bit
F_EMG_  equ {DRVDRV_ + 2}
ANYLITE equ {TURNSIG | F_BRK | F_EMG}
F_CAN   equ {CAN_BT < 2}
F_CAN_  equ {CAN_BT_ + 2}
F_OPC   equ $40                 ; opposite direction cancel record
F_MNC   equ $80                 ; manual cancel record

*       The turn signal table and many of the turn signal routines
* depend on F_LFT and F_RGT and LFT_BT and RGT_BT being in bits one and
* zero (but not particular which is which!).  If these are changed,
* there will be a domino effect on the rest of the code.

        SUBHEADER 'Resource Definitions'
        PAGE


XTAL    equ 1   ; crystal frequency, in MHz
RTIPOW  equ 1   ; bits 1 & 0 for timer hardware
RTIRATE equ {2 < RTIPOW}        ; 2 ^ RTIPOW
TIMOUT  equ {9155*XTAL/RTIRATE+1}       ; 5 minutes (if XTAL equ is correct)
* only one flash rate, 2/3 duty cycle
FLASH0  equ {32*XTAL/RTIRATE/4}         ; about 1/4 second off time
FLASH1  equ {32*XTAL/RTIRATE/2}         ; about 1/2 second on time

        org MOR
* Software Pulldown is used.
* Can't spare the third RC oscillator pin.
* Set RC because we assume the cheap oscillator.
* STOP must not convert to WAIT: saves power when engine off.
* Assume Low Voltage Reset would be meaningless.
* PA3 - PA0 interrupts are used to bring the chip up for brake, engine on.
* Accept level sensitive interrupts, since IRQ is disabled immediately.
* Assume COP might be useful here.
        fcb {RC | PIRQ | LEVEL | COPEN}

        org RAMBEG
DEBO    rmb 3   ; the debounce record chain
SSTBLE  rmb 1   ; temporary record of stable bits
TOGGLE  rmb 1   ; record of toggles
ISTATE  rmb 1   ; record of the current input state
STATES  rmb 1   ; current controller states
NGN_TIM rmb 2   ; engine time out timer
FLASH   rmb 1   ; flash timer
FLDARK  rmb 1   ; flash light or dark state, 11111111 == dark
FLMASK  rmb 1   ; flash mask to communicate between BRK, EMG, and TRN
TMPTRN  rmb 1   ; temporary for keep turn signal interpretation
LASTST  rmb 1   ; previous state record to separate timer from decode
        rmb 4
DBGCT   rmb 4   ; count how many times TIMSRV has executed


* ROM definitions follow:
        org ROMBEG

        SUBHEADER 'Timer Service Code'
        PAGE

*       The timer service routine has these functions: debouncing the
* inputs, flagging changes in TOGGLE, adjusting STATES to match the
* inputs, and maintaining two software clocks. Interrupts occur on timer
* overflow (1024 CPU cycles).

*       The debounce routine translates the physical inputs to logical
* inputs, maintains a record of the last three logical states sensed in
* DEBO, debounces (in parallel) each input on three identical states,
* flags changes in the debounced input in TOGGLE, and leaves the current
* debounced input in ISTATE.  Since debouncing occurs on three identical
* states, the debounce period is between two and three timer overflows
* -- 4 to 6 mS with a 1 MHz crystal, 2 to 3 mS with a 2 MHz crystal.

*       This debounce technique works because we don't care what is
* happening to the inputs between timer overflows, and because we assume
* there is no physical mechanism to induce noise coincident and rythmic
* with timer overflow.

*       The TOGGLE cancel (F_CAN) bit is suppressed on same-direction
* rotation. Trailing edge TOGGLE F_CAN and F_EMG are filtered out to
* simplify interpretation of those functions.  Opposite direction cancel
* (F_OPC) and manual cancel (F_MNC) ISTATE auxiliary flags are
* maintained to provide information on the source of the F_CAN bit, and
* are always filtered out of TOGGLE and STATES.  STATES F_CAN is
* adjusted according to TOGGLE F_CAN rather than ISTATE F_CAN, by which
* means we hope to not need to remember in which direction a cancel was
* sensed.  In other words, we hope that F_CAN does not toggle again
* until both return sensors are past the actuator field.

*       After adding an event queue and checking the timing results, two
* things became clear: there is plenty of time to handle all the changes
* to STATES without a queue, and TOGGLE makes a sort of horizontal
* priority queue anyway.  In other words, there is no reason to make the
* adjustments to STATES asynchronous when there is plenty of headroom
* for synchronous adjustments.  Moreover, the adjustments take less time
* than organizing the button events in a queue.  (Darn!  I was kind of
* proud of that six byte queue, and the shift technique for storing
* events into it.)

*       When moving the ISTATE turn signal bits to STATES, the STATES
* turn signal bits are examined to see if the direction has changed.  If
* a STATE turn signal bit is on, PORTB is set to wait for F_CAN.

*       The engine time-out counter communicates with the power down
* routine via the STATES engine on (F_NGN) bit.  In other words, STATES
* F_NGN bit is held off until the ISTATE F_NGN bit and the STATES F_EMG
* and F_BRK bits have been inactive for five minutes.  (This perhaps
* overloads the semantics of STATES F_NGN.)

*       In the current design, the turn signals are not masked with the
* ISTATE F_NGN bit, and so will function during the time-out period.
* This "feature" can easily be corrected in the debounce routine.

*       It should be relatively easy to add a turn signal time-out
* counter to turn off the turn signals after the same turn signal has
* been on an un-reasonable period, such as five or more minutes.
* Such a time-out might negatively impact safety when attempting to turn
* in backed-up traffic or onto a very busy highway, or when using the
* turn signal to show a car parked for delivery, etc.  It also might
* cause drivers to become even lazier.

*       Both timers reset in such a manner as to be in a known state at
* any time they are put into use.

*       The single emergency and turn signal signal flasher uses a single
* rate and duty cycle.

        SUBHEADER 'Timer Service Code -- Debounce and Toggle Filter'
        PAGE

        swi     ; since we have a few unused ROM locations
        swi
        swi

$CYCLE_ADDER_ON
TIMSRV  bset TOFR_,TSCR

* move the debounce chain
        lda DEBO+1
        sta DEBO+2
        lda DEBO
        sta DEBO+1

* record previous STATE
        lda STATES
        sta LASTST

* read the static inputs, converting physical to logical
        lda PORTA
        and #{BRK_SW | NGN_SW | CAN_BT}         ; clears DRV_DRV
        bih TMIEMG      ; invert EMG and fold it in
        ora #EMG_BTF    ; keep temporally very close to first read
TMIEMG  equ *
        lsla    ; remap
        lsla
        sta DEBO
        lda PORTB       ; PB2-7 always read 0
        ora DEBO
        sta DEBO

* read the turn signal return and manual cancel auxiliary flags
* (F_CAN & turn_sig_on) <=> read auxiliary flags
* Assume that DDRB showing output means turn signal on and waiting
* for cancel.
* Filtering F_CAN on same-direction turn is handled later.
* Remember that F_CAN should be sensed via TOGGLE, to avoid multiple
* interpretations of one pass of the actuator.

        bit #F_CAN      ; nothing at all without cancel
        beq TM0TURN
        ldx DDRB        ; cancel meaningless if not waiting for it
        beq TM0TURN
$CYCLE_ADDER_OFF
$CYCLE_ADDER_ON
        txa
        eor #{LFT_BT | RGT_BT}  ; strobe other side
        sta PORTB       ; set output state first
        sta DDRB        ; handy (data dependent) output state, huh?
        brset CAN_BT_,PORTA,TMTBIT6     ; CAN_BT_ into carry
TMTBIT6 rora    ; carry into bit 7
        clr DDRB        ; kill strobe: manual cancel?
        brset CAN_BT_,PORTA,TMTBIT7     ; CAN_BT_ into carry again
TMTBIT7 rora    ; carry into bit 7 again
        and #{F_OPC | F_MNC}
        ora DEBO
        bit #F_OPC      ; ignore turn signals if opposition cancel
        beq TMTNTN
        and #{$FF ^ TURNSIG}
TMTNTN  sta DEBO
        stx PORTB       ; restore turn signal inputs
        stx DDRB
TM0TURN equ *
* leave A == DEBO
$CYCLE_ADDER_OFF
$CYCLE_ADDER_ON

* debounce inputs, set flags in TOGGLE and ISTATE
* STABLE_BITS = ~((DEBO[0] ^ DEBO[1]) | (DEBO[1] ^ DEBO[2]))
* A == DEBO on entry
        eor DEBO+1
        sta TOGGLE      ; re-use TOGGLE, TOGGLE not valid
        lda DEBO+1
        eor DEBO+2
        ora TOGGLE      ; a bit for any input that bounced
        coma            ; bits that bounced are clear
        tax             ; STABLE_BITS
        and DEBO        ; state of stable bits
        sta SSTBLE      ; (unstable bits still clear)
        txa
        and ISTATE      ; look only at bits that are stable
        eor SSTBLE      ; flag bits that toggled
        sta TOGGLE      ; true TOGGLEs, but not yet filtered
        coma            ; mask of non-toggles
        and ISTATE      ; old state of non-toggles
        sta ISTATE      ; (toggled bits cleared)
        lda TOGGLE      ; mask of toggles
        and SSTBLE      ; new states of bits that toggled
        ora ISTATE
        sta ISTATE      ; ISTATE now has current debounced inputs
* leave A == ISTATE

* filter out TOGGLE F_CAN if same direction
* A == ISTATE on entry
        bit #{F_OPC | F_MNC}    ; opposite direction or manual?
        bne TMF1CAN
        bclr F_CAN_,TOGGLE      ; same direction
TMF1CAN equ *
* leave A == ISTATE

* filter out trailing edge of F_EMG and F_CAN in TOGGLE
* also filter out auxiliary TOGGLEs in passing
* ISTATE == A on entry
        ora #{F_NGN | F_BRK | F_LFT | F_RGT}    ; not filtered
        and TOGGLE      ; filter (logical) fall(en) edges
        and #{$FF ^ F_OPC ^ F_MNC}      ; auxiliaries should never TOGGLE
        sta TOGGLE
* leave A == TOGGLE

        SUBHEADER 'Timer Service Code -- Adjust STATES'
        PAGE

* move brake bit from ISTATE to STATES
        bclr F_BRK_,STATES
        brclr F_BRK_,ISTATE,TMF0BRK
        bset F_BRK_,STATES
TMF0BRK equ *

* toggle STATES F_EMG if TOGGLE F_EMG
* act on TOGGLE to avoid multiple reads of EMG button
        brclr F_EMG_,TOGGLE,TMF0EMG
        lda STATES
        eor #F_EMG
        sta STATES      ; STATES not yet valid!
TMF0EMG equ *
        bra TMSTRN      ; skip over the turn signal states table
* leaves A indeterminate

        swi     ; since we have a few unused ROM locations
        swi
        swi
        swi
        swi

* this table is very data dependent
TURNSTATE       equ *   ; depends on turn signals being rightmost
*            S'                   S  I
        fcb %00                 ; 00 00
        fcb %01                 ; 00 01
        fcb %10                 ; 00 10
        fcb {%00 | F_EMG}       ; 00 11 set F_EMG instead
        fcb %01                 ; 01 00
        fcb %01                 ; 01 01
        fcb %10                 ; 01 10 should not occur
        fcb %10                 ; 01 11 this inverts directions
        fcb %10                 ; 10 00
        fcb %01                 ; 10 01 should not occur
        fcb %10                 ; 10 10
        fcb %01                 ; 10 11 this inverts directions
        fcb %00                 ; 11 00 should not occur
        fcb %01                 ; 11 01 should not occur
        fcb %10                 ; 11 10 should not occur
        fcb %00                 ; 11 11 should not occur
* Since PEPROM accesses only one bit at a time,
* and since I can't figure out how to build the turn signal new state
* table in the PEPROM as part of this file anyway,
* and since we apparently have plenty of ROM,
* I decided this table is not a good candidate for the PEPROM.
* Darn!
* After careful consideration, I decided this table does not benefit
* from a more symbolic construction.  As it stands, it does not matter
* whether F_LFT or F_RGT is b0, only that they do reside in B0 and B1.

        swi     ; since we have a few unused ROM locations
        swi
        swi
        swi
        swi

* turn on STATES turn signal if ISTATE turn signal is on
* check to see if direction has changed
TMSTRN  lda STATES
        and #TURNSIG
        lsla
        lsla
        sta TMPTRN
        lda ISTATE
        and #TURNSIG
        ora TMPTRN
        tax
        lda TURNSTATE,x
        sta TMPTRN
        lda STATES
        and #{$FF ^ TURNSIG}
        ora TMPTRN      ; table is 0s in b2-7, except forced F_EMG
        sta STATES      ; may be reset by F_CAN
* leaves A == STATES

* cancel STATES turn signal if TOGGLE F_CAN is not filtered out
* Act on TOGGLE to avoid multiple reads of single actuator pass!
* A == STATES on entry
        brclr F_CAN_,TOGGLE,TMCAN0
        and #{$FF ^ TURNSIG}
        sta STATES      ; STATES not yet valid!
TMCAN0  equ *           ; turn signals are valid
* leaves A == STATES

* Now we should have no more than one turn signal on.
* set up steering wheel return sense state or clear it
* then latch turn signal input to keep it from reverting
* A == STATES on entry
        and #TURNSIG
        sta PORTB       ; another handy value dependency!
        sta DDRB        ; gate it out, if there
* I feel there is a better way, but I'm out of time
        sta TMPTRN      ; feed response back
        lda ISTATE
        and #{$FF ^ TURNSIG}
        ora TMPTRN
        sta ISTATE
* leaves A == ISTATE

* if not timed out, STATES F_NGN should not yet be clear!
        lda NGN_TIM+1
        ora NGN_TIM
        bne TMFNGN0
        bset F_NGN_,STATES       ; might time out before RTI
TMFNGN0 equ *

* STATES is still not valid!
* fall through to TMCLOCK

        SUBHEADER 'Timer Service Code -- Engine Time Out and Flasher Timers'
        PAGE

* reset engine time-out if F_NGN, F_EMG, or F_BRK TOGGLEs
TMCLOCK lda TOGGLE
        and #{F_NGN | F_EMG | F_BRK}
        beq TMORST
        lda #{TIMOUT & $FF }    ; reset value low byte
        sta NGN_TIM+1
        lda #{TIMOUT / $100 + 1}        ; borrow on 0, not -1
        sta NGN_TIM
TMORST  equ *

* if flashers TOGGLE off, reset the flashers
* if they TOGGLE on, clear the mask only if none on before
        lda #{F_EMG | TURNSIG}
        bit TOGGLE
        beq TMSCLK
        bit STATES      ; flasher active?
        bne TMMFLH
        lda #$FF        ; reset flash timer
        sta FLDARK
        lda #FLASH1
        sta FLASH
        bra TMSCLK

TMMFLH  bit LASTST      ; flashers already running?
        bne TMMFLH0     ; might be better to shine on every change?
        clr FLDARK
TMMFLH0 equ *

*       Although the Real Time Interrupt is masked, the software clocks
* clock on the RTI flag.  See RTIPOW for the RTI flag rate

TMSCLK  brclr RTIF_,TSCR,TIMDUN
        bset RTIFR_,TSCR

*       Engine time-out counter only runs when the ISTATE F_NGN bit is
* low and the emergency flashers and brakes (STATES F_EMG and F_BRK) are
* off.  Underflow is held off by cleared STATES F_NGN bit.  Communicates
* with power down routine via STATES F_NGN.

        brset F_NGN_,ISTATE,TMNGNR0
        lda STATES
        and #{F_NGN | F_EMG | F_BRK}    ; brakes or emg flashers?
        eor #F_NGN                      ; already counted out?
        bne TMNGNR0
        dec NGN_TIM+1
        bne TMNGNR0      ; initial count is adjusted to make this work
        dec NGN_TIM
        bne TMNGNR0
        bclr F_NGN_,STATES      ; signal timed out
        bset F_NGN_,TOGGLE      ; communicate change to STATES F_NGN
TMNGNR0 equ *   ; Only now is STATES valid!

*       The flasher counter runs only when STATES F_EMG, F_LFT, or F_RGT
* are set.  When there are no flashers running, FLDARK is set, causing
* FLASH1 (on time) to be loaded when when flasher timer starts.

TMFLASH lda #{F_EMG | TURNSIG}
        bit STATES      ; flasher active?
        beq TMFLH0
        dec FLASH
        bne TMFLH0
        lda #FLASH1
        com FLDARK
        bpl TMFLTM      ; result dark (1s) or light (0s)
        lda #FLASH0
TMFLTM  sta FLASH
TMFLH0  equ *

*       It might be amusing to install a turn signal time out so that if
* the turn signals remain flashing for more than ten minutes, they turn
* themselves off.  The entire mechanism should fit here.  As mentioned
* above, consider safety first!

TIMDUN
        inc DBGCT+3     ; count number of timer interrupts serviced
        bne TIMDUX      ; put here strictly for debugging sessions
        inc DBGCT+2     ; so I can have an idea of virtual time
        bne TIMDUX
        inc DBGCT+1
        bne TIMDUX
        inc DBGCT
TIMDUX
        bset IRQR_,ISCR ; POTA and IRQ activity may set IRQF
        rti
$CYCLE_ADDER_OFF

        swi     ; more unused ROM
        swi
        swi
        swi
        swi
        swi
        swi

        SUBHEADER 'Initializations'
        PAGE

* waking up simply resets the MCU
* if SWI occurs, the machine is not in a defined state, and must reset
IRQSRV
SWISRV
        ldx #$5C        ; unique
        stx COPR
        rsp     ; now falls through to RSTSRV
        bset IRQR_,ISCR ; it just gets us started

* general inits
RSTSRV
* disable external interrupts
        bclr IRQE_,ISCR ; disable external interrupts
        bset IRQR_,ISCR ; clear the external interrupt flag
* get ports stared right
        lda #DRVLMPS
        sta PORTA       ; initial output values
        lda #{DRVLMPS | DRVDRV}
        sta DDRA        ; set up directions
        sta PDRA        ; pull downs only on inputs
        clr PORTB       ; initial outputs (do port B just in case)
        clr DDRB        ; B initially input
        clr PDRB        ; with pull downs set
* make sure EPROM programming stuff is out of the way
*        clr EPROG       ; ICS05K complains, skip by hand
        clr PESCR
* set up specific variables
        lda #F_NGN      ; pretend the engine is on, to get us started
        sta DEBO        ; start debounce chain clear
        sta DEBO+1      ; DEBO+2 is thrown out on first timer interrupt
        sta STATES      ; no unknown states
        sta ISTATE
        lda #{TIMOUT & $FF }            ; reset time-out low byte
        sta NGN_TIM+1
        lda #{TIMOUT / $100 + 1}        ; counter borrows on 0, not -1
        sta NGN_TIM
        lda #$FF
        sta FLDARK      ; initial flash state is dark
        lda #FLASH1
        sta FLASH       ; initial flash time is on/light state
* clear the virtual debugging time counter
        clr DBGCT
        clr DBGCT+1
        clr DBGCT+2
        clr DBGCT+3
* set up hardware timer
        ldx #$6A        ; unique use of X
        stx COPR        ; b0 clear, about to change timer rate
        lda #{RTIPOW}   ; set rate
        sta TSCR
        ora #{TOFR | RTIFR}     ; clear pending interrupts
        sta TSCR
        bset TOIE_,TSCR ; first section overflow interrupt only

* Falls through to DOSTATES

        SUBHEADER 'DOSTATES: Interpret States'
        PAGE

*       The debounce and timer routines make the output state machine
* simple.  All indicator lamps are set according to the indicator STATES,
* then masked by the current flasher output state (FLDARK).  Then, if the
* brake light STATES bit is set, the rear lamps are forced on.  Then,
* DRVDRV is only set if lamps are on (to save power when the engine is
* off).

*       Finally, if STATES F_NGN bit is clear, the power-down routine is
* executed (and never returned from).  Again, requiring the time-out
* counter to hold off the STATES engine-on bit may be a bit of a
* semantic overload, but it seems more natural than making the STATES
* F_NGN bit a mere mirror of ISTATE F_NGN, and then maintaining a
* separate flag which would operate by the same rules as STATES F_NGN
* does in the present design.  It also seems less data coupled than
* having the power-down sense routine examine the internals of the
* counter.

*       FLDARK is used slightly ambiguously:  in the flasher timer
* routine, bit 7 is used to determine whether to use the on period or
* off period.  In the output state machine, below, all lamp bits are
* used to mask out the signal lamps when the flasher says they should be
* dark.  Rather than concern myself in several places with the state of
* non-lamp bits, I mask them out where they could cause trouble.

DOSTATES:
        wait    ; wait for something intelligible to debounce
        ldx #$78        ; unique
        stx COPR        ; tell COP we're in control

* update the (inverted) lamp output state
        lda #DRVLMPS    ; guess nothing shines (inverted drive)
* A == lamp output state

* now mask in inverted drive bits for any flashers that are on
        brclr F_EMG_,STATES,STT0EMG
        and #{$FF ^ DRVLMPS}
STT0EMG equ *
* A == lamp output state

        brclr F_LFT_,STATES,STT0LFT
        and #{$FF ^ LFTDRV}
STT0LFT equ *
* A == lamp output state

        brclr F_RGT_,STATES,STT0RGT
        and #{$FF ^ RGTDRV}
STT0RGT equ *
* A == lamp output state

* mask out flasher lamps if flasher is in off state
        ora FLDARK
* A == lamp output state

* check for brake lights
STTBRK  brclr F_BRK_,STATES,STT0BRK
        and #{$FF ^ REARDRV}    ; brake has higher priority than flash
STT0BRK equ *
* A == lamp output state

        and #DRVLMPS    ; filter out extraneous bits
        sta FLMASK
        eor #DRVLMPS    ; any lamps driven?
        beq STT0FLH     ; invert the CCR zero bit and move it to DRVDRV
        bset DRVDRV_,FLMASK
STT0FLH equ *
* now A is free

        lda PORTA
        and #{$FF ^ DRVLMPS ^ DRVDRV}
        ora FLMASK
        sta PORTA

* all done with lamps until next timer interrupt
* loop to wait if engine not timed-out
        brset F_NGN_,STATES,DOSTATES

* At this point, we know there has been no engine, emergency flasher, or
* brake activity for about five minutes.

* go to power conserving state
* first shut down all lamps
        lda #DRVLMPS    ; DRVDRV, CAN_BT also off!
* should be     lda #{DRVLMPS & ~DRVDRV & ~CAN_BT} for clarity
        sta PORTA       ; don't feed CAN_BT to pull downs
* now make sure power down state is valid!
        lda #{DRVLMPS | DRVDRV | CAN_BT}
        sta DDRA        ; interrupts only from NGN_SW, BRK_SW, EMG_BT
        sta PDRA        ; pull downs only on inputs
        clr PORTB       ; initial outputs (do port B just in case)
        clr DDRB        ; B initially input
        clr PDRB        ; with pull downs set
* make sure EPROM programming stuff is out of the way
*        clr EPROG       ; ICS complains, skip by hand
        clr PESCR
* enable external interrupts
        bset IRQE_,ISCR ; DO NOT clear the flag (don't miss anything)
        stop    ; also shuts timer down
        swi     ; should never come here (IRQSRV resests stack pointer)
        swi     ; more unused ROM
        swi
        swi

        SUBHEADER 'Interrupt Vectors'
        PAGE

        org VECTORS
        fdb TIMSRV
        fdb IRQSRV
        fdb SWISRV
        fdb RSTSRV

        SUBHEADER 'Symbol Table'
        end
