; Memory map and other global definitions for fig-FORTH for SH-3
; Joel Matthew Rees, Hyougo Polytec Center
; 2014.02.25
; 
; Borrowing heavily from the fig FORTH 6800 model by Dave Lion, et. al.,
; and from the bif-6809 near-fig FORTH by the author of this code.
;
; Licensed extended under GPL v. 2 or 3, or per the following:
; ------------------------------------LICENSE-------------------------------------
;
; Copyright (c) 2009, 2010, 2011 Joel Matthew Rees
;
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
;
; The above copyright notice and this permission notice shall be included in
; all copies or substantial portions of the Software.
;
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
; THE SOFTWARE.
;
; --------------------------------END-OF-LICENSE----------------------------------


	.cpu	sh3

; For huge things, like U/ (USLASH),
; which might be cut in half and repeated or something. But probably not.
; PRIORITY_SIZE:	.DEFINE	"1"

; Uncomment this if building for ROM:
; BUILD_FOR_ROM:		.DEFINE	"1"

; Comment this out when all forward references are resolved,
; and test stuff is no longer needed.
SHIMMED:		.DEFINE	"1"	
TESTING:		.DEFINE	"1"


NATURAL_SIZE:	.equ	4	; 4 byte word
HALF_SIZE:		.equ	( NATURAL_SIZE / 2 )
ALIGN_MASK: 	.equ	( H'FF & ~(NATURAL_SIZE - 1) )
ALL_BITS8:		.equ	( H'FF & -1 )

; byte offsets for various parts of significance
; In the SH3, natural width is 32 bits, half-width is 16, byte is 8.
;
	.AIFDEF	_LIT	; (least significant byte first)
LSHALF:			.equ	0
LSBYTEinNAT:	.equ	0
LSBYTEinHALF:	.equ	0
MSHALF:			.equ	HALF_SIZE
MSBYTEinNAT:	.equ	( NATURAL_SIZE - 1 )
MSBYTEinHALF:	.equ	( HALF_SIZE - 1 )
	.AELSE	; _BIG is default	; (most significant byte first)
LSHALF:			.equ	HALF_SIZE
LSBYTEinNAT:	.equ	( NATURAL_SIZE - 1 )
LSBYTEinHALF:	.equ	( HALF_SIZE - 1 )
MSHALF:			.equ	0
MSBYTEinNAT:	.equ	0
MSBYTEinHALF:	.equ	0
	.AENDI
; 
; Convenience definitions? -- Or just confusing?
BYTE_hLO_bLO:	.equ	LSBYTEinNAT
BYTE_hLO_bHI:	.equ	( LSHALF + MSBYTEinHALF )	; high byte of low half
BYTE_hHI_bLO:	.equ	( MSHALF + LSBYTEinHALF )	; low byte of high half
BYTE_hHI_bHI:	.equ	MSBYTEinNAT
;
; The word "word" is soooooo confusing! 
; (Word == half in many people's minds, but "word" has other meanings, especially here.)
; (Long == natural in many people's minds, but what happens when 64 bits happen?)
;
; ******************************************************************************
; *             ***** VERY IMPORTANT NOTE on byte/word order *****
; * 
; * FORTH allows the CPU's natural byte order. 
; *
; * ****** BUT! ******
; *
; * fig-FORTH assumes that 
; * DOUBLE natural width integers 
; * will be Most-Significant-word-First in memory.
; * 
; * So, even on CPUs running least-significant-byte-first, 
; * DOUBLE words will still be stored most-significant-natural-word-first.
; * 
; * (Don't play games with this. It's bad mojo.)
; ******************************************************************************


; ******************************************************************************
; Setting up the overall memory map:
;
; The SH-3 has modern dev tools, so we don't have to pay much attention 
; to memory layout.
;
; But we do need a map to work with.
;
; By tradition, the start of object code contained jumps to the initialization code:
; COLD entry point (4 byte jump) and
; WARM entry point (4 byte jump), followed by the
; Init constants table (about 20 natural-width entries).
;
; The init constants would be followed by the dictionary, 
; usually starting with the inner interpreter, 
; and proceeding to higher-level stuff:
;
; ------------
; low address:
; JMP to start (But maybe handled through the reset code?)
; 
; Global constants table
;
; machine-level initialization code
;
; primatives (ROMmable), including inner interpreter
;
; utility definitions (ROMmable)
;
; normal expression use definitions (ROMmable)
;
; the outer FORTH interpreter (ROMmable)
;
; development use definitions (ROMmable)
; 
; high address of ROMmable part
; -----------------------------
;
; If the object were ROMmable, at the end of the object code (ROM image)
; there would be a non-linked definition of "FORTH"
; to be copied to RAM, and the RAM part of the dictionary would link to that.
;
; In modern processors, ROM can be placed above or below RAM, 
; so we shouldn't concern ourselves too much about where it is
; (or even what the map looks like within it).
;
; What concerns us is the RAM map, which is traditionally not well defined.
;
; These are the large areas of memory which we need to specify allocation for:
; 
; ------------------------------------------------------------------------------
; Prebuilt dictionary (potentially ROMmable).
; 
; Run-time dictionary (starts with the "FORTH" entry).
;    The un-allocated area of the run-time dictionary 
;    is used as a general purpose text buffer.
;
; Parameter stack.
;    fig-FORTH had this growing down towards the dictionary, 
;    which we now know is an invitation for attacks.
;
; Terminal line input buffer.
;
; Return, or flow-of-control stack.
;    fig-FORTH had this growing down toward the input buffer, 
;    which, again, we now know is a recipe for vulnerabilities.
;
; Per-USER variable table(s).
;
; Disk buffers.
; 
; ******************************************************************************
; The 6800 model modified the map to match customary memory use on the 6800:
;
; The bottom 128 bytes or so of memory in a 6800 were generally RAM, 
; because of the direct-page fast-addressing mode. 
; (ROM at the bottom might have made more sense, 
; to deal with small integers mistakenly used as pointers.)
;
; From 0x80 to 0xff was also in the direct page range, 
; but might be ROM, for fast access to constants. 
; (And, beginning with the 6801, IIRC, heavily used routines.)
;
; From 0x100 to 0x1000 might not be populated, or might have RAM.
;
; The very top of memory, 0xF000 to 0xFFFF, was often populated 
; with a monitor (debug/BIOS) ROM, to allow the reset and interupt vectors 
; to be easily read out of ROM.
;
; The 6800 fig-FORTH model used the following map:
;
; ------------------------------------------------------------------------------
;
; 0x00E0: Scratch pad RAM (substitute for CPU temporary registers).
; 0x00F0: The virtual machine registers (IP, SP, RP, W).
;         ...
; 0x0100: Run-time dictionary
;         ...
;         Parameter stack
; 0x0F30: Terminal input buffer
;         ...
;         Return stack
; 0x1000: Pre-built (ROMmable) dictionary
;    Including COLD and WARM entry jumps and initialization table at the front.
;         ...
; 0x3000: disk buffers (4*128)
;         ...
; 0x3210: RAM disk (very small)
;         ...
; 0x3FFF: End of RAM
;
; So, ....
;
; ******************************************************************************
; Some general limits to build within:

; Small while debugging, so we don't have to hunt too far.
RUNTIME_DICTIONARY_SIZE:	.equ	h'1000	

; Small while debugging, so we don't have to hunt too far.
PARAMETER_STACK_SIZE:	.equ h'60	

; Keep it small while debugging.
CONTROL_STACK_SIZE:	.equ	h'60	

; Something to help avoid things bumping into things.
ALLOC_GAP:	.equ	h'10	

; This can be bigger later.
TERMINAL_INPUT_BUFFER_SIZE:	.equ	h'100

; Well, maybe we'll get this far some day.
N_USERS:	.equ 8	

; This can be bigger later, too.
DISK_BUFFER_SIZE:	.equ	h'100
DISK_BUFFER_RECORD_SIZE:	.equ	2*NATURAL_SIZE+DISK_BUFFER_SIZE
DISK_BUFFER_COUNT:	.equ	4

; RAM disk could be useful 
; even after the full system is running with real (flash?) disks.
; So could a "ROM" disk with initial error messages. Maybe. Hmm. Anyway.
RAM_DISK_SIZE:	.equ	h'8000


; ******************************************************************************
; Mapping the virtual machine registers:

; r0 must be used for many intermediate values.
; r0 -- r7 are bank-switched on interrupt.
; r8 -- r15 are not bank-switched on interrupt.
; All registers have to be explicitly saved during a full task-switch, 
; but the supervisor monitor code can keep a lot of context in the bank-switched part.

; Registers that should be in r8 -- r15 are (as much as possible)
; * temporaries that we agree to explicitly save;
; * true globals (global contant tables, global library pointers, etc.);
; * and per-task globals that have no parallel in the supervisor, if there are any.
; * And the flow-of-control stack pointer, maybe.

; For system calls, accessing the switched bank is not too hard.

; FORTH tracks the call/return sequence in a separate stack from the parameters.
fRP:	.reg	r15	; control/Return stack Pointer
fSP:	.reg	r7	; parameter Stack Pointer

; fig-FORTH has primitive multi-tasking, with a per-User table of variables
fUP:	.reg	r6	; User (task/thread/process) base Pointer

fIP:	.reg	r5	; virtual Instruction Pointer

; Defined symbols (objects/routines) are called "WORDs" in classic FORTH.
fW:		.reg	r4	; currently executing Word definition pointer
;
; By keeping the FORTH model in banked registers 
; that the SH3 switches out during interrupts, 
; we can save a lot of time during interrupt processing.


; ******************************************************************************
; Structure of the per-USER variables table:
	.section usertable, dummy
	.org	0
UORIG:	.equ $
	.res.l	3	; three reserved variables, not sure why
XSPZER:	.equ $-UORIG
	.res.l	1	; initial top of data stack for this user
XRZERO:	.equ $-UORIG
	.res.l	1	; initial top of return stack
XTIB:	.equ $-UORIG
	.res.l	1	; start of terminal input buffer
XWIDTH:	.equ $-UORIG
	.res.l	1	; name field width
XWARN:	.equ $-UORIG
	.res.l	1	; warning message mode (0 = no disc)
XFENCE:	.equ $-UORIG
	.res.l	1	; fence for FORGET
XDP:	.equ $-UORIG
	.res.l	1	; dictionary pointer
XVOCL:	.equ $-UORIG
	.res.l	1	; vocabulary linking
XBLK:	.equ $-UORIG
	.res.l	1	; disc block being accessed
XIN:	.equ $-UORIG
	.res.l	1	; scan pointer into the block
XOUT:	.equ $-UORIG
	.res.l	1	; cursor position
XSCR:	.equ $-UORIG
	.res.l	1	; disc screen being accessed ( O=terminal )
XOFSET:	.equ $-UORIG
	.res.l	1	; disc sector offset for multi-disc
XCONT:	.equ $-UORIG
	.res.l	1	; last word in primary search vocabulary
XCURR:	.equ $-UORIG
	.res.l	1	; last word in extensible vocabulary
XSTATE:	.equ $-UORIG
	.res.l	1	; flag for 'interpret' or 'compile' modes
XBASE:	.equ $-UORIG
	.res.l	1	; number base for I/O numeric conversion
XDPL:	.equ $-UORIG
	.res.l	1	; decimal point place
XFLD:	.equ $-UORIG
	.res.l	1	; 
XCSP:	.equ $-UORIG
	.res.l	1	; current stack position, for compile checks
XRNUM:	.equ $-UORIG
	.res.l	1	; 
XHLD:	.equ $-UORIG
	.res.l	1	; 
XDELAY:	.equ $-UORIG
	.res.l	1	; carriage return delay count
XCOLUM:	.equ $-UORIG
	.res.l	1	; carriage width
IOSTAT:	.equ $-UORIG
	.res.l	1	; last acia status from write/read
; ( 4 spares! )
	.res.l	1	; 
	.res.l	1	; 
	.res.l	1	; 
	.res.l	1	; 
;
UTABLESZ:	.equ $-UORIG


; Structure of the symbol table entry:
; Length byte of the symbol name (NFA)
; 	high bit set to flag the start byte
;	next bit is MIMM 
;	next bit is MHID
;	5 bits of length, 0 really does mean zero.
; Length - 1 bytes of name, 7 bit characters. (Woe unto ye who use UTF-8, shift-JIS, etc.)
; Last byte, high bit set to flag the end byte.
; In alignment sensitive architectures, 0 to alignment - 1 padding. (SH3 has 0 to 3.)
; Link to the previous name in the vocabulary. (LFA)
; Link to the (machine language level) characteristic definition for the symbol table entry.
;   With "low-level words", this just links to the code that follows. (CFA)
; Parameter field(s). (PFA)
;   For "low-level words", this is machine code.
;   For global constants and variables, this contains the value.
;   For per-USER (task) constants and variables, the offset into
;     the current per-USER table.
;   For "high-level words", this is a list of virtual instruction pointers,
;     in other words, pointers to the characterictics (CFA) of the symbols 
;     compiled into the definition.
;


; mode flags (to go with the length byte):
CTFLAG:	.equ H'80	; Count byte in dictionary has high bit set.
MIMM:	.equ H'40	; precedence is IMMEDIATE execution (in length byte -- NFA)
MHID:	.equ H'20	; SMUDGEd definition (in length byte -- NFA)
CTMASK:	.equ ( H'FF & ~( CTFLAG | MIMM ) )	; MHID hides SMUDGEd definitions this way.
; MHID limits length to 31
;
TAILFLAG:	.equ H'80	; Trailing character in dictionary has high bit set.
TAILMASK:	.equ ( H'FF & ~TAILFLAG )	; Expose the tail character.



_PREVNAME:	.assign	0	; allocation/dictionary link (terminated by zero)

	.macro	HIHEADER	name, characteristic, inherited, mode=0
; Symbol name length and mode (Too much stuff in one byte, really.)
	.data.b	(.len("\name")&H'1f)|\mode|H'80
; Symbol name
_s\characteristic:	.sdata	.substr("\name", 0, .len("\name")-1)
; Terminate the name with high bit set (bad news for multi-byte names).
	.data.b	.substr("\name", .len("\name")-1, 1)+H'80
; Symbol names float a little, yes this is a little awkward.
	.align	NATURAL_SIZE
; (Allocation) link to previously defined symbol's header.
_l\characteristic
	.data.l	_PREVNAME
; Use the SH-3 assembler to track the last symbol. 
; (This is not always a good idea.)
_PREVNAME:	.assign _s\characteristic
; Point to the characteristic code for this Word (symbol) to execute.
\characteristic	.equ	$
	.data.l	_f\inherited
; Point to the "parameter" area of the symbol.
_f\characteristic	.equ $
; This area will contain executable code for primitive (leaf) definitions.
; It will contain a list of virtual instructions for non-primitive (non-leaf) definitions.
; For language global/static constants and variables, it will contain the actual value.
; For per-User constants and variables, it will contain the offset in the user table.
; And so (ahem) forth.
	.endm

	.macro	HEADER	name, characteristic, mode=0
	HIHEADER	"\name", \characteristic, \characteristic, \mode
	.endm


; More as an example than to be actually used:
	.macro	fSAFECALL	cfa
	sts.l	pr, @-fRP
	mov.l	imm\@, fW
	jsr 	@fW
	nop
	bra 	exit\@
	nop
imm\@:
	.data.l		\cfa
exit\@:
	lds.l	@fRP+
	.endm


; Utility macros:
; These should help reduce errors.
	.macro	mTARGET	target
	.data.l	\target-$-NATURAL_SIZE
	.endm


; Convenience macros:
; *** Use these instead of stealing code. 
; Branch to address is limited to +- 4k, 
; and stolen code is easy to lose track of.
; Or, maybe you can let an optimizer move the code more reliably for you!

; See xPALIGN
; Immediate masks are only available for r0!
	.macro	mALIGNr0
	add 	#NATURAL_SIZE-1, r0		; Only works where NATURAL_SIZE is a power of 2.
	and 	#ALIGN_MASK, r0		; SH3's NATURAL_SIZE is 4, or 2^2, so it works.
	.endm



