﻿// Copyright (c) 2014 panacoran <panacoran@users.sourceforge.jp>
// This program is part of OmegaChart.
// OmegaChart is licensed under the Apache License, Version 2.0.

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using Zanetti.Data;

namespace Zanetti.DataSource.Specialized
{
    internal class KdbComDataSource : DailyDataSource
    {
        public KdbComDataSource(int[] dates) : base(dates)
        {
        }

        public override void Run()
        {
            GetMarketVolume(); // 先に東証一部の出来高を読み込む。
            var newdata = new Dictionary<int, Dictionary<int, NewDailyData>>();
            foreach (var date in _dates)
            {
                newdata[date] = FillData(date);
                SendMessage(AsyncConst.WM_ASYNCPROCESS, (date & DATE_MASK), AsyncConst.LPARAM_PROGRESS_SUCCESSFUL);
            }
            foreach (AbstractBrand br in Env.BrandCollection.Values)
            {
                if (br.Market == MarketType.B && !IsSupportedIndex(br.Code) || br.Market == MarketType.Custom)
                    continue;
                using (var farm = (DailyDataFarm)br.CreateDailyFarm(_dates.Length))
                {
                    var traceFlag = false;
                    foreach (var date in _dates)
                    {
                        NewDailyData td;
                        if (newdata[date].TryGetValue(br.Code, out td))
                        {
                            farm.UpdateDataFarm(date, td);                            
                        }
                        else
                        {
                            if (traceFlag)
                                continue;
                            traceFlag = true;
                            Debug.WriteLine("Data not found(k-db.com) : code=" + br.Code + " market=" + br.Market);
                        }
                        farm.Save(Util.GetDailyDataFileName(br.Code));
                    }
                    SendMessage(AsyncConst.WM_ASYNCPROCESS, br.Code, AsyncConst.LPARAM_PROGRESS_SUCCESSFUL);
                }
            }
        }

        private Dictionary<DateTime, int> _indexVolume;

        private void GetMarketVolume()
        {
            _indexVolume = new Dictionary<DateTime, int>();
            const string url = "http://k-db.com/site/toukei.aspx?market=T1&download=csv";
            using (var reader = new StreamReader(Util.HttpDownload(url), Encoding.GetEncoding("shift_jis")))
            {
                string line;
                while ((line = reader.ReadLine()) != null)
                {
                    var tokens = line.Split(',');
                    switch (tokens[0])
                    {
                        case "東証1部":
                        case "日付":
                            continue;
                    }
                    var date = DateTime.Parse(tokens[0]);
                    var volume = double.Parse(tokens[1]);
                    _indexVolume[date] = (int)(volume * 0.001);
                }
            }
        }
       
        private Dictionary<int, NewDailyData> FillData(int date)
        {
            var result = new Dictionary<int, NewDailyData>();
            var d2 = Util.IntToDate(date);
            var url = "http://k-db.com/site/download.aspx?p=all&download=csv&date=" + d2.ToString("yyyy-MM-dd");
            using (var reader = new StreamReader(Util.HttpDownload(url), Encoding.GetEncoding("shift_jis")))
            {
                string line;
                while ((line = reader.ReadLine()) != null)
                {
                    var tokens = line.Split(',');
                    int code;
                    int volume;
                    var pv = 1.0;
                    if (tokens[0] == "10" || tokens[0] == "67")
                    {
                        code = tokens[0] == "67" ? (int)BuiltInIndex.Nikkei225 : (int)BuiltInIndex.TOPIX;
                        volume = _indexVolume[d2];
                        pv = 100;
                    }
                    else if (tokens[0].Length == 6)
                    {
                        code = int.Parse(tokens[0].Substring(0, 4));
                        var br = Env.BrandCollection.FindBrand(code) as BasicBrand;
                        if (br == null || !CheckMarket(br.Market, tokens[2]))
                            continue;
                        volume = (int)ParseField(tokens[8]);
                    }
                    else
                    {
                        continue;
                    }
                    var td = new NewDailyData
                    {
                        volume = volume,
                        open = (int)(ParseField(tokens[4]) * pv),
                        high = (int)(ParseField(tokens[5]) * pv),
                        low = (int)(ParseField(tokens[6]) * pv),
                        close = (int)(ParseField(tokens[7]) * pv)
                    };
                    result[code] = td;
                }
            }
            return result;
        }

        private bool CheckMarket(MarketType market, string name)
        {
            switch (name)
            {
                case "東証":
                case "東証1部":
                    return market == MarketType.T1;
                case "東証2部":
                    return market == MarketType.T2;
                case "東証マザーズ":
                case "東証マザーズ外国":
                    return market == MarketType.M;
                case "東証TPM":
                    return market == MarketType.Custom;
                case "東証1部外国":
                    return market == MarketType.T1;
                case "大証":
                case "大証1部":
                    return market == MarketType.O1;
                case "大証2部":
                    return market == MarketType.O2;
                case "東証JQグロース":
                case "東証JQスタンダード":
                case "東証JQスタンダード外国":
                case "JQ":
                case "JQスタンダード":
                case "JQスタンダード外国":
                case "JQグロース":
                case "JQNEO":
                    return market == MarketType.J;
                case "HCスタンダード":
                case "HCスタンダード外国":
                case "HCグロース":
                    return market == MarketType.H;
                case "福証":
                case "福証QBoard":
                case "札証":
                case "札証アンビシャス":
                case "名証":
                    return false;
            }
            return false;
        }

        private bool IsSupportedIndex(int code)
        {
            return code == (int)BuiltInIndex.Nikkei225 ||
                   code == (int)BuiltInIndex.TOPIX;
        }

        private double ParseField(string s)
        {
            // 無効なフィールドを示す"-"を0として扱う。
            return s == "-" ? 0 : double.Parse(s);
        }
    }
}