package org.dyndns.nuda.plugin;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;

import org.dyndns.nuda.di.DIManager;
import org.dyndns.nuda.logger.CommonLogger;
import org.dyndns.nuda.logger.LoggerAdaptor;
import org.dyndns.nuda.tools.util.StringUtil;

/**
 * Java標準のサービスプロバイダ機構を用いたプラグイン機能を提供します
 * 
 * @author koseki
 * 
 */
public class PluginLoader {
	// クラスローダ上のサービス参照プリフィックス
	private static final String CLASS_PATH_PREFIX = "META-INF/services";

	// ロガー
	private static LoggerAdaptor logger = CommonLogger.getLoggerAdaptor()
			.getLogger(PluginLoader.class);
//	private static LoggerAdaptor logger = new StandardConsoleLoggerAdaptor().getLogger("SysPluginLoader");

	public PluginLoader() {
		SysPluginLoader.initializePluginLoader();
	}
	
	/**
	 * クラスローダより、プラガブルなクラスをロードし初期化を行います。<br />
	 * 
	 * このメソッドは、デフォルトでsun.misc.Serviceクラスを利用してプラグインの起動を行います。<br />
	 * もし、上記クラスがクラスローダ上に存在しない場合は、META-INF/
	 * services配下のサービスプロバイダ定義を用いてプラグインの起動を行います。<br />
	 * 
	 * @param pluginClass
	 *            プラグインクラス
	 * 
	 * @throws IllegalArgumentException
	 *             引数がnullまたは有効なJavaインタフェースクラスでない場合にロードされます
	 */
	public void loadPlugin(Class<? extends Plugable> pluginClass) {
		
		if (pluginClass == null) {
			Exception e = new IllegalArgumentException("引数がnullです");
			logger.error("エラー:", e);
			
			return;
		}
		if (!pluginClass.isInterface()) {
			String message = StringUtil.format(
					"クラス{}は有効なプラグインインタフェースクラスではありません",
					pluginClass.getCanonicalName());
			Exception e = new IllegalArgumentException(message);
			logger.error("エラー:", e);
			//e.printStackTrace();
			return;
		}

		if (isExistsSunMiscServiceProviderArchitecture()) {

			@SuppressWarnings({ "rawtypes" })
			Iterator i = sun.misc.Service.providers(pluginClass);

			int count = 0;
			while (i.hasNext()) {
				Object instance = null;

				try {
					instance = i.next();
					PluginDescription desc = instance.getClass().getAnnotation(PluginDescription.class);
//					if (instance.getClass().isAnnotationPresent(
//							PluginDescription.class)) {
					if (desc != null) {
//						PluginDescription desc = instance.getClass()
//								.getAnnotation(PluginDescription.class);
						logger.debug(
								"sun.miscパッケージのサービスプロバイダ機能を使用して、プラグイン[{}]のロードを行います",
								desc.name());
					} else {
						logger.debug("sun.miscパッケージのサービスプロバイダ機能を使用して、プラグイン[{}]のロードを行います",
								instance.getClass().getCanonicalName());
					}
				} catch (Throwable t) {

					if (t.getCause() instanceof InstantiationException) {
						InstantiationException ie = (InstantiationException) t
								.getCause();

						logger.error(
								"[{}]のプラグインエントリ：クラス[{}]のインスタンス化に失敗しました",
								"META-INF/services/"
										+ pluginClass.getCanonicalName(),
								ie.getMessage());
					} else {
						// クラスが見つからなかった場合など
						logger.error("エラー:", t);
					}

					continue;
				}
				try {
					Plugable plugable = Plugable.class.cast(instance);

					DIManager.inject(plugable);

					plugable.init();

					logger.debug(
							"[{}]のプラグインエントリ：プラグインクラス[{}]のロードに成功しました",
							"META-INF/services/"
									+ pluginClass.getCanonicalName(), plugable
									.getClass().getCanonicalName());

					count++;
				} catch (Throwable e) {
					// ClassNotFoundException
					// InstantiationException
					// ServiceConfigurationError
					if (e instanceof ClassCastException) {
						logger.error(
								"[{}]のプラグインエントリ：クラス[{}]は有効なプラグインインタフェースを実装しておりません",
								"META-INF/services/"
										+ pluginClass.getCanonicalName(),
								instance.getClass().getCanonicalName());
					} else {
						logger.error("エラー:", e);
					}
				}
			}
			if (count == 0) {
				logger.debug("{}", "ロード対象プラグインが存在しませんでした[{}]", pluginClass.getCanonicalName());
			}
			return;
		} else {
			// sun.miscパッケージが参照できない場合(oracleのJava標準ライブラリ未使用の場合:IBMのVMを使用している場合など)
			logger.debug("{}",
					"CommonToolsライブラリ独自のサービスプロバイダ機能を使用して、プラグインのロードを行います");

			if (pluginClass.isAnnotationPresent(PluginDescription.class)) {
				PluginDescription desc = pluginClass
						.getAnnotation(PluginDescription.class);
				logger.debug(
						"CommonToolsライブラリ独自のサービスプロバイダ機能を使用して、プラグイン[{}]のロードを行います",
						desc.name());
			} else {
				logger.debug("{}",
						"CommonToolsライブラリ独自のサービスプロバイダ機能を使用して、プラグインのロードを行います");
			}

			String clsName = pluginClass.getCanonicalName();

			// コンテキストクラスローダを取得
			ClassLoader cl = Thread.currentThread().getContextClassLoader();

			InputStream is = null;
			InputStreamReader isr = null;
			BufferedReader br = null;

			// クラスローダ上のパスを組み立て
			StringBuilder b = new StringBuilder();
			b.append(CLASS_PATH_PREFIX);
			b.append("/");
			b.append(clsName);
			String path = b.toString();

			try {
				// リソースをロードする
				Enumeration<URL> urls = cl.getResources(path);

				// ロード対象クラスが格納されるリスト
				List<String> classList = new ArrayList<String>();

				// 取得したURL分ループする
				while (urls.hasMoreElements()) {
					try {
						// URLからストリームをオープンし、コンテンツを取得する
						URL url = urls.nextElement();
						is = url.openStream();
						isr = new InputStreamReader(is);
						br = new BufferedReader(isr);

						while (br.ready()) {
							String line = br.readLine();
							classList.add(line);
						}

					} finally {
						if (is != null) {
							is.close();
						}
						if (isr != null) {
							isr.close();
						}
						if (br != null) {
							br.close();
						}
					}
				}

				// クラスリストのサイズが0以上の場合にプラグインのロード処理を行う
				if (classList.size() > 0) {
					// サービス(プラグイン)ロード処理
					for (String aClsName : classList) {
						Class<?> plugin = null;
						try {
							plugin = cl.loadClass(aClsName);
						} catch (ClassNotFoundException e) {
							logger.error(
									"[{}]のプラグインエントリ：クラスパス上にクラス[{}]が存在しませんんでした",
									"META-INF/services/" + aClsName,
									e.getMessage());
							continue;
						}

						boolean isAssignable = pluginClass
								.isAssignableFrom(plugin);

						if (isAssignable) {
							Object instance = null;
							try {
								instance = plugin.newInstance();
							} catch (InstantiationException e) {
								logger.error(
										"[{}]のプラグインエントリ：クラス[{}]のインスタンス化に失敗しました:引数なしのパブリックコンストラクタが存在しません",
										"META-INF/services/" + aClsName,
										e.getMessage());
								continue;
							} catch (IllegalAccessException e2) {
								// プラグインのコンストラクタにアクセスできない
								logger.error(
										"[{}]のプラグインエントリ：クラス[{}]のインスタンス化に失敗しました:引数なしのパブリックコンストラクタが存在しません",
										"META-INF/services/" + aClsName,
										e2.getMessage());
								continue;
							}

							Plugable p = (Plugable) instance;
							DIManager.inject(p);
							// プラグインの初期化を実行
							p.init();
						}
						if (!isAssignable) {
							logger.error(
									"[{}]のプラグインエントリ：クラス[{}]は有効なプラグインインタフェースを実装しておりません",
									"META-INF/services/"
											+ pluginClass.getCanonicalName(),
									aClsName);
						}
					}

				}
			} catch (IOException e2) {
				// 指定したURLが示すリソースが存在しない
				logger.error("エラー:", e2);
			}
		}

	}

	/**
	 * Java標準ライブラリで提供されたサービスプロバイダロード機構を、コンテキストクラスローダが提供可能かを検査します
	 * 
	 * @return
	 */
	private static boolean isExistsSunMiscServiceProviderArchitecture() {
		// return false;
		try {
			Class.forName("sun.misc.Service");
			return true;
		} catch (Exception e) {
			return false;
		}
	}
}
