package org.dyndns.nuda.dynamic.compiler.resolver;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.tools.JavaFileObject.Kind;

import org.dyndns.nuda.dynamic.compiler.SourceBean;
import org.dyndns.nuda.dynamic.compiler.SourceResolver;
import org.dyndns.nuda.dynamic.compiler.resolver.renderer.SourceRenderer;

/**
 * 文字列をコンパイルソースとするJavaSourceのためのリゾルバクラスです.<br />
 * 
 * <ul>
 * このクラスは以下のソースオブジェクトを受け入れ可能です
 * <li>
 * java.lang.String</li>
 * <li>
 * java.lang.StringBuilder</li>
 * <li>
 * java.io.File</li>
 * <li>
 * java.io.StringWriter</li>
 * <li>
 * org.dyndns.nuda.dynamic.compiler.resolver.VelocityTemplateSource<br />
 * (Velocity)</li>
 * </ul>
 * 
 * <ul>
 * 以下のクラスは導入検討中のクラスです
 * <li>java.io.FileInputStream</li>
 * </ul>
 * 
 * @author nkoseki
 * 
 */
public class TextSourceResolver implements SourceResolver {
	
	// ----------------------------------------------------
	private static final String		STR_PTN_PACKAGE			= "package (.+?);";
	private static final String		STR_PTN_CLASS			=
																".+? class (.+?) (extends|implements)?";
	private static final int		INT_IDX_PACKAGE			= 1;
	private static final int		INT_IDX_CLASS			= 1;
	
	private static final Pattern	PTN_PACKAGE				=
																Pattern
																	.compile(
																		STR_PTN_PACKAGE,
																		Pattern.DOTALL);
	private static final Pattern	PTN_CLASS				= Pattern.compile(
																STR_PTN_CLASS,
																Pattern.DOTALL);
	
	// ----------------------------------------------------
	private String					source					= "";
	private String					packageName				= "";
	private String					simpleClassName			= "";
	private Object					target					= null;
	
	private SourceRenderer			stringRenderer			=
																new StringRenderer();
	private SourceRenderer			stringWriterRenderer	=
																new StringWriterRenderer();
	private SourceRenderer			fileRenderer			=
																new FileRenderer();
	private SourceRenderer			stringBuilderRenderer	=
																new StringBuilderRenderer();
	
	//	private SourceRenderer			velocityRenderer		=
	//																new VelocityTemplateRenderer();
	
	// ----------------------------------------------------
	public TextSourceResolver() {
	}
	
	public TextSourceResolver(final String source) {
		this.source = source;
	}
	
	public TextSourceResolver(final File source) {
		String tmpSource = "";
		
		this.source = tmpSource;
	}
	
	public TextSourceResolver(final StringWriter writer) {
		this.source = writer.toString();
	}
	
	// ----------------------------------------------------
	@Override
	public SourceBean resolve() {
		// 許容型コンテンツをjava.lang.Stringに変換する
		if (this.target != null) {
			if (this.stringRenderer.accept(this.target)) {
				this.source = this.stringRenderer.render(this.target);
			} else if (this.stringWriterRenderer.accept(this.target)) {
				this.source = this.stringWriterRenderer.render(this.target);
			} else if (this.fileRenderer.accept(this.target)) {
				this.source = this.fileRenderer.render(this.target);
			} else if (this.stringBuilderRenderer.accept(this.target)) {
				this.source = this.stringBuilderRenderer.render(this.target);
			}
			//			else if (this.velocityRenderer.accept(this.target)) {
			//				this.source = this.velocityRenderer.render(this.target);
			//			}
		} else {
			if (this.source == null) {
				return new NOPResolver().resolve();
			}
		}
		
		// パッケージ名を抽出
		Matcher packageNameMatcher = PTN_PACKAGE.matcher(this.source);
		if (packageNameMatcher.find()) {
			this.packageName = packageNameMatcher.group(INT_IDX_PACKAGE);
		}
		
		// クラス名を抽出
		Matcher classNameMatcher = PTN_CLASS.matcher(this.source);
		if (classNameMatcher.find()) {
			this.simpleClassName = classNameMatcher.group(INT_IDX_CLASS);
		}
		
		SourceBean result = new SourceBean();
		result.setPackageName(this.packageName);
		result.setSimpleClassName(this.simpleClassName);
		result.setSourceCode(this.source);
		
		return result;
	}
	
	@Override
	public String toString() {
		return "[package="
			+ this.packageName
			+ "][class="
			+ this.simpleClassName
			+ "]";
	}
	
	@Override
	public boolean accept(final Object target) {
		if (this.stringRenderer.accept(target)) {
			return true;
		} else if (this.stringWriterRenderer.accept(target)) {
			return true;
		} else if (this.fileRenderer.accept(target)) {
			return true;
		} else if (this.stringBuilderRenderer.accept(target)) {
			return true;
		}
		//		else if (this.velocityRenderer.accept(target)) {
		//			return true;
		//		}
		//System.out.println("not acceptable");
		return false;
	}
	
	private class StringRenderer implements SourceRenderer {
		
		@Override
		public boolean accept(final Object target) {
			if (target == null) {
				return false;
			}
			
			// Stringテスト
			if (target instanceof String) {
				String target0 = (String) target;
				if (target0.isEmpty()) {
					return false;
				} else {
					return true;
				}
			} else {
				return false;
			}
		}
		
		@Override
		public String render(final Object target) {
			return (String) target;
		}
		
	}
	
	private class StringBuilderRenderer implements SourceRenderer {
		
		@Override
		public boolean accept(final Object target) {
			if (target == null) {
				return false;
			}
			
			// Stringテスト
			if (target instanceof StringBuilder) {
				StringBuilder target0 = (StringBuilder) target;
				if (target0.toString().isEmpty()) {
					return false;
				} else {
					return true;
				}
			} else {
				return false;
			}
		}
		
		@Override
		public String render(final Object target) {
			StringBuilder b = (StringBuilder) target;
			return b.toString();
		}
		
	}
	
	private class StringWriterRenderer implements SourceRenderer {
		
		@Override
		public boolean accept(final Object target) {
			if (target == null) {
				return false;
			}
			
			// StringWriterテスト
			if (target instanceof StringWriter) {
				StringWriter target0 = (StringWriter) target;
				if (target0.toString().isEmpty()) {
					return false;
				} else {
					return true;
				}
			} else {
				return false;
			}
		}
		
		@Override
		public String render(final Object target) {
			StringWriter sw = (StringWriter) target;
			String result = sw.toString();
			sw.flush();
			try {
				sw.close();
				return result;
			} catch (IOException e) {
				e.printStackTrace();
				return "";
			}
			
		}
		
	}
	
	private class FileRenderer implements SourceRenderer {
		
		@Override
		public boolean accept(final Object target) {
			if (target == null) {
				return false;
			}
			
			// ファイルテスト
			if (target instanceof File) {
				File target0 = (File) target;
				if (!target0.exists()) {
					return false;
				} else {
					if (target0.isDirectory()) {
						return false;
					} else {
						String fileName = target0.getName();
						fileName = fileName.toLowerCase();
						if (fileName.indexOf(Kind.SOURCE.extension) == -1) {
							return false;
						} else {
							return true;
						}
					}
				}
			} else {
				return false;
			}
		}
		
		@Override
		public String render(final Object target) {
			File file = (File) target;
			
			FileInputStream fis = null;
			InputStreamReader isr = null;
			BufferedReader br = null;
			String result = "";
			StringWriter sw = null;
			PrintWriter pw = null;
			
			try {
				try {
					fis = new FileInputStream(file);
					isr = new InputStreamReader(fis);
					br = new BufferedReader(isr);
					sw = new StringWriter();
					pw = new PrintWriter(sw);
					while (br.ready()) {
						
						pw.println(br.readLine());
					}
					
					result = sw.toString();
					
					return result;
				} finally {
					if (fis != null) {
						fis.close();
					}
					if (isr != null) {
						isr.close();
					}
					if (br != null) {
						br.close();
					}
					if (pw != null) {
						pw.flush();
						pw.close();
					}
					if (sw != null) {
						sw.flush();
						sw.close();
					}
				}
			} catch (IOException e) {
				e.printStackTrace();
				return result;
			}
		}
		
	}
	
	//	private class VelocityTemplateRenderer implements SourceRenderer {
	//		
	//		@Override
	//		public boolean accept(final Object target) {
	//			if (target == null) {
	//				return false;
	//			}
	//			
	//			// StringWriterテスト
	//			if (target instanceof VelocityTemplateSource) {
	//				VelocityTemplateSource src = (VelocityTemplateSource) target;
	//				if (src.vctx == null || src.template == null) {
	//					return false;
	//				} else {
	//					return true;
	//				}
	//			} else {
	//				return false;
	//			}
	//		}
	//		
	//		@Override
	//		public String render(final Object target) {
	//			VelocityTemplateSource template = (VelocityTemplateSource) target;
	//			
	//			StringWriter sw = new StringWriter();
	//			
	//			template.template.merge(template.vctx, sw);
	//			String result = sw.toString();
	//			sw.flush();
	//			try {
	//				try {
	//					return result;
	//				} finally {
	//					if (sw != null) {
	//						sw.close();
	//					}
	//				}
	//			} catch (IOException e) {
	//				e.printStackTrace();
	//				return null;
	//			}
	//		}
	//		
	//	}
	
	@Override
	public void setTarget(final Object target) {
		this.target = target;
	}
	
	@Override
	public Object getTarget() {
		return this.target;
	}

	@Override
	public void init() {
		AutoResolver.addResolver(this);
	}
	
}
