/**
 * A simple tool to help identify which pixiv accounts you've
 * downloaded media from, have since unfollowed, but haven't removed
 * the directory.
 * 
 * Bugs:
 *  If you encounter any errors, please let me know.
 *  I don't include any form of automated error reporting,
 *  so please consider attaching the "pixiv_remove_unfollowed.log"
 *  file.
 */
module app;

import core.stdc.stdlib : exit;

import std.experimental.logger : FileLogger;

import std.container : DList;
import std.file;
import std.path;
import std.stdio;
import std.string;

import mlib.configparser;
import mlib.directories;

import pixivd;
import pixivd.types;

struct Context
{
	string phpsessid;
	string pixivBaseDir;
}

private FileLogger logger;

// Initialize a Context.
// Could exit the program if it failed to find the user configuration directory,
// or if the user doesn't enter a PHPSESSID value when prompted.
Context fetchContext()
{
	Context ctx;

	DirEntry configDir = open(Directory.config);
	if (configDir.name.length == 0) {
		logger.critical("directories.d open(Directory.config) failed to return a valid DirEntry.");
		stderr.writeln("Failed to determine your configuration directory, please consider setting the" ~
			" XDG_CONFIG_HOME environment variable.");
		exit(1);
	}

	auto phpsessidPath = buildPath(configDir.name, "pixivd", "phpsessid.txt");
	string phpsessid;

	if (false == exists(phpsessidPath)) {
		mkdirRecurse(buildPath(configDir.name, "pixivd"));
		writeln("PHPSESSID not found, please enter:");
		phpsessid = readln().strip();
		if (phpsessid.empty) {
			stderr.writeln("No PHPSESSID entered, exiting.");
			exit(1);
		}
		auto f = File(phpsessidPath, "w");
		f.writeln(phpsessid);
		f.close();

		ctx.phpsessid = phpsessid;
		return ctx;
	} else {
		auto phpsessidFile = File(phpsessidPath, "r");
		phpsessid = phpsessidFile.readln().strip();

		if (phpsessid.empty) {
			writeln("PHPSESSID not found, please enter:");
			phpsessid = readln().strip();
			if (phpsessid.empty) {
				stderr.writeln("No PHPSESSID entered, exiting.");
				exit(1);
			}
			phpsessidFile.writeln(phpsessid);
		}
		phpsessidFile.close();
	}

	ctx.phpsessid = phpsessid;
	return ctx;
}

string getPixivPictureDir()
{
	auto configDirPath = open(Directory.config);
	auto configFilePath = buildPath(configDirPath.name, "pixiv_down", "settings.conf");

	if (false == exists(configFilePath)) {
		logger.warningf("\"%s\" does not exist.", configFilePath);
		return "";
	}

	scope configparser = new ConfigParser();
	configparser.read(configFilePath);

	return configparser.get("output", "base_folder");
}

DList!string processDirs(string pixivDirPath)
{
	import std.ascii : isDigit;

	auto dirs = dirEntries(pixivDirPath, SpanMode.shallow);
	auto uids = DList!string();

	foreach(dir; dirs) {
		auto bname = baseName(dir);
		if (bname.length == 0 || false == isDigit(bname[0])) {
			continue;
		}
		auto uid = bname.split('_')[0];
		uids.insert(uid);
	}

	return uids;
}

void removeUserId(in ref Context ctx, string id) {
	foreach(dir; dirEntries(ctx.pixivBaseDir, SpanMode.shallow)) {
		string dirname = baseName(dir.name);
		if (dirname.startsWith(id)) {
			rmdirRecurse(dir);
			writefln("Deleted directory: %s", dirname);
		}
	}
}

void removeUnfollowed(in ref Context ctx, DList!string uids)
{
	import core.thread : Thread;
	import core.time : seconds;

	auto client = new Client(ctx.phpsessid);

	foreach (id; uids) {
		FullUser user;
		try {
			user = client.fetchUser(id);
		} catch (PixivJSONException pje) {
			logger.warningf("(PixivJSONException) %s", pje.msg);
			writefln("User ID %s has either left pixiv, or does not exist.", id);
			write("Do you want to remove the related directories? [y/N]: ");
			string input = readln().strip().toLower();

			if (input == "y" || input == "yes") {
				removeUserId(ctx, id);
			}
			continue;
		} catch (Exception e) {
			logger.critical(e.msg);

			stderr.writefln("Error encountered fetching user %s. No files were modified.", id);
			continue;
		}

		if (false == user.following) {
			writefln("Found user not followed: %s (%s)", user.userName, user.userId);
			write("Do you want to remove the related directories? [y/N]: ");
			string input = readln().strip().toLower();

			if (input == "y" || input == "yes") {
				writeln("Will remove user data.");
				removeUserId(ctx, id);
			}
		} else {
			writefln("Found followed user: %s", user.userName);
		}
		writeln("Sleeping 10 seconds");
		Thread.sleep(10.seconds);
	}
}


int main(string[] args)
{
	logger = new FileLogger("pixiv_remove_unfollowed.log");

	Context ctx = fetchContext();

	ctx.pixivBaseDir = getPixivPictureDir();
	if (0 == ctx.pixivBaseDir.length) {
		stderr.writeln("Error: Could not retrieve pixiv_down configuration file. " ~
			"Possibly because pixiv_down hasn't been used before.");
		return 1;
	}

	DList!string uids = processDirs(ctx.pixivBaseDir);
	if (uids.empty) {
		writefln("No users found in %s!", ctx.pixivBaseDir);
		return 0;	
	}

	removeUnfollowed(ctx, uids);
	return 0;
}
