/*
 * This is free and unencumbered software released into the public domain.
 *
 * Anyone is free to copy, modify, publish, use, compile, sell, or distribute this
 * software, either in source code form or as a compiled binary, for any purpose,
 * commercial or non-commercial, and by any means.
 *
 * In jurisdictions that recognize copyright laws, the author or authors of this
 * software dedicate any and all copyright interest in the software to the public
 * domain. We make this dedication for the benefit of the public at large and to
 * the detriment of our heirs and successors.  We intend this dedication to be an
 * overt act of relinquishment in perpetuity of all present and future rights to
 * this software under copyright law.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */


/**
 * Common 'Trash' operations for the computers Recycle Bin.
 *
 * Only for POSIX systems (currently) and follows the XDG specification.
 *
 * Authors: mio <stigma@disroot.org>
 * Date: January 29, 2023
 * License: public domain
 * Standards: The FreeDesktop.org Trash Specification 1.0
 * Version: 0.1.0
 *
 * Macros:
 *   DREF = <a href="https://dlang.org/phobos/$1.html#$2">$2</a>
 *   LREF = <a href="#$1">$1</a>
 */
module mlib.trash;

import core.stdc.errno;

import std.file;
import std.path;
import std.process : environment;
import std.stdio;

/*
 * Permanetely delete all trashed records.
 *
 * This currently throws an Exception as it's not yet implemented.
 */
// void emptyTrash()
// {
//     throw new Exception(__PRETTY_FUNCTION__ ~ " not implemented");
// }

/*
 * Restore one (or all: "") trashed records
 *
 * Params:
 *  pathInTrash = The unique filename in the trash directory to
 *                restore.  By not providing an argument (or by
 *                passing `""`) this will restore _all_ files.
 *
 * Note: This currently throws an Exception as it's not yet
 *       implemented.
 */
// void restoreTrash(string pathInTrash = "")
// {
//     throw new Exception(__PRETTY_FUNCTION__ ~ " not implemented");
// }

/*
 * List all the files and directories currently inside the trash.
 *
 * Returns: A list of strings containing every filename in the trash.
 *
 * Note: This currently throws an Exception as it's not yet
 *       implemented.
 */
// string[] listTrash()
// {
//     throw new Exception(__PRETTY_FUNCTION__ ~ " not implemented");
// }


/**
 * Trash the file or directory at *path*.
 *
 * Params:
 *  path = The path to move to the trash.
 *
 * Throws:
 *  - $(DREF std_file, FileException) if the file cannot be trashed.
 */
void trash(string path)
{
    scope string pathInTrash;
    trash(path, pathInTrash);
}

/**
 * Trash the file or directory at *path*, and sets *pathInTrash* to the
 * path at which the file can be found within the trash.
 *
 * Params:
 *  path = The path to move to the trash.
 *  pathInTrash = The path at which the newly trashed item can be found.
 *
 * Throws:
 *  - $(DREF std_file, FileException) if the file cannot be trashed.
 */
void trash(string path, out string pathInTrash)
{
    version (Posix) {
        _posix_trash(path, pathInTrash);
    } else {
        throw new Exception(__PRETTY_FUNCTION__ ~ " is not supported on your OS");
    }
}

/**
 * Erase the file from the operating system.
 *
 * This skips the "trashing" operation and unlinks the file from the
 * system and recovers the space.  Files which have been erased are
 * not recoverable.
 *
 * Throws:
 *  - $(DREF std_file, FileException) if the file cannot be removed.
 */
void erase(string path)
{
    // Really just a convenience function.
    remove(path);
}


private:

/*
 * System independant functions.
 * These will call the system specific function.
 */

ulong getDevice(string path) {
    version (Posix) {
        return _posix_getDevice(path);
    }
}

string getHomeDirectory() {
    version (Posix) {
        return environment["HOME"];
    }
}

bool isParent(string parent, string path) {
    import std.string : startsWith;

    path = path.absolutePath;
    parent = parent.absolutePath;

    return startsWith(path, parent);
}

string getInfo(string src, string topdir) {
    import std.uri : encode;
    import std.datetime.systime : Clock;

    if (false == isParent(topdir, src)) {
        src = src.absolutePath;
    } else {
        src = relativePath(src, topdir);
    }

    string info = "[Trash Info]\n";
    info ~= "Path=" ~ encode(src) ~ "\n";


    /*
     * Prior to D 2.099.0, the toISOExtString method didn't
     * have a precision argument, which means it includes
     * fractional seconds by default.  So to accommodate
     * for earlier versions, just trim it off.
     */
    static if (__VERSION__ < 2099L) {
        import std.string : split;

        string dateTime = Clock.currTime.toISOExtString().split(".")[0];

        info ~= "DeletionDate=" ~ dateTime ~ "\n";
    } else {
        info ~= "DeletionDate=" ~ Clock.currTime.toISOExtString(0) ~ "\n";
    }


    return info;
}

/*
 * System specific implementation of the above functions.
 */

version(Posix) {

    import core.sys.posix.sys.stat;
    import std.conv : to;
    import std.string : toStringz;

    void _posix_trash(string path, out string pathInTrash) {
        if (false == exists(path)) {
            throw new FileException(path, ENOENT);
        }

        /*  "When trashing a file or directory, the implementation SHOULD check
         *   whether the user has the necessary permissions to delete it, before
         *   starting the trashing operation itself". */
        uint attrs = getAttributes(path);
        if (false == ((S_IRUSR & attrs) && (S_IWUSR & attrs))) {
            throw new FileException(path, EACCES);
        }

        ulong pathDev = getDevice(path);
        ulong trashDev = getDevice(getHomeDirectory());


        // $topdir
        string topdir;
        // $trash
        string trash;

        /* w.r.t. homeTrash:
         *  "Files that the user trashes from the same file system (device/partition) SHOULD
         *   be stored here ... If this directory is needed for a trashing operation but does
         *   not exist, the implementation SHOULD automatically create it, without warnings
         *   or delays. */
        if (pathDev == trashDev) {
            topdir = _xdg_datahome();
            trash = buildPath(topdir, "Trash");
        } else {
            /*  "The implementation MAY also support trashing files from the rest of the
             *   system (including other partitions, shared network resources, and removable
             *   devices) into the "home trash" directory."
             *
             * I can only really test the partitions and removable devices, but I don't
             * have my desktop setup with multiple partitions.  Will check with removable
             * devices, but want to see same file system usage work first. */
            throw new Exception("The device for the Trash directory and the device for the path are different.");
        }

        string basename = baseName(path);
        string filename = stripExtension(basename);
        string ext = extension(basename);

        // $trash/files
        string filesDir = buildPath(trash, "files");
        if (false == exists(filesDir)) {
            mkdirRecurse(filesDir);
        }

        // $trash/info
        string infoDir = buildPath(trash, "info");
        if (false == exists(infoDir)) {
            mkdirRecurse(infoDir);
        }

        /*  "The names in [$trash/files and $trash/info] are to be determined by the
         *   implementation; the only limitation is that they must be unique within the
         *   directory. Even if a file with the same name and location gets trashed many times,
         *   each subsequent trashing must not overwrite a previous copy." */
        size_t counter = 0;
        string destname = basename;
        string infoFilename = destname.setExtension(".trashinfo");
        while (exists(buildPath(filesDir, destname)) || exists(buildPath(infoDir, infoFilename))) {
            counter += 1;
            destname = filename ~ "_" ~ to!string(counter) ~ ext;
            infoFilename = destname.setExtension(".trashinfo");
        }

        {
            /* "When trashing a file or directory, the implementation MUST create the
             * corresponding file in $trash/info first." */
            auto infoFile = File(buildPath(infoDir, infoFilename), "w");
            infoFile.write(getInfo(path, topdir));
        }
        {
            string filesPath = buildPath(filesDir, destname);
            rename(path, filesPath);
            pathInTrash = filesPath;
        }

        /* TODO: Directory size cache */
    }

    ulong _posix_getDevice(string path) {
        stat_t statbuf;
        lstat(toStringz(path), &statbuf);

        return statbuf.st_dev;
    }

    string _xdg_datahome()
    {
        if ("XDG_DATA_HOME" in environment) {
            return environment["XDG_DATA_HOME"];
        } else {
            return buildPath(environment["HOME"], ".local", "share");
        }
    }
} // End of version(Posix)
