/*
 * This is free and unencumbered software released into the public domain.
 *
 * Anyone is free to copy, modify, publish, use, compile, sell, or distribute this
 * software, either in source code form or as a compiled binary, for any purpose,
 * commercial or non-commercial, and by any means.
 *
 * In jurisdictions that recognize copyright laws, the author or authors of this
 * software dedicate any and all copyright interest in the software to the public
 * domain. We make this dedication for the benefit of the public at large and to
 * the detriment of our heirs and successors.  We intend this dedication to be an
 * overt act of relinquishment in perpetuity of all present and future rights to
 * this software under copyright law.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/**
 * This module provides quick & easy access to the XDG folder locations on GNU/Linux.  Currently,
 * only POSIX (XDG Base Directory Specification) is supported.  OS X (Standard Directories) and
 * Windows (Known Folder) will be supported at a later date.
 *
 * The main goal of this module is to provide a minimal and simple API.
 *
 * API
 * ---
 * enum Directory
 *   {
 *    home,
 *    config,
 *    data,
 *    cache,
 *   }
 *
 * DirEntry open(Directory);
 * ---
 *
 * This module supports D version greater than or equal to 2.068.0.
 *
 * Authors: mio <stigma@disroot.org>
 * Bugs: Doesn't check if the environment variable value is empty.
 * Date: June 12, 2021
 * License: public domain
 * Standards: XDG Base Directory Specification 0.8
 * Version: 0.X.X
 */
module mlib.directories;

import std.file : DirEntry;

public enum Directory
  {
   home,
   data,
   config,
   state,
   //   dataDirs,   /* XDG */
   //   configDirs, /* XDG */
   cache,
   runtime,
  }

/++
 Return a DirEntry pointing to `dt`.
 
 If the folder couldn't be found, an empty DirEntry is returned.
 
 Examples:
 ----
 import directories;
 import std.stdio;

 int main()
 {
   DirEntry userConfDir = open(Directory.config);
   /* Handle error */
   if (0 == userConfDir.name.length)
     {
       stderr.writeln("Failed to find user config directory.");
       return 1;
     }
   writefln("User config directory is %s", userConfDir.name);
   return 0;
 }
 ----
+/
public DirEntry open(Directory dt) nothrow// @safe
{
  /* 
   * Posix covers a few operating systems, if yours has a alternate
   * structure that is preferred over XDG, let me know.
   */
  version (Posix) enum supported = true;
  else enum supported = false;
  /* More operating systems will be supported soon-ish. */

  static if (false == supported)
    {
      import core.stdc.stdio : fprintf, stderr;

      fprintf(stderr, "*** error: The operating system you're running isn't supported. ***\n");
      // TODO: webpage for general contribution guide reporting.
      assert(false, "Unsupported platform.");
    }

  immutable string path = getPath(dt);
  if (path is null) return DirEntry();
  try {
    return DirEntry(path);
  } catch (Exception e) {
    return DirEntry();
  }
}

///
unittest
{
  import std.process : environment;
  import std.path : buildPath;

  /*
   * Environment Variables (for checking against)
   * Currently only tests on Posix since that's the only supported
   * platform.
   */
  immutable string homeE = environment.get("HOME");
  immutable string configE = environment.get("XDG_CONFIG_HOME",
					     buildPath(homeE, ".config"));
  immutable string dataE = environment.get("XDG_DATA_HOME",
					   buildPath(homeE, ".local", "share"));
  immutable string cacheE = environment.get("XDG_CACHE_HOME",
					    buildPath(homeE, ".cache"));

  // Compare against folders.d (note that folders is cross-platform, so
  // some errors may occur on non-POSIX)
  assert(homeE == open(Directory.home));
  assert(configE == open(Directory.config));
  assert(dataE == open(Directory.data));
  assert(cacheE == open(Directory.cache));
  import std.stdio;
  writeln(typeof(open(Directory.cache)).stringof);
}

private:

void errLog(string msg) nothrow @trusted
{
  import core.stdc.stdio : fprintf, stderr;
  
  // TODO: webpage for general issue reporting.
  // fprintf(stderr, "**  info: report bugs to https://yume-neru.neocities.org/bugs.html **\n");
  fprintf(stderr, "*** error: %s ***\n", msg.ptr);
}

immutable(string) getPath(in Directory dt) nothrow @safe
{
  switch (dt)
    {
    case Directory.home:
      return home();
    case Directory.data:
      return data();
    case Directory.config:
      return config();
    case Directory.cache:
      return cache();
    default:
      assert(false, "Unsupported Directory");
    }
}

immutable(string) home() nothrow @trusted
{
  import std.path : isAbsolute;
  import std.process : environment;
    
  string homeE;
    
  try {
    homeE = environment.get("HOME");
  } catch (Exception e) {
    homeE = null;
  }
    
  if (homeE is null) {
    import std.string : fromStringz;
    const(char)* pwdHome = fallbackHome();
    if (pwdHome !is null)
      homeE = cast(string)(pwdHome.fromStringz).dup;
    if (false == homeE.isAbsolute)
      homeE = null;
  }
  return homeE;
}

immutable(string) data() nothrow @safe
{
  import std.path : buildPath, isAbsolute;
  import std.process : environment;

  string dataE;

  try {
    dataE = environment.get("XDG_DATA_HOME");
  } catch (Exception e) {
    dataE = null;
  }

  if (dataE is null || false == dataE.isAbsolute)
    dataE = buildPath(home(), ".local", "share");

  return dataE;
}

immutable(string) config() nothrow @safe
{
  import std.path : buildPath, isAbsolute;
  import std.process : environment;
    
  string configE;
    
  try {
    configE = environment.get("XDG_CONFIG_HOME");
  } catch (Exception e) {
    configE = null;
  }
    
  if (configE is null || false == configE.isAbsolute)
    configE = buildPath(home(), ".config");

  return configE;
}

/* TODO: xdgState() nothrow @safe */

immutable(string) cache() nothrow @safe
{
  import std.path : buildPath, isAbsolute;
  import std.process : environment;

  string cacheE;

  try {
    cacheE = environment.get("XDG_CACHE_HOME");
  } catch (Exception e) {
    cacheE = null;
  }

  if (cacheE is null || false == cacheE.isAbsolute)
    cacheE = buildPath(home(), ".cache");

  return cacheE;
}

/* Helpers */

const(char)* fallbackHome() nothrow @trusted
{
    import core.stdc.string : strdup;

    passwd* pw;
    char* home;

    setpwent();
    pw = getpwuid(getuid());
    endpwent();

    if (pw is null || pw.pw_dir is null)
        return null;

    home = strdup(pw.pw_dir);
    return home;
}

@system @nogc extern(C) nothrow
{
    /* <bits/types.h> */
    alias gid_t = uint;
    alias uid_t = uint;
    
    /* <pwd.h> */
    struct passwd
    {
        /// Username
        char* pw_name;
        /// Hashed passphrase, if shadow database is not in use
        char* pw_password;
        /// User ID
        uid_t pw_uid;
        /// Group ID
        gid_t pw_gid;
        /// "Real" name
        char* pw_gecos;
        /// Home directory
        char* pw_dir;
        /// Shell program
        char* pw_shell;
    }
    
    /// Rewind the user database stream
    extern void setpwent();
    
    /// Close the user database stream
    extern void endpwent();
    
    /// Retrieve the user database entry for the given user ID
    extern passwd* getpwuid(uid_t uid);
    
    /* <unistd.h> */
    /// Returns the real user ID of the calling process.
    extern uid_t getuid();
}
