#
# Copyright (c) 2023 supercell
#
# SPDX-License-Identifier: BSD-3-Clause
#

module Luce
  class ListItem
    getter lines : Array(String)
    getter task_list_item_state : TaskListState?

    def initialize(@lines : Array(String), @task_list_item_state : TaskListState? = nil)
    end
  end

  enum TaskListState
    CHECKED
    UNCHECKED
  end

  # Base class for both ordered and unorderd lists.
  abstract class ListSyntax < BlockSyntax
    def can_end_block?(parser : BlockParser) : Bool
      # An empty list cannot interrupt a paragraph. See
      # https://spec.commonmark.org/0.29/#example-255.
      # Ideally, `BlockSyntax.can_end_block?` should be changed to a
      # method which accepts a `BlockParser`, but this would be a
      # breaking change, so we're going with this temporarily.
      match = pattern.match(parser.current)
      # The seventh group, in both `Luce.ol_pattern` and
      # `Luce.ul_pattern` is the text after the delimiter.
      if !match.nil? && match[7]?
        return !match[7].empty?
      end
      false
    end

    abstract def list_tag : String

    @@_blocks_in_list : Array(Regex) = [
      Luce.blockquote_pattern,
      Luce.header_pattern,
      Luce.hr_pattern,
      Luce.indent_pattern,
      Luce.ul_pattern,
      Luce.ol_pattern,
    ] of Regex

    def self.blocks_in_list : Array(Regex)
      @@_blocks_in_list
    end

    @@_whitespace_re = Regex.new("[ \t]*")

    def parse(parser : BlockParser) : Node
      task_list_parser_enabled = self.class == UnorderedListWithCheckboxSyntax ||
                                 self.class == OrderedListWithCheckboxSyntax
      items = [] of ListItem
      child_lines = [] of String
      task_list_item_state : TaskListState? = nil

      end_item = ->{
        if !child_lines.empty?
          items << ListItem.new(child_lines, task_list_item_state: task_list_item_state)
          child_lines = [] of String
        end
      }

      parse_task_list_item = ->(text : String) {
        _pattern = Regex.new(%q{^ {0,3}\[([ xX])\][ \t]})

        if task_list_parser_enabled && _pattern.matches?(text)
          match = _pattern.match(text).not_nil!
          task_list_item_state = match[1] == " " ? TaskListState::UNCHECKED : TaskListState::CHECKED
          text.sub(_pattern, "")
        else
          task_list_item_state = nil
          text
        end
      }

      possible_match : Regex::MatchData? = nil
      try_match = ->(pattern : Regex) {
        possible_match = pattern.match(parser.current)
        possible_match != nil
      }

      list_marker : String? = nil
      indent : String? = nil
      # In case the first number in an ordered list is not 1,
      # use it as the "start"
      start_number : Int32? = nil

      while !parser.done?
        leading_space = @@_whitespace_re.match(parser.current).not_nil![0]
        leading_expanded_tab_length = UnorderedListSyntax.expanded_tag_length(leading_space)
        if Luce.empty_pattern.matches?(parser.current)
          if Luce.empty_pattern.matches?(parser.next || "")
            # Two blank lines end a list.
            break
          end
          # Add a blank line to the current list item
          child_lines << ""
        elsif !indent.nil? && indent.size <= leading_expanded_tab_length
          # Strip off indent and add to current item
          line = parser.current
            .sub(leading_space, " " * leading_expanded_tab_length)
            .sub(indent, "")
          child_lines << parse_task_list_item.call(line)
        elsif try_match.call(Luce.hr_pattern)
          # Horizontal rule takes precedence to a new list item.
          break
        elsif try_match.call(Luce.ul_pattern) ||
              try_match.call(Luce.ol_pattern)
          match = possible_match.not_nil!
          preceding_whitespace = match[1].not_nil!
          digits = match[2]? || ""
          if start_number.nil? && !digits.empty?
            start_number = digits.to_i32
          end

          marker = match[3].not_nil!
          first_whitespace = match[5]? || ""
          rest_whitespace = match[6]? || ""
          content = match[7]? || ""
          is_blank = content.empty?
          if !list_marker.nil? && list_marker != marker
            # Changing the bullet or ordered list delimiter starts a new list
            break
          end
          list_marker = marker
          marker_as_spaces = " " * (digits.size + marker.size)
          if is_blank
            # See http://spec.commonmark.org/0.28/#list-items under "3. Item
            # starting with a blank line."
            #
            # If the list item starts with a blank line, the final piece of the
            # indentation is just a single space.
            indent = "#{preceding_whitespace}#{marker_as_spaces} "
          elsif rest_whitespace.size >= 4
            # See http://spec.commonmark.org/0.28/#list-items under "2. Item
            # starting with indented code."
            #
            # If the list item starts with indented code, we need to _not_ count
            # any indentation past the required whitespace character.
            indent = preceding_whitespace + marker_as_spaces + first_whitespace
          else
            indent = preceding_whitespace + marker_as_spaces +
                     first_whitespace + rest_whitespace
          end
          # End the current list item and start a new one
          end_item.call
          child_lines << parse_task_list_item.call("#{rest_whitespace}#{content}")
        elsif BlockSyntax.at_block_end?(parser)
          # Done with list
          break
        else
          # If the previous item is a blank line, this means we're done with the
          # list and are starting a new top-level paragraph.
          if !child_lines.empty? && child_lines.last == ""
            parser.encountered_blank_line = true
            break
          end

          # Anything else is paragraph continuation text.
          child_lines << parser.current
        end
        parser.advance
      end

      end_item.call
      item_nodes = [] of Node

      items.each { |item| remove_leading_empty_line(item) }
      any_empty_lines = remove_trailing_empty_lines(items)
      any_empty_lines_between_blocks = false
      contains_task_list = false

      items.each do |item|
        checkbox_to_insert : Element?
        unless item.task_list_item_state.nil?
          contains_task_list = true
          checkbox_to_insert = Element.with_tag("input")
          checkbox_to_insert.attributes["type"] = "checkbox"
          if item.task_list_item_state == TaskListState::CHECKED
            checkbox_to_insert.attributes["checked"] = "true"
          end
        end
        item_parser = BlockParser.new(item.lines, parser.document)
        children = item_parser.parse_lines
        item_element = if checkbox_to_insert.nil?
                         Element.new("li", children)
                       else
                         e = Element.new("li", [checkbox_to_insert] + children)
                         e.attributes["class"] = "task-list-item"
                         e
                       end
        item_nodes << item_element
        any_empty_lines_between_blocks =
          any_empty_lines_between_blocks || item_parser.encountered_blank_line?
      end

      # Must strip paragraph tags if the list is "tight"
      # http://spec.commonmak.org/0.28/#lists
      list_is_tight = !any_empty_lines && !any_empty_lines_between_blocks

      if list_is_tight
        # We must post-process the list items, converting any top-level paragraph
        # elements to just text elements.
        item_nodes.each do |item|
          if item.is_a? Element
            children = item.children
            if !children.nil?
              i = 0
              while i < children.size
                child = children[i]
                if child.is_a? Element && child.tag == "p"
                  children.delete_at(i)
                  children.insert_all(i, child.children.not_nil!)
                end
                i += 1
              end
            end
          end
        end
      end

      list_element = Element.new(list_tag, item_nodes)
      if list_tag == "ol" && start_number != 1
        list_element.attributes["start"] = "#{start_number}"
      end

      list_element.attributes["class"] = "contains-task-list" if contains_task_list
      list_element
    end

    protected def remove_leading_empty_line(item : ListItem) : Nil
      if !item.lines.empty? && Luce.empty_pattern.matches?(item.lines.first)
        item.lines.delete_at(0)
      end
    end

    # Removes any trailing
    protected def remove_trailing_empty_lines(items : Array(ListItem)) : Bool
      any_empty = false
      i = 0
      while i < items.size
        if items[i].lines.size == 1
          i += 1
          next
        end
        while !items[i].lines.empty? &&
              Luce.empty_pattern.matches?(items[i].lines.last)
          if i < items.size - 1
            any_empty = true
          end
          items[i].lines.pop
        end
        i += 1
      end
      any_empty
    end

    protected def self.expanded_tag_length(input : String) : Int32
      size = 0
      input.codepoints.each do |char|
        size += char == 0x9 ? 4 - (size % 4) : 1
      end
      size
    end
  end
end
