#
# Copyright (c) 2023 supercell
#
# SPDX-License-Identifier: BSD-3-Clause
#

module Luce
  abstract class BlockSyntax
    # Return the regex used to identify the beginning of this block,
    # if any.
    abstract def pattern : Regex

    def can_end_block?(parser : BlockParser)
      true
    end

    def can_parse?(parser : BlockParser)
      pattern.matches? parser.current
    end

    abstract def parse(parser : BlockParser) : Node?

    def parse_child_lines(parser : BlockParser) : Array(String?)
      # Grab all of the lines that form the block element
      child_lines = [] of String?

      until parser.done?
        match = pattern.match(parser.current)
        break if match.nil?
        child_lines += match[0]
        parser.advance
      end

      child_lines
    end

    # Return whether or not *parser*'s current line should end the
    # previous block
    def self.at_block_end?(parser : BlockParser) : Bool
      return true if parser.done?
      parser.block_syntaxes.any? { |s| s.can_parse?(parser) && s.can_end_block?(parser) }
    end

    # Generates a valid HTML anchor from the inner text of *element*.
    def self.generate_anchor_hash(element : Element) : String
      element.children.not_nil!.first.text_content
        .downcase
        .strip
        .gsub(/[^a-z0-9 _-]/, "")
        .gsub(/\s/, "-")
    end
  end
end
