package jp.nanah.bastub.service;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.poi.ss.usermodel.Row;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import jp.nanah.bastub.data.JsonInfo;
import jp.nanah.bastub.util.BastubUtils;

@Service
public class JsonService {

	protected static final Logger logger = LoggerFactory.getLogger(JsonService.class);

	public static final String JSON_DATA_NONE_TEXT = "{\"key\":\"auto created.\"}";

	@Value("${auto_create:false}")
	protected boolean isAutoCreate;

	/**
	 * [4]
	 * @param file
	 * @return
	 * @throws IOException
	 */
	public JsonInfo readJsonFile(File file) {
		JsonInfo jsonInfo = new JsonInfo();

		//ファイルがなかったら新規に作る
		if (file.exists() == false) {
			file.getParentFile().mkdirs();
			jsonInfo.setErrorMessage("JSONファイルが未設定です");
			try {
				if (isAutoCreate){
					FileUtils.writeByteArrayToFile(file, JSON_DATA_NONE_TEXT.getBytes());
					//logger.info("応答JSON - 自動生成[正常]: path=[{}]", file.getAbsolutePath());
					file.setLastModified(0); //自動生成であることがわかるようにするため
				}
			} catch (Exception e){
				logger.info("応答JSON - 自動生成 [失敗]: path=[{}]", file.getAbsolutePath());
			}
			return jsonInfo;
		}

		FileInputStream fis = null;
		try {
			fis = new FileInputStream(file);
			byte[] buff = new byte[(int)file.length()];
			IOUtils.readFully(fis, buff);
			String text = new String(buff, "UTF-8");

			//自動生成したファイルと同じなら未設定と判断する
			if (text.equals(JSON_DATA_NONE_TEXT)){
				jsonInfo.setErrorMessage("JSONファイルが未設定です");
				return jsonInfo;
			}

			//text = BastubUtils.replaceEmbedText(text);

			//${xxxx}の前後がダブルクォーテーションでくくっていないときは
			//数値化指定(#N)付きの文字列にする。
			String reptext = adjustReplaceNumberValue(text);

			String trimText = reptext.trim();
			if (trimText.substring(0,1).equals("[")){
				reptext = "{\"dummy\":" + reptext + "}";
				jsonInfo.setTopArray(true);
			}
			try {
				JSONObject jsonObject = new JSONObject( reptext );
				jsonInfo.setJsonObject(jsonObject);
			} catch (Throwable th){
				jsonInfo.setErrorMessage("JSON解析に失敗しました。書式が正しくない可能性があります。");
			}

		} catch (Throwable th) {
			logger.warn("応答ファイル読み込み失敗: path=[{}]", file.getAbsolutePath(), th);

		} finally {
			IOUtils.closeQuietly(fis);
		}

		return jsonInfo;
	}

	/**
	 * JSONの中に数値の置き換え文字があったら、その前後にダブルクォーテーションをつけて
	 * 文字列データにする。そうしないとJSONObjectがJSONと認識しない。
	 *
	 * @param org
	 * @param numKeys
	 * @return
	 */
	private String adjustReplaceNumberValue(String org) {
		int pos = 0;
		String dst = org; //org.replaceAll("\r|\n", "");

		while (pos < dst.length()) {
			int top = dst.indexOf("${", pos);
			if (top < 0) {
				break;
			}

			int last = BastubUtils.getNumberLast(dst, top);
			if (last >= 0) {
				String v = dst.substring(top, last) + "#N";
				dst = dst.substring(0, top) + "\"" + v + "\"" + dst.substring(last);
				pos = last + 1;
			} else {
				pos = pos + 1;
			}
		}
		return dst;
	}

	/**
	 * [5]
	 *
	 *
	 * @param keyList
	 * @param jsonObj
	 * @param dataList
	 */
	public void setDataToJsonObject(List<String> keyList, JSONObject jsonObj, List<Row> dataList, UsingInfo ui) {
		if (keyList == null){
			keyList = new ArrayList<String>();
		}
		if (dataList == null){
			dataList = new ArrayList<Row>();
		}

		// 【走査１】
		// １．直値で置き換え文字なら、
		// 　　　・1行目から順に、未使用データを適用する。
		// 　　　　つまりすでに適用していれば2行目、3行目のデータを使う。
		// 　　　・1行目データならKey/ValueをHashMapに記録。
		//       ※直値でない場合は特に何もしない。走査２で処理する。

		UsingInfo ud = new UsingInfo(dataList, ui);

		//置換文字が２か所で使われたとき同じ結果を返すようにするため
		List<String> allkeys = BastubUtils.sortIterator(jsonObj.keys());

		for (String key : allkeys){

			//JSONのキーを取得
			if (key == null){
				continue;
			}

			if (jsonObj.optJSONArray(key) != null) {
				//配列はあとで処理
				continue;
			} else if (jsonObj.optJSONObject(key) != null) {
				//JSONオブジェクトのときもあとで処理
				continue;
			} else {
				//直値ならここで置換
				String orgval = jsonObj.optString(key);
				if (orgval != null){
					try {
						String dstval = ud.getDirectValue(keyList, key, orgval);
						//nullを設定するとキーがremoveされてしまうのでここはnullじゃない前提。
						jsonObj.put(key, dstval);
					} catch (Exception e){
						logger.warn("置換異常", e);
					}
				}
			}
		}

		// 【走査２】
		// 処理に入る前に、操作１で置き換えたHashMapに合致したデータに絞り込む。
		// 1.配列なら、配列長が１かそれ以上かを見る。	//実値のとき、置き換え文字だったら置き換える。
		// 　　配列長＝１なら、絞り込みデータN件１つ１つで展開する。	//置き換えは先頭行から。一度使った行は使わず2行目、3行目…を適用する。
		// 　　展開結果を結果を文字列で受け取り、その文字列がユニークなら配列に登録、既出なら無視して、次のデータを処理する。	//
        // 2.JSONオブジェクトなら無条件で再帰呼び出し。データは絞り込み済のデータを渡す。
        // 3. 配列長＞１なら、1つ目の要素は1つ目の要素、2つ目の要素は2つ目とし、配列長だけ実施する。

		List<Row> nextLayerTarget = ud.getLoweredData();
		if (nextLayerTarget == null){
			return;
		}

		for (String key : allkeys){

			//JSONのキーを取得
			if (key == null){
				continue;
			}
			List<String> nextKeyList = new ArrayList<String>(keyList);
			nextKeyList.add(key);

			if (jsonObj.optJSONArray(key) != null) {

				JSONArray jsonAry = jsonObj.optJSONArray(key);
				JSONArray extendAry = getExtendJsonArray(nextKeyList, jsonAry, nextLayerTarget, ui);
				try {
					//展開したJSONを元の配列データと置き換える
					jsonObj.put(key, extendAry);
				} catch (Throwable th){
					logger.warn("配列データ設定異常", th);
				}

			} else if (jsonObj.optJSONObject(key) != null) {

				JSONObject nextJson = jsonObj.optJSONObject(key);
				setDataToJsonObject(nextKeyList, nextJson, nextLayerTarget, ui);

			}
		}

	}

	/**
	 *
	 *
	 * @param templateJson
	 * @param resultTarget
	 * @return
	 */
	public JSONArray getExtendJsonArray(List<String> keyList, JSONArray jsonAry, List<Row> dataList, UsingInfo ui) {
		JSONArray extendAry = new JSONArray();
		UsingInfo ud = new UsingInfo(dataList, ui);
		for (int i=0; i<jsonAry.length(); i++){

			try {
				if (jsonAry.optJSONObject(i) != null){
					//JSONObjectのとき
					String baseText = jsonAry.getJSONObject(i).toString();
					List<String> extendedList = new ArrayList<String>();
					for (int r=1; r<dataList.size(); r++){

						List<Row> lowerDataList = new ArrayList<Row>();
						lowerDataList.add(dataList.get(0));
						lowerDataList.add(dataList.get(r));
						JSONObject extendJson = new JSONObject(baseText);
						setDataToJsonObject(keyList, extendJson, lowerDataList, ui);
						String extendText = extendJson.toString();

						if (extendedList.contains(extendText) == false){
							extendedList.add(extendText);
							extendAry.put(extendAry.length(), extendJson);
						}
					}
					//配列の中を直接書き換えるからこれでOK

				} else if (jsonAry.optJSONArray(i) != null){
					//配列のとき
					JSONArray thisAry = jsonAry.getJSONArray(i);
					JSONArray childAry = getExtendJsonArray(keyList, thisAry, dataList, ui);
					extendAry.put(extendAry.length(), childAry);
				} else {
					//実値のとき
					String orgval = jsonAry.getString(i);
					if (orgval != null){
						try {
							String dstval = ud.getDirectValue(keyList, "", orgval);
							//nullを設定するとキーがremoveされてしまうのでここはnullじゃない前提。
							extendAry.put(extendAry.length(), dstval);
						} catch (Exception e){
							logger.warn("置換異常", e);
						}
					}
				}
			} catch (Throwable th){
				logger.warn("応答データ生成異常",th);
			}
		}

		return extendAry;
	}
}
