#include "Mix/Class/Sound/StreamTask.h"
#include "Mix/Class/Sound/Manager.h"
#include "Mix/Class/Sound/TaskMediator.h"
#include "Mix/Plugin/Sound/IDecoder.h"

namespace Mix{ namespace Sound{

StreamTask::VoiceContext::VoiceContext( void )
{
	m_hEvent[0] = ::CreateEventW( NULL, FALSE, FALSE, NULL );
	m_hEvent[1] = ::CreateEventW( NULL, FALSE, FALSE, NULL );
}

StreamTask::VoiceContext::~VoiceContext( void )
{
	if( m_hEvent[0] != NULL )
	{
		::CloseHandle( m_hEvent[0] );
	}

	if( m_hEvent[1] != NULL )
	{
		::CloseHandle( m_hEvent[1] );
	}
}

void StreamTask::VoiceContext::Reset( void )
{
	::ResetEvent( m_hEvent[VoiceContext::BUFFERING] );
	::ResetEvent( m_hEvent[VoiceContext::END] );
}

UInt32 StreamTask::VoiceContext::Poll( void )
{
	UInt32 ret = ::WaitForMultipleObjects( 2, m_hEvent, FALSE, 0 );
	if( ret == WAIT_TIMEOUT )
	{
		return static_cast<UInt32>( VoiceContext::NONE );
	}

	return ( ret - WAIT_OBJECT_0 );
}

void StreamTask::VoiceContext::OnStreamEnd( void )
{
	::SetEvent( m_hEvent[VoiceContext::END] );
}

void StreamTask::VoiceContext::OnBufferEnd( void* pBufferContext )
{
	::SetEvent( m_hEvent[VoiceContext::BUFFERING] );
}

StreamTask::StreamTask( Mix::Sound::TaskMediator* pTaskMediator ) :
m_pBaseTaskMediator( pTaskMediator ),
m_pDecoder( NULL ),
m_pVoice( NULL ),
m_CurBlock( 0 ),
m_bLoop( False ),
m_State( STATE_IDLE ),
m_SuspendCount( 0 )
{
	UInt32 i;

	MIX_ADD_REF( m_pBaseTaskMediator );

	for( i = 0; i < MAX_BLOCK; i++ )
	{
		m_pBufferArray[i] = NULL;
	}
}

StreamTask::~StreamTask( void )
{
	UInt32 i;

	if( m_pVoice != NULL )
	{
		m_pVoice->DestroyVoice();
	}

	for( i = 0; i < MAX_BLOCK; i++ )
	{
		Mix::Memory::Free( m_pBufferArray[i] );
	}

	MIX_RELEASE( m_pDecoder );
	MIX_RELEASE( m_pBaseTaskMediator );
}

IXAudio2VoiceCallback* StreamTask::GetVoiceCallbackPtr( void )
{
	return &m_VoiceContext;
}

void StreamTask::SetDecoder( Mix::Plugin::Sound::IDecoder* pDecoder )
{
	m_pDecoder = pDecoder;
}

void StreamTask::SetVoice( IXAudio2SourceVoice* pVoice )
{
	m_pVoice = pVoice;
}

Boolean StreamTask::Initialize( void )
{
	UInt32 i;

	//ubÑm
	for( i = 0; i < MAX_BLOCK; i++ )
	{
		m_pBufferArray[i] = static_cast<UInt8*>( Mix::Memory::Allocate( m_pDecoder->GetFormat()->nAvgBytesPerSec ) );
		if( m_pBufferArray[i] == NULL )
		{
			return False;
		}
	}

	return True;
}

Boolean StreamTask::Process( void )
{
	Boolean bContinue = True;

	//eXe[^X̏
	switch( m_State )
	{
	case STATE_IDLE:
		bContinue = ProcIdle();
		break;

	case STATE_STOPPED:
		ProcStopped();
		break;
	}

	return bContinue;
}

Boolean StreamTask::ProcIdle( void )
{
	Boolean bContinue = True;
	Int16 eventType;
	Int16 eventOpt;

	//Cxg
	if( m_pBaseTaskMediator->PopEvent( eventType, eventOpt ) == True )
	{
		switch( eventType )
		{
		case Mix::Sound::TaskMediator::EVENT_TYPE_PLAY:
			ProcPlay( ( eventOpt != 0x00 ) );
			break;
		case Mix::Sound::TaskMediator::EVENT_TYPE_STOP:
			ProcStop();
			break;
		case Mix::Sound::TaskMediator::EVENT_TYPE_SUSPEND:
			ProcSuspend();
			break;
		case Mix::Sound::TaskMediator::EVENT_TYPE_RESUME:
			ProcResume();
			break;
		case Mix::Sound::TaskMediator::EVENT_TYPE_TERMINATE:
			bContinue = False;
			break;
		}
	}

	//{CXReLXg
	switch( m_VoiceContext.Poll() )
	{
	case VoiceContext::BUFFERING:
		ProcBuffering();
		break;

	case VoiceContext::END:
		ProcStop();
		break;
	}

	return bContinue;
}

void StreamTask::ProcStopped( void )
{
	XAUDIO2_VOICE_STATE stat;

	m_pVoice->GetState( &stat );

	if( stat.BuffersQueued == 0 )
	{
		m_pDecoder->Reset();
		m_VoiceContext.Reset();

		m_CurBlock = 0;
		m_SuspendCount = 0;

		m_State = STATE_IDLE;

		m_pBaseTaskMediator->SetState( Mix::Sound::TaskMediator::STATE_STOP );
	}
}

void StreamTask::ProcPlay( Boolean bLoop )
{
	m_bLoop = bLoop;

	for( UInt32 i = 0; i < MAX_BLOCK; i++ )
	{
		ProcBuffering();
	}

	if( m_pVoice->Start() == S_OK )
	{
		m_pBaseTaskMediator->SetState( Mix::Sound::TaskMediator::STATE_PLAY );
	}
}

void StreamTask::ProcStop( void )
{
	m_pVoice->Stop();
	m_pVoice->FlushSourceBuffers();

	m_State = STATE_STOPPED;
}

void StreamTask::ProcSuspend( void )
{
	if( m_SuspendCount++ == 0 )
	{
		m_pVoice->Stop();
	}
}

void StreamTask::ProcResume( void )
{
	if( --m_SuspendCount == 0 )
	{
		m_pVoice->Start();
	}
}

void StreamTask::ProcBuffering( void )
{
	if( m_pDecoder->GetSize() <= m_pDecoder->GetPos() )
	{
		return;
	}

	XAUDIO2_BUFFER buffer = {0};
	UInt32 readSize;

	if( m_bLoop == TRUE )
	{
		BYTE* pDst = &( m_pBufferArray[m_CurBlock][0] );
		int readCountDown;

		readCountDown = readSize = m_pDecoder->GetFormat()->nAvgBytesPerSec;

		do
		{
			UInt32 size = m_pDecoder->Read( pDst, readCountDown );
			if( size == 0 )
			{
				m_pDecoder->Reset();
			}
			else
			{
				readCountDown -= size;
				pDst += size;
			}
		}
		while( readCountDown > 0 );
	}
	else
	{
		readSize = static_cast<UInt32>( min( m_pDecoder->GetFormat()->nAvgBytesPerSec, ( m_pDecoder->GetSize() - m_pDecoder->GetPos() ) ) );

		Mix::Memory::Zero( m_pBufferArray[m_CurBlock], m_pDecoder->GetFormat()->nAvgBytesPerSec );
		m_pDecoder->Read( m_pBufferArray[m_CurBlock], readSize );
	}

	buffer.AudioBytes = m_pDecoder->GetFormat()->nAvgBytesPerSec;
	buffer.pAudioData = m_pBufferArray[m_CurBlock];
	if( m_pDecoder->GetSize() <= m_pDecoder->GetPos() )
	{
		buffer.Flags = XAUDIO2_END_OF_STREAM;
	}

	m_pVoice->SubmitSourceBuffer( &buffer );

	m_CurBlock++;
	m_CurBlock %= MAX_BLOCK;

//	TraceLine( _T( "Buffering : pos[%d] size[%d]" ), static_cast<UInt32>( m_pDecoder->GetWavePos() ), readSize );
}

}}
