#include "Mix/Class/Sound/StreamEmitter.h"

#include "Mix/Class/Sound/Manager.h"
#include "Mix/Class/Sound/Listener.h"
#include "Mix/Class/Sound/EmitterTaskMediator.h"
#include "Mix/File/IReader.h"

namespace Mix{ namespace Sound{

StreamEmitter* StreamEmitter::CreateInstance(	Mix::Sound::Listener* pListener,
												Mix::File::IReader* pReader,
												Mix::Sound::EmitterTaskMediator* pTaskMediator,
												const Mix::Vector3& localFront,
												const Mix::Vector3& localUp,
												UInt32 channels,
												UInt32 masterChannels )
{
	return new StreamEmitter( pListener, pReader, pTaskMediator, localFront, localUp, channels, masterChannels );
}

StreamEmitter::StreamEmitter(	Mix::Sound::Listener* pListener,
								Mix::File::IReader* pReader,
								Mix::Sound::EmitterTaskMediator* pTaskMediator,
								const Mix::Vector3& localFront,
								const Mix::Vector3& localUp,
								UInt32 channels,
								UInt32 masterChannels ) :
m_pListener( NULL ),
m_pReader( NULL ),
m_pTaskMediator( NULL ),
m_LocalFront( localFront ),
m_LocalUp( localUp ),
m_WorldFront( localFront ),
m_WorldUp( localUp )
{
	MIX_ASSERT( pListener != NULL );
	MIX_ASSERT( pReader != NULL );
	MIX_ASSERT( pTaskMediator != NULL );

	MIX_ADD_REF( pListener );
	m_pListener = pListener;

	MIX_ADD_REF( pReader );
	m_pReader = pReader;

	MIX_ADD_REF( pTaskMediator );
	m_pTaskMediator = pTaskMediator;

	m_ChannelAzimuths.resize( channels, 1.0f );

	m_Param.pCone = NULL;
	m_Param.OrientFront.x = m_LocalFront.x;
	m_Param.OrientFront.y = m_LocalFront.y;
	m_Param.OrientFront.z = m_LocalFront.z;
	m_Param.OrientTop.x = m_LocalUp.x;
	m_Param.OrientTop.y = m_LocalUp.y;
	m_Param.OrientTop.z = m_LocalUp.z;
	m_Param.Position.x = 0.0f;
	m_Param.Position.y = 0.0f;
	m_Param.Position.z = 0.0f;
	m_Param.Velocity.x = 0.0f;
	m_Param.Velocity.y = 0.0f;
	m_Param.Velocity.z = 0.0f;
	m_Param.InnerRadius = 0.0f;
	m_Param.InnerRadiusAngle = ( X3DAUDIO_PI / 4.0f );
	m_Param.ChannelCount = channels;
	m_Param.ChannelRadius = 1.0f;
	m_Param.pChannelAzimuths = &( m_ChannelAzimuths[0] );
	m_Param.pVolumeCurve = NULL;
	m_Param.pLFECurve = NULL;
	m_Param.pLPFDirectCurve = NULL;
	m_Param.pLPFReverbCurve = NULL;
	m_Param.pReverbCurve = NULL;
	m_Param.CurveDistanceScaler = 1.0f;
	m_Param.DopplerScaler = 1.0f;
}

StreamEmitter::~StreamEmitter( void )
{
	m_pTaskMediator->PushEvent( Mix::Sound::EmitterTaskMediator::EVENT_TYPE_TERMINATE, 0 );

	MIX_RELEASE( m_pReader );
	MIX_RELEASE( m_pTaskMediator );
	MIX_RELEASE( m_pListener );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Sound::IEmitter
////////////////////////////////////////////////////////////////////////////////////////////////////

const Mix::Vector3& StreamEmitter::GetLocalFront( void ) const
{
	return m_LocalFront;
}

const Mix::Vector3& StreamEmitter::GetLocalUp( void ) const
{
	return m_LocalUp;
}

const Mix::Matrix4x4& StreamEmitter::GetWorldMatrix( void ) const
{
	return m_WorldMatrix;
}

void StreamEmitter::SetWorldMatrix( const Mix::Matrix4x4& mat )
{
	m_WorldMatrix = mat;

	m_WorldFront = m_WorldMatrix.TransformSR( m_LocalFront );
	m_WorldUp = m_WorldMatrix.TransformSR( m_LocalUp );

	m_Param.OrientFront.x = m_WorldFront.x;
	m_Param.OrientFront.y = m_WorldFront.y;
	m_Param.OrientFront.z = m_WorldFront.z;
	m_Param.OrientTop.x = m_WorldUp.x;
	m_Param.OrientTop.y = m_WorldUp.y;
	m_Param.OrientTop.z = m_WorldUp.z;
	m_Param.Position.x = m_WorldMatrix.m30;
	m_Param.Position.y = m_WorldMatrix.m31;
	m_Param.Position.z = m_WorldMatrix.m32;
}

const Mix::Vector3& StreamEmitter::GetWorldFront( void ) const
{
	return m_WorldFront;
}

const Mix::Vector3& StreamEmitter::GetWorldUp( void ) const
{
	return m_WorldUp;
}

const Mix::Vector3& StreamEmitter::GetVelocity( void ) const
{
	return m_Velocity;
}

void StreamEmitter::SetVelocity( const Mix::Vector3& velocity )
{
	m_Velocity = velocity;

	m_Param.Velocity.x = m_Velocity.x;
	m_Param.Velocity.y = m_Velocity.y;
	m_Param.Velocity.z = m_Velocity.z;
}

UInt32 StreamEmitter::GetChannelCount( void ) const
{
	return m_Param.ChannelCount;
}

Float32 StreamEmitter::GetChannelRadius( void ) const
{
	return m_Param.ChannelRadius;
}

void StreamEmitter::SetChannelRadius( Float32 radius )
{
	m_Param.ChannelRadius = max( 0.0f, radius );
}

Float32 StreamEmitter::GetChannelAzimuth( UInt32 channel ) const
{
	if( channel >= m_Param.ChannelCount )
	{
		return 0.0f;
	}

	return m_ChannelAzimuths[channel];
}

void StreamEmitter::SetChannelAzimuth( UInt32 channel, Float32 azimuth )
{
	if( channel >= m_Param.ChannelCount )
	{
		return;
	}

	m_ChannelAzimuths[channel] = MIX_CLAMP( azimuth, 0.0f, MIX_2PI );
}

Float32 StreamEmitter::GetDopplerScaler( void ) const
{
	return m_Param.DopplerScaler;
}

void StreamEmitter::SetDopplerScaler( Float32 scaler )
{
	m_Param.DopplerScaler = MIX_CLAMP( scaler, 0.0f, MIX_FLOAT_MAX );
}

void StreamEmitter::Update( void )
{
	m_pTaskMediator->UpdateDSPSettings( m_pListener->GetParam(), this->m_Param );
}

Boolean StreamEmitter::IsCloneable( void ) const
{
	return m_pReader->IsBuffered();
}

Boolean StreamEmitter::Clone( Mix::Sound::IEmitter** ppEmitter )
{
	if( ppEmitter == NULL )
	{
		return False;
	}

	Mix::Sound::IEmitter* pEmitter = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_ASSERT( Mix::Sound::GetInternalManagerPtr() != NULL );

	if( Mix::Sound::GetInternalManagerPtr()->CloneStreamEmitter( m_pListener, m_pReader, m_LocalFront, m_LocalUp, &pEmitter ) == False )
	{
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// p[^ݒ
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pEmitter->SetChannelRadius( GetChannelRadius() );

	for( UInt32 i = 0; i < GetChannelCount(); i++ )
	{
		pEmitter->SetChannelAzimuth( i, GetChannelAzimuth( i ) );
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppEmitter ) = pEmitter;

	return True;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Sound::IDevice
////////////////////////////////////////////////////////////////////////////////////////////////////

Boolean StreamEmitter::IsPlaying( void )
{
	return ( m_pTaskMediator->GetState() == Mix::Sound::TaskMediator::STATE_PLAY );
}

void StreamEmitter::Play( Boolean bLoop )
{
	m_pTaskMediator->PushEvent( Mix::Sound::EmitterTaskMediator::EVENT_TYPE_STOP, 0 );
	m_pTaskMediator->PushEvent( Mix::Sound::EmitterTaskMediator::EVENT_TYPE_PLAY, ( bLoop == True )? 1 : 0 );
}

void StreamEmitter::Stop( void )
{
	m_pTaskMediator->PushEvent( Mix::Sound::EmitterTaskMediator::EVENT_TYPE_STOP, 0 );
}

void StreamEmitter::Suspend( void )
{
	m_pTaskMediator->PushEvent( Mix::Sound::EmitterTaskMediator::EVENT_TYPE_SUSPEND, 0 );
}

void StreamEmitter::Resume( void )
{
	m_pTaskMediator->PushEvent( Mix::Sound::EmitterTaskMediator::EVENT_TYPE_RESUME, 0 );
}

const wchar_t* StreamEmitter::GetFilePath( void ) const
{
	return m_pReader->GetFilePath();
}

}}
