#include "Mix/Class/Sound/StaticEmitter.h"

#include "Mix/Class/Sound/Manager.h"
#include "Mix/Class/Sound/Listener.h"
#include "Mix/Class/Sound/EmitterTaskMediator.h"
#include "Mix/Memory/IBuffer.h"

namespace Mix{ namespace Sound{

StaticEmitter* StaticEmitter::CreateInstance(	Mix::Sound::Listener* pListener,
												Mix::Sound::EmitterTaskMediator* pTaskMediator,
												const WAVEFORMATEX* pFormat,
												Mix::Memory::IBuffer* pBuffer,
												const Mix::Vector3& front,
												const Mix::Vector3& up,
												UInt32 masterChannels,
												const wchar_t* pFilePath )
{
	return new StaticEmitter( pListener, pTaskMediator, pFormat, pBuffer, front, up, masterChannels, pFilePath );
}

StaticEmitter::StaticEmitter(	Mix::Sound::Listener* pListener,
								Mix::Sound::EmitterTaskMediator* pTaskMediator,
								const WAVEFORMATEX* pFormat,
								Mix::Memory::IBuffer* pBuffer,
								const Mix::Vector3& front,
								const Mix::Vector3& up,
								UInt32 masterChannels,
								const wchar_t* pFilePath ) :
m_pListener( NULL ),
m_pTaskMediator( NULL ),
m_pBuffer( NULL ),
m_LocalFront( front ),
m_LocalUp( up ),
m_WorldFront( front ),
m_WorldUp( up ),
m_FilePath( pFilePath )
{
	MIX_ASSERT( pListener != NULL );
	MIX_ASSERT( pTaskMediator != NULL );
	MIX_ASSERT( pFormat != NULL );
	MIX_ASSERT( pBuffer != NULL );
	MIX_ASSERT( pFilePath != NULL );

	MIX_ADD_REF( pListener );
	m_pListener = pListener;

	MIX_ADD_REF( pTaskMediator );
	m_pTaskMediator = pTaskMediator;

	MIX_ADD_REF( pBuffer );
	m_pBuffer = pBuffer;

	Mix::Memory::Copy( &m_Format, pFormat, sizeof( WAVEFORMATEX ) );

	m_ChannelAzimuths.resize( m_Format.nChannels, 1.0f );

	m_Param.pCone = NULL;
	m_Param.OrientFront.x = m_WorldFront.x;
	m_Param.OrientFront.y = m_WorldFront.y;
	m_Param.OrientFront.z = m_WorldFront.z;
	m_Param.OrientTop.x = m_WorldUp.x;
	m_Param.OrientTop.y = m_WorldUp.y;
	m_Param.OrientTop.z = m_WorldUp.z;
	m_Param.Position.x = 0.0f;
	m_Param.Position.y = 0.0f;
	m_Param.Position.z = 0.0f;
	m_Param.Velocity.x = 0.0f;
	m_Param.Velocity.y = 0.0f;
	m_Param.Velocity.z = 0.0f;
	m_Param.InnerRadius = 2.0f;
	m_Param.InnerRadiusAngle = ( X3DAUDIO_PI / 4.0f );
	m_Param.ChannelCount = m_Format.nChannels;
	m_Param.ChannelRadius = 1.0f;
	m_Param.pChannelAzimuths = &( m_ChannelAzimuths[0] );
	m_Param.pVolumeCurve = NULL;
	m_Param.pLFECurve = NULL;
	m_Param.pLPFDirectCurve = NULL;
	m_Param.pLPFReverbCurve = NULL;
	m_Param.pReverbCurve = NULL;
	m_Param.CurveDistanceScaler = 1.0f;
	m_Param.DopplerScaler = 1.0f;
}

StaticEmitter::~StaticEmitter( void )
{
	m_pTaskMediator->PushEvent( Mix::Sound::EmitterTaskMediator::EVENT_TYPE_TERMINATE, 0 );

	MIX_RELEASE( m_pBuffer );
	MIX_RELEASE( m_pTaskMediator );
	MIX_RELEASE( m_pListener );
}

const WAVEFORMATEX* StaticEmitter::GetFormat( void ) const
{
	return &m_Format;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Sound::IEmitter
////////////////////////////////////////////////////////////////////////////////////////////////////

const Mix::Vector3& StaticEmitter::GetLocalFront( void ) const
{
	return m_LocalFront;
}

const Mix::Vector3& StaticEmitter::GetLocalUp( void ) const
{
	return m_LocalUp;
}

const Mix::Vector3& StaticEmitter::GetWorldFront( void ) const
{
	return m_WorldFront;
}

const Mix::Vector3& StaticEmitter::GetWorldUp( void ) const
{
	return m_WorldUp;
}

const Mix::Matrix4x4& StaticEmitter::GetWorldMatrix( void ) const
{
	return m_WorldMatrix;
}

void StaticEmitter::SetWorldMatrix( const Mix::Matrix4x4& mat )
{
	m_WorldMatrix = mat;

	m_WorldFront = m_WorldMatrix.TransformSR( m_LocalFront );
	m_WorldUp = m_WorldMatrix.TransformSR( m_LocalUp );

	m_Param.OrientFront.x = m_WorldFront.x;
	m_Param.OrientFront.y = m_WorldFront.y;
	m_Param.OrientFront.z = m_WorldFront.z;
	m_Param.OrientTop.x = m_WorldUp.x;
	m_Param.OrientTop.y = m_WorldUp.y;
	m_Param.OrientTop.z = m_WorldUp.z;
	m_Param.Position.x = m_WorldMatrix.m30;
	m_Param.Position.y = m_WorldMatrix.m31;
	m_Param.Position.z = m_WorldMatrix.m32;
}

const Mix::Vector3& StaticEmitter::GetVelocity( void ) const
{
	return m_Velocity;
}

void StaticEmitter::SetVelocity( const Mix::Vector3& velocity )
{
	m_Velocity = velocity;

	m_Param.Velocity.x = m_Velocity.x;
	m_Param.Velocity.y = m_Velocity.y;
	m_Param.Velocity.z = m_Velocity.z;
}

UInt32 StaticEmitter::GetChannelCount( void ) const
{
	return m_Param.ChannelCount;
}

Float32 StaticEmitter::GetChannelRadius( void ) const
{
	return m_Param.ChannelRadius;
}

void StaticEmitter::SetChannelRadius( Float32 radius )
{
	m_Param.ChannelRadius = max( 0.0f, radius );
}

Float32 StaticEmitter::GetChannelAzimuth( UInt32 channel ) const
{
	if( channel >= m_Param.ChannelCount )
	{
		return 0.0f;
	}

	return m_ChannelAzimuths[channel];
}

void StaticEmitter::SetChannelAzimuth( UInt32 channel, Float32 azimuth )
{
	if( channel >= m_Param.ChannelCount )
	{
		return;
	}

	m_ChannelAzimuths[channel] = MIX_CLAMP( azimuth, 0.0f, MIX_2PI );
}

Float32 StaticEmitter::GetDopplerScaler( void ) const
{
	return m_Param.DopplerScaler;
}

void StaticEmitter::SetDopplerScaler( Float32 scaler )
{
	m_Param.DopplerScaler = MIX_CLAMP( scaler, 0.0f, MIX_FLOAT_MAX );
}

void StaticEmitter::Update( void )
{
	m_pTaskMediator->UpdateDSPSettings( m_pListener->GetParam(), this->m_Param );
}

Boolean StaticEmitter::IsCloneable( void ) const
{
	return True;
}

Boolean StaticEmitter::Clone( Mix::Sound::IEmitter** ppEmitter )
{
	if( ppEmitter == NULL )
	{
		return False;
	}

	Mix::Sound::IEmitter* pEmitter = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_ASSERT( Mix::Sound::GetInternalManagerPtr() != NULL );

	if( Mix::Sound::GetInternalManagerPtr()->CloneStaticEmitter( m_pListener, m_FilePath.GetConstPtr(), &m_Format, m_pBuffer, m_LocalFront, m_LocalUp, &pEmitter ) == False )
	{
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// p[^ݒ
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pEmitter->SetChannelRadius( GetChannelRadius() );

	for( UInt32 i = 0; i < GetChannelCount(); i++ )
	{
		pEmitter->SetChannelAzimuth( i, GetChannelAzimuth( i ) );
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppEmitter ) = pEmitter;

	return True;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Sound::IDevice
////////////////////////////////////////////////////////////////////////////////////////////////////

Boolean StaticEmitter::IsPlaying( void )
{
	return ( m_pTaskMediator->GetState() == Mix::Sound::TaskMediator::STATE_PLAY );
}

void StaticEmitter::Play( Boolean bLoop )
{
	m_pTaskMediator->PushEvent( Mix::Sound::TaskMediator::EVENT_TYPE_PLAY, ( bLoop == True )? 1 : 0 );
}

void StaticEmitter::Stop( void )
{
	m_pTaskMediator->PushEvent( Mix::Sound::TaskMediator::EVENT_TYPE_STOP, 0 );
}

void StaticEmitter::Suspend( void )
{
	m_pTaskMediator->PushEvent( Mix::Sound::TaskMediator::EVENT_TYPE_SUSPEND, 0 );
}

void StaticEmitter::Resume( void )
{
	m_pTaskMediator->PushEvent( Mix::Sound::TaskMediator::EVENT_TYPE_RESUME, 0 );
}

const wchar_t* StaticEmitter::GetFilePath( void ) const
{
	return m_FilePath.GetConstPtr();
}

}}
