#include "Mix/Class/Sound/StaticController.h"
#include "Mix/Class/Sound/Manager.h"
#include "Mix/Memory/IBuffer.h"

namespace Mix{ namespace Sound{

StaticController* StaticController::CreateInstance( const Mix::Sound::StaticController* pSrc, IXAudio2SourceVoice* pVoice )
{
	return new StaticController( pSrc, pVoice );
}

StaticController* StaticController::CreateInstance( Mix::Sound::Manager* pMgr, const wchar_t* fileName, const WAVEFORMATEX* pFormat, IXAudio2SourceVoice* pVoice, Mix::Memory::IBuffer* pBuffer )
{
	return new StaticController( pMgr, fileName, pFormat, pVoice, pBuffer );
}

StaticController::StaticController( const Mix::Sound::StaticController* pSrc, IXAudio2SourceVoice* pVoice ) :
m_pMgr( pSrc->m_pMgr ),
m_FilePath( pSrc->m_FilePath ),
m_pVoice( pVoice ),
m_pBuffer( pSrc->m_pBuffer ),
m_SuspendCount( 0 ),
m_Panpot( 0.0f )
{
	MIX_ADD_REF( m_pMgr );
	MIX_ADD_REF( m_pBuffer );

	Mix::Memory::Copy( &m_Format, &( pSrc->m_Format ), sizeof( WAVEFORMATEX ) );

	m_Data.Flags = XAUDIO2_END_OF_STREAM;
	m_Data.AudioBytes = m_pBuffer->GetSize();
	m_Data.pAudioData = static_cast<const BYTE*>( m_pBuffer->GetConstPointer() );
	m_Data.PlayBegin = 0;
	m_Data.PlayLength = 0;
	m_Data.LoopBegin = XAUDIO2_NO_LOOP_REGION;
	m_Data.LoopLength = 0;
	m_Data.LoopCount = 0;
	m_Data.pContext = NULL;
}

StaticController::StaticController( Mix::Sound::Manager* pMgr, const wchar_t* filePath, const WAVEFORMATEX* pFormat, IXAudio2SourceVoice* pVoice, Mix::Memory::IBuffer* pBuffer ) :
m_pMgr( pMgr ),
m_FilePath( filePath ),
m_pVoice( pVoice ),
m_pBuffer( pBuffer ),
m_SuspendCount( 0 ),
m_Panpot( 0.0f )
{
	MIX_ADD_REF( m_pMgr );

	Mix::Memory::Copy( &m_Format, pFormat, sizeof( WAVEFORMATEX ) );

	m_Data.Flags = XAUDIO2_END_OF_STREAM;
	m_Data.AudioBytes = m_pBuffer->GetSize();
	m_Data.pAudioData = static_cast<const BYTE*>( m_pBuffer->GetConstPointer() );
	m_Data.PlayBegin = 0;
	m_Data.PlayLength = 0;
	m_Data.LoopBegin = XAUDIO2_NO_LOOP_REGION;
	m_Data.LoopLength = 0;
	m_Data.LoopCount = 0;
	m_Data.pContext = NULL;
}

StaticController::~StaticController( void )
{
	if( m_pVoice != NULL )
	{
		m_pVoice->DestroyVoice();
	}

	MIX_RELEASE( m_pBuffer );
	MIX_RELEASE( m_pMgr );
}

const WAVEFORMATEX* StaticController::GetFormat( void ) const
{
	return &m_Format;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Sound::IController
////////////////////////////////////////////////////////////////////////////////////////////////////

Float32 StaticController::GetVolume( void ) const
{
	Float32 volume;

	m_pVoice->GetVolume( &volume );

	return volume;
}

void StaticController::SetVolume( Float32 volume )
{
	m_pVoice->SetVolume( volume );
}

Float32 StaticController::GetPanpot( void ) const
{
	return m_Panpot;
}

void StaticController::SetPanpot( Float32 panpot )
{
	Float32 cv[2];

	if( panpot >= 0.0f )
	{
		cv[0] = ( 1.0f - panpot );
		cv[1] = 1.0;
	}
	else
	{
		cv[0] = 1.0;
		cv[1] = ( 1.0f + panpot );
	}

	if( m_pVoice->SetChannelVolumes( 2, cv ) == S_OK )
	{
		m_Panpot = panpot;
	}
}

Boolean StaticController::IsCloneable( void ) const
{
	return True;
}

Boolean StaticController::Clone( Mix::Sound::IController** ppController )
{
	return m_pMgr->CloneStaticController( this, ppController );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Sound::IDevice
////////////////////////////////////////////////////////////////////////////////////////////////////

Boolean StaticController::IsPlaying( void )
{
	XAUDIO2_VOICE_STATE state;

	m_pVoice->GetState( &state );

	return ( state.BuffersQueued > 0 );
}

void StaticController::Play( Boolean bLoop )
{
	Stop();

	m_Data.LoopCount = ( bLoop == True )? XAUDIO2_LOOP_INFINITE : 0;

	if( m_pVoice->SubmitSourceBuffer( &m_Data ) != S_OK )
	{
		return;
	}

	if( m_pVoice->Start( 0, XAUDIO2_COMMIT_NOW ) != S_OK )
	{
		return;
	}
}

void StaticController::Stop( void )
{
	m_pVoice->Stop( 0 , XAUDIO2_COMMIT_NOW );
	m_pVoice->FlushSourceBuffers();

	m_SuspendCount = 0;
}

void StaticController::Suspend( void )
{
	if( m_SuspendCount++ == 0 )
	{
		m_pVoice->Stop( 0 , XAUDIO2_COMMIT_NOW );
	}
}

void StaticController::Resume( void )
{
	if( --m_SuspendCount == 0 )
	{
		m_pVoice->Start();
	}
}

const wchar_t* StaticController::GetFilePath( void ) const
{
	return m_FilePath.GetConstPtr();
}

}}
