#include "Mix/Class/Scene/Common/Planter.h"

#include "Mix/Scene/IMaterial.h"
#include "Mix/Scene/IDefaultMaterial.h"

namespace Mix{ namespace Scene{ namespace Common{

////////////////////////////////////////////////////////////////////////////////////////////////////
// Planter : 萔
////////////////////////////////////////////////////////////////////////////////////////////////////

const wchar_t* Planter::FAILED_CREATE = L"v^[̍쐬Ɏs";
const wchar_t* Planter::FAILED_SET_MATERIAL = L"v^[̃}eA̐ݒɎs";

const UInt32 Planter::VS_INPUT_SIGUNATURE = ( 1 << Mix::Graphics::VLS_POSITION ) |
											( 1 << Mix::Graphics::VLS_COLOR ) |
											( 1 << Mix::Graphics::VLS_NORMAL ) |
											( 1 << Mix::Graphics::VLS_TEXTURE );

const Mix::Vector3 Planter::SIMPLE_POS_TABLE[4] = 
{
	Mix::Vector3( -0.5f,  0.0f,  0.0f ),
	Mix::Vector3( -0.5f, +1.0f,  0.0f ),
	Mix::Vector3( +0.5f, +1.0f,  0.0f ),
	Mix::Vector3( +0.5f,  0.0f,  0.0f ),
};

const Mix::Vector3 Planter::CROSS1_POS_TABLE[16] =
{
	Mix::Vector3( -0.353553f, 0.000000f,  0.353553f ),
	Mix::Vector3( -0.353553f, 1.000000f,  0.353553f ),
	Mix::Vector3(  0.000000f, 1.000000f,  0.000000f ),
	Mix::Vector3(  0.000000f, 0.000000f,  0.000000f ),

	Mix::Vector3(  0.353553f, 0.000000f,  0.353553f ),
	Mix::Vector3(  0.353553f, 1.000000f,  0.353553f ),
	Mix::Vector3(  0.000000f, 1.000000f,  0.000000f ),
	Mix::Vector3(  0.000000f, 0.000000f,  0.000000f ),

	Mix::Vector3(  0.353553f, 0.000000f, -0.353553f ),
	Mix::Vector3(  0.353553f, 1.000000f, -0.353553f ),
	Mix::Vector3(  0.000000f, 1.000000f,  0.000000f ),
	Mix::Vector3(  0.000000f, 0.000000f,  0.000000f ),

	Mix::Vector3( -0.353553f, 0.000000f, -0.353553f ),
	Mix::Vector3( -0.353553f, 1.000000f, -0.353553f ),
	Mix::Vector3(  0.000000f, 1.000000f,  0.000000f ),
	Mix::Vector3(  0.000000f, 0.000000f,  0.000000f ),
};

const Mix::Vector3 Planter::CROSS2_POS_TABLE[24] =
{
	Mix::Vector3( -0.500000f, 0.000000f,  0.000000f ),
	Mix::Vector3( -0.500000f, 1.000000f,  0.000000f ),
	Mix::Vector3(  0.000000f, 1.000000f,  0.000000f ),
	Mix::Vector3(  0.000000f, 0.000000f,  0.000000f ),

	Mix::Vector3( -0.250000f, 0.000000f,  0.433013f ),
	Mix::Vector3( -0.250000f, 1.000000f,  0.433013f ),
	Mix::Vector3(  0.000000f, 1.000000f,  0.000000f ),
	Mix::Vector3(  0.000000f, 0.000000f,  0.000000f ),

	Mix::Vector3(  0.250000f, 0.000000f,  0.433013f ),
	Mix::Vector3(  0.250000f, 1.000000f,  0.433013f ),
	Mix::Vector3(  0.000000f, 1.000000f,  0.000000f ),
	Mix::Vector3(  0.000000f, 0.000000f,  0.000000f ),

	Mix::Vector3(  0.500000f, 0.000000f, -0.000000f ),
	Mix::Vector3(  0.500000f, 1.000000f, -0.000000f ),
	Mix::Vector3(  0.000000f, 1.000000f,  0.000000f ),
	Mix::Vector3(  0.000000f, 0.000000f,  0.000000f ),

	Mix::Vector3(  0.250000f, 0.000000f, -0.433013f ),
	Mix::Vector3(  0.250000f, 1.000000f, -0.433013f ),
	Mix::Vector3(  0.000000f, 1.000000f,  0.000000f ),
	Mix::Vector3(  0.000000f, 0.000000f,  0.000000f ),

	Mix::Vector3( -0.250000f, 0.000000f, -0.433013f ),
	Mix::Vector3( -0.250000f, 1.000000f, -0.433013f ),
	Mix::Vector3(  0.000000f, 1.000000f,  0.000000f ),
	Mix::Vector3(  0.000000f, 0.000000f,  0.000000f ),
};

const UInt32 Planter::DEF_STATUS = Planter::DAT_MAX;

////////////////////////////////////////////////////////////////////////////////////////////////////
// Planter
////////////////////////////////////////////////////////////////////////////////////////////////////

Boolean Planter::CheckMaterial( Mix::Scene::IMaterial* pMaterial, const wchar_t* pLabel, const wchar_t* pDebugName )
{
	MIX_ASSERT( pMaterial != NULL );

	UInt32 vis;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ^Cṽ`FbN
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pMaterial->GetType() != Mix::Scene::IMaterial::DEFAULT )
	{
		MIX_LOG_ERROR( L"%s : ftHg}eÂ݃T|[gĂ܂ : DebugName[%s]", Planter::FAILED_CREATE, MIX_SAFE_NAME( pDebugName ) );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ̓VOl`̃`FbN
	////////////////////////////////////////////////////////////////////////////////////////////////////

	vis = pMaterial->GetVertexInputSigunature( Mix::Scene::IMaterial::TR_SIMPLE );

	if( vis != Planter::VS_INPUT_SIGUNATURE )
	{
		if( MIX_TESTBIT( vis, ( 1 << Mix::Graphics::VLS_NORMAL ) ) == 0 )
		{
			MIX_LOG_ERROR( L"%s : VF[hX^Cṽ}eA͎gpł܂(@) : DebugName[%s]", Planter::FAILED_CREATE, MIX_SAFE_NAME( pDebugName ) );
		}
		else if( MIX_TESTBIT( vis, ( 1 << Mix::Graphics::VLS_TEXTURE ) ) == 0 )
		{
			MIX_LOG_ERROR( L"%s : eNX`}bsOsȂ}eAݒ肵悤Ƃ܂(UVW) : DebugName[%s]", Planter::FAILED_CREATE, MIX_SAFE_NAME( pDebugName ) );
		}
		else if(	( MIX_TESTBIT( vis, ( 1 << Mix::Graphics::VLS_BINORMAL ) ) != 0 ) ||
					( MIX_TESTBIT( vis, ( 1 << Mix::Graphics::VLS_TANGENT ) ) != 0 ) )
		{
			MIX_LOG_ERROR( L"%s : ov}bsO̓T|[gĂ܂(ڐ) : DebugName[%s]", Planter::FAILED_CREATE, MIX_SAFE_NAME( pDebugName ) );
		}
		else
		{
			MIX_LOG_ERROR( L"%s : sȓ̓VOl`̃}eAݒ肵悤Ƃ܂ : DebugName[%s]", Planter::FAILED_CREATE, MIX_SAFE_NAME( pDebugName ) );
		}

		return False;
	}

	return True;
}

Planter* Planter::CreateInstance(	Mix::Scene::IMaterial* pMaterial,
									UInt32 numModel, const Mix::Scene::IPlanter::GRASS_MODEL* models,
									UInt32 numUnit, const Mix::Scene::IPlanter::GRASS_UNIT* units,
									const wchar_t* pDebugName )
{
	return new Planter( pMaterial, numModel, models, numUnit, units, pDebugName );
}

Planter* Planter::CreateInstance(	const Mix::Matrix4x4& worldMat,
									Mix::Scene::IMaterial* pMaterial,
									UInt32 numModel, const Mix::Scene::IPlanter::GRASS_MODEL* models,
									UInt32 numUnit, const Mix::Scene::IPlanter::GRASS_UNIT* units,
									const wchar_t* pDebugName )
{
	return new Planter( worldMat, pMaterial, numModel, models, numUnit, units, pDebugName );
}

Planter::Planter(	Mix::Scene::IMaterial* pMaterial,
					UInt32 numModel, const Mix::Scene::IPlanter::GRASS_MODEL* models,
					UInt32 numUnit, const Mix::Scene::IPlanter::GRASS_UNIT* units,
					const wchar_t* pDebugName ) :
m_Status( Planter::DEF_STATUS ),
m_pOctObj( NULL ),
m_pMaterial( NULL ),
m_bLocalLighting( False ),
m_bDraw( True ),
m_TRPriority( 0 ),
m_WCResult( Mix::Scene::WCR_FRONT )
{
#ifdef _DEBUG
	m_DebugName = pDebugName;
#endif //_DEBUG

	Initialize( pMaterial, numModel, models, numUnit, units, pDebugName );
}

Planter::Planter(	const Mix::Matrix4x4& worldMat,
					Mix::Scene::IMaterial* pMaterial,
					UInt32 numModel, const Mix::Scene::IPlanter::GRASS_MODEL* models,
					UInt32 numUnit, const Mix::Scene::IPlanter::GRASS_UNIT* units,
					const wchar_t* pDebugName ) :
m_Status( Planter::DEF_STATUS ),
m_pOctObj( NULL ),
m_pMaterial( NULL ),
m_bLocalLighting( False ),
m_bDraw( True ),
m_TRPriority( 0 ),
m_WCResult( Mix::Scene::WCR_FRONT )
{
#ifdef _DEBUG
	m_DebugName = pDebugName;
#endif //_DEBUG

	Initialize( pMaterial, numModel, models, numUnit, units, pDebugName );
	TransformUnits( worldMat );
}

Planter::~Planter( void )
{
	MIX_ASSERT( m_pOctObj == NULL );

	MIX_RELEASE( m_pMaterial );
}

void Planter::Initialize(	Mix::Scene::IMaterial* pMaterial,
							UInt32 numModel, const Mix::Scene::IPlanter::GRASS_MODEL* models,
							UInt32 numUnit, const Mix::Scene::IPlanter::GRASS_UNIT* units,
							const wchar_t* pDebugName )
{
	MIX_ASSERT( pMaterial != NULL );
	MIX_ASSERT( numModel > 0 );
	MIX_ASSERT( models != NULL );
	MIX_ASSERT( numUnit > 0 );
	MIX_ASSERT( units != NULL );

	UInt32 i;
	const IPlanter::GRASS_MODEL* pSrcModel;
	Planter::INTERNAL_MODEL* pDstModel;
	Planter::INTERNAL_MODEL* pDstModelEnd;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_ADD_REF( pMaterial );
	m_pMaterial = pMaterial;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	m_ModelList.reserve( numModel );

	// eNX`W̐ݒ

	for( i = 0; i < numModel; i++ )
	{
		const Mix::Scene::IPlanter::GRASS_MODEL& src = models[i];
		const Mix::Vector2& srcUV0 = src.uv[0];
		const Mix::Vector2& srcUV1 = src.uv[1];

		Planter::INTERNAL_MODEL dst;
		Mix::Vector2* tex;
		Planter::INTERNAL_MODEL_DATA* dat = dst.dat;

		// SIMPLE
		tex = &( dst.texCoords[Planter::TEX_COORD_SIMPLE] );

		tex[0].Set( srcUV0.x, srcUV1.y );
		tex[1].Set( srcUV0.x, srcUV0.y );
		tex[2].Set( srcUV1.x, srcUV0.y );
		tex[3].Set( srcUV1.x, srcUV1.y );

		// CROSS1 and CROSS2
		if( src.quality != Mix::Scene::IPlanter::GQ_SIMPLE )
		{
			Float32 srcHalfU = ( srcUV0.x + srcUV1.x ) * 0.5f;

			tex = &( dst.texCoords[Planter::TEX_COORD_CROSS] );

			tex[0].Set( srcUV0.x, srcUV1.y );
			tex[1].Set( srcUV0.x, srcUV0.y );
			tex[2].Set( srcHalfU, srcUV0.y );
			tex[3].Set( srcHalfU, srcUV1.y );

			tex[4].Set( srcUV1.x, srcUV1.y );
			tex[5].Set( srcUV1.x, srcUV0.y );
			tex[6].Set( srcHalfU, srcUV0.y );
			tex[7].Set( srcHalfU, srcUV1.y );
		}

		switch( src.quality )
		{
		case IPlanter::GQ_SIMPLE:
			dat[Planter::DAT_NEAR].texCoordsPoint = 4;
			dat[Planter::DAT_NEAR].numPoint = 4;
			dat[Planter::DAT_NEAR].points = Planter::SIMPLE_POS_TABLE;

			dat[Planter::DAT_MID].texCoordsPoint = 4;
			dat[Planter::DAT_MID].numPoint = 4;
			dat[Planter::DAT_MID].points = Planter::SIMPLE_POS_TABLE;

			dat[Planter::DAT_FAR].texCoordsPoint = 4;
			dat[Planter::DAT_FAR].numPoint = 4;
			dat[Planter::DAT_FAR].points = Planter::SIMPLE_POS_TABLE;
			break;

		case IPlanter::GQ_CROSS1:
			dat[Planter::DAT_NEAR].texCoordsPoint = 8;
			dat[Planter::DAT_NEAR].numPoint = 16;
			dat[Planter::DAT_NEAR].points = Planter::CROSS1_POS_TABLE;

			dat[Planter::DAT_MID].texCoordsPoint = 8;
			dat[Planter::DAT_MID].numPoint = 16;
			dat[Planter::DAT_MID].points = Planter::CROSS1_POS_TABLE;

			dat[Planter::DAT_FAR].texCoordsPoint = 4;
			dat[Planter::DAT_FAR].numPoint = 4;
			dat[Planter::DAT_FAR].points = Planter::SIMPLE_POS_TABLE;
			break;

		case IPlanter::GQ_CROSS2:
			dat[Planter::DAT_NEAR].texCoordsPoint = 12;
			dat[Planter::DAT_NEAR].numPoint = 24;
			dat[Planter::DAT_NEAR].points = Planter::CROSS2_POS_TABLE;

			dat[Planter::DAT_MID].texCoordsPoint = 8;
			dat[Planter::DAT_MID].numPoint = 16;
			dat[Planter::DAT_MID].points = Planter::CROSS1_POS_TABLE;

			dat[Planter::DAT_FAR].texCoordsPoint = 4;
			dat[Planter::DAT_FAR].numPoint = 4;
			dat[Planter::DAT_FAR].points = Planter::SIMPLE_POS_TABLE;
			break;

		default:
			MIX_ASSERT( src.quality <= IPlanter::GQ_CROSS2 );
		}

		m_ModelList.push_back( dst );
	}

	// eNX`W̃|C^ݒ

	pSrcModel = &( models[0] );
	pDstModel = &( m_ModelList[0] );
	pDstModelEnd = pDstModel + m_ModelList.size();

	while( pDstModel != pDstModelEnd )
	{
		const Mix::Vector2* texCoords = pDstModel->texCoords;
		Planter::INTERNAL_MODEL_DATA* dat = pDstModel->dat;

		switch( pSrcModel->quality )
		{
		case IPlanter::GQ_SIMPLE:
			dat[Planter::DAT_NEAR].texCoords = &( texCoords[Planter::TEX_COORD_SIMPLE] );
			dat[Planter::DAT_MID].texCoords = &( texCoords[Planter::TEX_COORD_SIMPLE] );
			dat[Planter::DAT_FAR].texCoords = &( texCoords[Planter::TEX_COORD_SIMPLE] );
			break;

		case IPlanter::GQ_CROSS1:
		case IPlanter::GQ_CROSS2:
			dat[Planter::DAT_NEAR].texCoords = &( texCoords[Planter::TEX_COORD_CROSS] );
			dat[Planter::DAT_MID].texCoords = &( texCoords[Planter::TEX_COORD_CROSS] );
			dat[Planter::DAT_FAR].texCoords = &( texCoords[Planter::TEX_COORD_SIMPLE] );
			break;

		default:
			MIX_ASSERT( pSrcModel->quality <= IPlanter::GQ_CROSS2 );
		}

		pSrcModel++;
		pDstModel++;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////

	m_UnitList.reserve( numUnit );
	m_SubsetList.reserve( numUnit );

	for( i = 0; i < numUnit; i++ )
	{
		const IPlanter::GRASS_UNIT& srcUnit = units[i];

		MIX_ASSERT( srcUnit.modelIndex < m_ModelList.size() );

		Planter::INTERNAL_UNIT dstUnit;
		Mix::Scene::Common::WIDGET_SPHERE_SUBSET subset;

		// jbg //

		dstUnit.pModel = &( m_ModelList[srcUnit.modelIndex] );
		dstUnit.worldMat = srcUnit.worldMat;

		m_UnitList.push_back( dstUnit );

		// TuZbg //

		subset.refractClass = Mix::Scene::Common::RRC_UNKNOWN;
		subset.minZ = 0.0f;
		subset.maxZ = 0.0f;
		subset.quadCount = 0;
		subset.quads = NULL;
		subset.bounds.center = Mix::Vector3::Zero();
		subset.bounds.radius = 0.0f;

//		ȉ FlashSubsets Őݒ肳
//		  subset.priority;
//		  subset.pMaterial;

		m_SubsetList.push_back( subset );
	}

	FlashSubsets();

	////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG
	Mix::StringW tempStr;
	tempStr.Sprintf( L"%s/Quads", MIX_SAFE_NAME( pDebugName ) );
	m_Quads.Initialize( Planter::QUADS_DEF_SIZE, Planter::QUADS_RESIZE_STEP, tempStr.GetConstPtr() );
#else //_DEBUG
	m_Quads.Initialize( Planter::QUADS_DEF_SIZE, Planter::QUADS_RESIZE_STEP );
#endif //_DEBUG

	UpdateQuads( Planter::DAT_NEAR );
}

void Planter::TransformUnits( const Mix::Matrix4x4& mat )
{
	MIX_ASSERT( m_UnitList.size() > 0 );

	Planter::INTERNAL_UNIT* pUnit = &( m_UnitList[0] );
	Planter::INTERNAL_UNIT* pUnitEnd = pUnit + m_UnitList.size();

	while( pUnit != pUnitEnd )
	{
		pUnit->worldMat *= mat;
		pUnit++;
	}
}

void Planter::UpdateQuads( UInt32 index )
{
	m_Quads.Clear();

	if( index >= Planter::DAT_MAX )
	{
		return;
	}

	Planter::INTERNAL_UNIT* pUnitBegin = &( m_UnitList[0] );
	Planter::INTERNAL_UNIT* pUnitEnd = pUnitBegin + m_UnitList.size();
	Planter::INTERNAL_UNIT* pUnit;

	Mix::Scene::Common::WIDGET_SPHERE_SUBSET* pSubsetBegin = &( m_SubsetList[0] );
	Mix::Scene::Common::WIDGET_SPHERE_SUBSET* pSubset;

	Mix::Vector3 unitMin( +MIX_FLOAT_MAX, +MIX_FLOAT_MAX, +MIX_FLOAT_MAX );
	Mix::Vector3 unitMax( -MIX_FLOAT_MAX, -MIX_FLOAT_MAX, -MIX_FLOAT_MAX );
	Mix::Vector3 aabbMin( +MIX_FLOAT_MAX, +MIX_FLOAT_MAX, +MIX_FLOAT_MAX );
	Mix::Vector3 aabbMax( -MIX_FLOAT_MAX, -MIX_FLOAT_MAX, -MIX_FLOAT_MAX );

	UInt32 i;
	UInt32 j;

	pUnit = pUnitBegin;
	pSubset = pSubsetBegin;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// jbgATuZbg( ꕔ )̍XV
	////////////////////////////////////////////////////////////////////////////////////////////////////

	while( pUnit != pUnitEnd )
	{
		const Mix::Matrix4x4& worldMat = pUnit->worldMat;
		const Planter::INTERNAL_MODEL* pModel = pUnit->pModel;
		const Planter::INTERNAL_MODEL_DATA& dat = pModel->dat[index];
		UInt32 numPoint = dat.numPoint;
		const Mix::Vector3* points = dat.points;

		Mix::Scene::Common::WIDGET_QUAD* pQuad;
		Mix::Vector3* quadPoints;
		Mix::Vector3 quadCenter;

		pUnit->quadStart = m_Quads.GetCount();

		unitMin.Set( +MIX_FLOAT_MAX, +MIX_FLOAT_MAX, +MIX_FLOAT_MAX );
		unitMax.Set( -MIX_FLOAT_MAX, -MIX_FLOAT_MAX, -MIX_FLOAT_MAX );

		for( i = 0; i < numPoint; i += 4 )
		{
			pQuad = m_Quads.Add();

			quadPoints = &( pQuad->points[0] );
			quadCenter = Mix::Vector3::Zero();

			for( j = 0; j < 4; j++ )
			{
				quadPoints[j] = worldMat * points[i + j];
				quadCenter += quadPoints[j];

				unitMin = Mix::Vector3::Min( unitMin, quadPoints[j] );
				unitMax = Mix::Vector3::Max( unitMax, quadPoints[j] );
			}

			pQuad->pos = quadCenter * 0.25f; // quadCenter / 4
			pQuad->localPoints = NULL;
//			pQuad->points;
			pQuad->dataSize = 28; // color( Mix::Vector4 ) + normal( Mix::Vector3 )
			pQuad->color.Set( 1.0f, 1.0f, 1.0f, 1.0f );
			pQuad->normal = worldMat * Mix::Vector3( 0.0f, 1.0f, 0.0f );
			pQuad->texCoords = &( dat.texCoords[( i < dat.texCoordsPoint )? 0 : 4] );
			pQuad->texWriteOffset = 40; // pos( Mix::Vector3 ) + color( Mix::Vector4 ) + normal( Mix::Vector3 )
		}

		aabbMin = Mix::Vector3::Min( aabbMin, unitMin );
		aabbMax = Mix::Vector3::Max( aabbMax, unitMax );

		pSubset->quadCount = numPoint >> 2;
		pSubset->bounds.center = ( unitMin + unitMax ) * 0.5f;
		pSubset->bounds.radius = ( unitMax - unitMin ).GetLengthF() * 0.5f;

		pSubset++;
		pUnit++;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// TuZbg̃Nbh̃|C^ݒ
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pUnit = pUnitBegin;
	pSubset = pSubsetBegin;

	while( pUnit != pUnitEnd )
	{
		pSubset->quads = m_Quads.GetBeginPtr() + pUnit->quadStart;

		pSubset++;
		pUnit++;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// EXV
	////////////////////////////////////////////////////////////////////////////////////////////////////

	m_Bounds.min = aabbMin;
	m_Bounds.max = aabbMax;
	m_Bounds.ComputePoints();
}

void Planter::FlashSubsets( void )
{
	MIX_ASSERT( m_pMaterial != NULL );

	if( m_SubsetList.size() > 0 )
	{
		Mix::Scene::Common::WIDGET_SPHERE_SUBSET* pSubset = &( m_SubsetList[0] );
		Mix::Scene::Common::WIDGET_SPHERE_SUBSET* pSubsetEnd = pSubset + m_SubsetList.size();

		while( pSubset != pSubsetEnd )
		{
			pSubset->priority = m_TRPriority;
			pSubset->pMaterial = m_pMaterial;

			pSubset++;
		}
	}
}

void Planter::SetOctreeObject( Mix::Scene::Common::PlanterObject* pOctObj )
{
	if( m_pOctObj != NULL )
	{
		m_pOctObj->Destroy();
		m_pOctObj = NULL;
	}

	m_pOctObj = pOctObj;
}

Boolean Planter::InternalRefresh( Float32 foStartZ, Float32 foMidZ, Float32 foEndZ, Float32 z )
{
	MIX_ASSERT( m_bDraw == True );

	UInt32 nextStatus;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// Xe[^X߂
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( foStartZ > z )
	{
		nextStatus = Planter::DAT_NEAR;
	}
	else if( foMidZ > z )
	{
		nextStatus = Planter::DAT_MID;
	}
	else if( foEndZ > z )
	{
		nextStatus = Planter::DAT_FAR;
	}
	else
	{
		nextStatus = Planter::DAT_MAX;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// Xe[^XɉďԂϊ
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( nextStatus != m_Status )
	{
		m_Status = nextStatus;
		UpdateQuads( m_Status );
	}

	return ( m_Quads.IsEmpty() == False );
}

UInt32 Planter::GetSubsetCount( void ) const
{
	return m_SubsetList.size();
}

Mix::Scene::Common::WIDGET_SPHERE_SUBSET* Planter::GetSubsets( void )
{
	return &( m_SubsetList[0] );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Scene::IPlanter
////////////////////////////////////////////////////////////////////////////////////////////////////

const Mix::Geometry::AABB& Planter::GetBounds( void ) const
{
	return m_Bounds;
}

Boolean Planter::IsIgnored( void ) const
{
	return ( m_pOctObj != NULL )? m_pOctObj->IsIllegal() : True;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Scene::IWidget
////////////////////////////////////////////////////////////////////////////////////////////////////

void Planter::GetMaterial( Mix::Scene::IMaterial** ppMaterial )
{
	MIX_ASSERT( ppMaterial != NULL );

	MIX_ADD_REF( m_pMaterial );
	( *ppMaterial ) = m_pMaterial;
}

Boolean Planter::SetMaterial( Mix::Scene::IMaterial* pMaterial )
{
	if( pMaterial == NULL )
	{
		return False;
	}

#ifdef _DEBUG
	if( Planter::CheckMaterial( pMaterial, Planter::FAILED_SET_MATERIAL, m_DebugName.GetConstPtr() ) == False )
	{
		return False;
	}
#else //_DEBUG
	if( Planter::CheckMaterial( pMaterial, Planter::FAILED_SET_MATERIAL, Mix::STR_SAFE_NAME ) == False )
	{
		return False;
	}
#endif //_DEBUG

	if( m_pMaterial != pMaterial )
	{
		MIX_ADD_REF( pMaterial );
		m_pMaterial = pMaterial;

		FlashSubsets();
	}

	return True;
}

Boolean Planter::CanLocalLighting( void ) const
{
	MIX_ASSERT( m_pMaterial != NULL );

	return ( ( m_pMaterial->IsLighting() == True ) && ( m_bLocalLighting == True ) );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Scene::IEntity
////////////////////////////////////////////////////////////////////////////////////////////////////

Boolean Planter::IsDraw( void ) const
{
	return m_bDraw;
}

void Planter::SetDraw( Boolean state )
{
	m_bDraw = state;
}

Boolean Planter::IsLocalLighting( void ) const
{
	return m_bLocalLighting;
}

void Planter::SetLocalLighting( Boolean state )
{
	if( m_bLocalLighting != state )
	{
		m_bLocalLighting = state;

		FlashSubsets();
	}
}

UInt32 Planter::GetTransparencyPriority( void ) const
{
	return m_TRPriority;
}

void Planter::SetTransparencyPriority( UInt32 priority )
{
	if( m_TRPriority != priority )
	{
		m_TRPriority = priority;

		FlashSubsets();
	}
}

Mix::Scene::WATER_CONTAINS_RESULT Planter::GetDefaultWaterContainsResult( void ) const
{
	return m_WCResult;
}

void Planter::SetDefaultWaterContainsResult( Mix::Scene::WATER_CONTAINS_RESULT result )
{
	m_WCResult = result;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Scene::IRendererObject
////////////////////////////////////////////////////////////////////////////////////////////////////

Mix::Scene::IRendererObject::TYPE Planter::GetType( void ) const
{
	return IRendererObject::PLANTER;
}

}}}
