#include "Mix/Class/Scene/Common/MotionController.h"

#include "Mix/File/IReader.h"
#include "Mix/Scene/IMotionListener.h"

#include "Mix/Class/Scene/Common/ActorModel.h"
#include "Mix/Class/Scene/Common/MotionMixer.h"
#include "Mix/Class/Scene/Common/RootMotionCurve.h"
#include "Mix/Class/Scene/Common/MotionCurve.h"
#include "Mix/Class/Scene/Common/Motion.h"
#include "Mix/Class/Scene/Common/MotionState.h"
#include "Mix/Class/Scene/Common/MotionDeformer.h"
#include "Mix/Class/Scene/Common/MotionEntryProcesser.h"

#include <queue>

namespace Mix{ namespace Scene{ namespace Common{

const wchar_t* MotionController::FAILED_CREATE = L"[VRg[[̍쐬Ɏs";

////////////////////////////////////////////////////////////////////////////////////////////////////
// MotionController
////////////////////////////////////////////////////////////////////////////////////////////////////

MotionController* MotionController::CreateInstance(	Mix::Scene::Common::ActorModel* pModel,
													Mix::Scene::Common::MotionMixer* pMixer,
													const wchar_t* pName,
													UInt32 priority )
{
	return new MotionController( pModel, pMixer, pName, priority );
}

MotionController::MotionController(	Mix::Scene::Common::ActorModel* pModel,
									Mix::Scene::Common::MotionMixer* pMixer,
									const wchar_t* pName,
									UInt32 priority ) :
m_pModel( pModel ),
m_pMixer( pMixer ),
m_Name( pName ),
m_Priority( priority ),
m_Action( MotionController::ACTION_QUEUEING ),
m_TransitionFramesInv( 1.0f ),
m_TransitionFrame( 1.0f ),
m_TransitionRatio( 1.0f )
{
	for( UInt32 i = 0; i < MotionController::MOTION_MAX; i++ )
	{
		m_MotionTable[i] = NULL;
	}

#ifdef _DEBUG

	Mix::String debugName;

	debugName.Sprintf( L"%s/%s", pModel->GetName(), pName );
	m_CommandQueue.Initialize( MotionController::COMANND_DEF_CAPACITY, MotionController::COMANND_RESIZE_STEP, debugName.GetConstPtr() );

#else //_DEBUG

	m_CommandQueue.Initialize( MotionController::COMANND_DEF_CAPACITY, MotionController::COMANND_RESIZE_STEP );

#endif //_DEBUG
}

MotionController::~MotionController( void )
{
	for( MotionController::MotionMap::iterator it = m_MotionMap.begin(); it != m_MotionMap.end(); ++it )
	{
		MotionController::MOTION* pMotion = &( it->second );

		if( pMotion->pState != NULL )
		{
			pMotion->pState->Dispose();
		}

		MIX_RELEASE( pMotion->pInterface );
		MIX_RELEASE( pMotion->pState );
		MIX_RELEASE( pMotion->pDeformer );
		MIX_RELEASE( pMotion->pEntryProcesser );
		MIX_RELEASE( pMotion->pListener );
	}
}

const wchar_t* MotionController::GetName( void ) const
{
	return m_Name.GetConstPtr();
}

UInt32 MotionController::GetPriority( void ) const
{
	return m_Priority;
}

Mix::Scene::MOTION_HANDLE MotionController::GetHandle( const wchar_t* pName ) const
{
	if( ( pName == NULL ) ||
		( ::wcslen( pName ) == 0 ) )
	{
		return MOTION_HANDLE();
	}

	MotionController::HandleMap::const_iterator it_h = m_HandleMap.find( pName );
	if( it_h == m_HandleMap.end() )
	{
		return MOTION_HANDLE();
	}

	return it_h->second;
}

Mix::Scene::MOTION_HANDLE MotionController::Add( Mix::Scene::IMotion* pMotion )
{
	if( pMotion == NULL )
	{
		return NULL;
	}

	MotionController::HandleMap::iterator it = m_HandleMap.find( pMotion->GetName() );
	if( it != m_HandleMap.end() )
	{
		return NULL;
	}

	Mix::Scene::MOTION_HANDLE handle = ProcAdd( static_cast<Mix::Scene::Common::Motion*>( pMotion ) );
	if( handle != NULL )
	{
		MIX_ADD_REF( pMotion );
	}
	else
	{
		return NULL;
	}

	return handle;
}

Boolean MotionController::Remove( const Mix::Scene::MOTION_HANDLE& handle )
{
	MotionController::MotionMap::iterator it_m = m_MotionMap.find( handle );
	if( it_m == m_MotionMap.end() )
	{
		return False;
	}

	MotionController::HandleMap::iterator it_h = m_HandleMap.find( it_m->second.pInterface->GetName() );
	if( it_h == m_HandleMap.end() )
	{
		return False;
	}

	MotionController::MOTION* pMotion = &( it_m->second );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// [Ve[u͂
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( ( m_MotionTable[MotionController::MOTION_PREVIOUS] != NULL ) &&
		( m_MotionTable[MotionController::MOTION_PREVIOUS]->handle == handle ) )
	{
		m_MotionTable[MotionController::MOTION_PREVIOUS] = NULL;
	}
	else if(	( m_MotionTable[MotionController::MOTION_CURRENT] != NULL ) &&
				( m_MotionTable[MotionController::MOTION_CURRENT]->handle == handle ) )
	{
		m_MotionTable[MotionController::MOTION_PREVIOUS] = NULL;
		m_MotionTable[MotionController::MOTION_CURRENT] = NULL;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pMotion->pState != NULL )
	{
		pMotion->pState->Dispose();
	}

	MIX_RELEASE( pMotion->pInterface );
	MIX_RELEASE( pMotion->pState );
	MIX_RELEASE( pMotion->pDeformer );
	MIX_RELEASE( pMotion->pEntryProcesser );
	MIX_RELEASE( pMotion->pListener );

	m_MotionMap.erase( it_m );
	m_HandleMap.erase( it_h );

	return True;
}

Boolean MotionController::Get( const Mix::Scene::MOTION_HANDLE& handle, Mix::Scene::IMotion** ppMotion )
{
	MotionController::MotionMap::iterator it_m = m_MotionMap.find( handle );
	if( it_m == m_MotionMap.end() )
	{
		return False;
	}

	MIX_ADD_REF( it_m->second.pInterface );
	( *ppMotion ) = it_m->second.pInterface;

	return True;
}

Boolean MotionController::GetState( const Mix::Scene::MOTION_HANDLE& handle, Mix::Scene::IMotionState** ppMotionState )
{
	if( ppMotionState == NULL )
	{
		return False;
	}

	MotionController::MotionMap::iterator it_m = m_MotionMap.find( handle );
	if( it_m == m_MotionMap.end() )
	{
		return False;
	}

	MotionState* pState = it_m->second.pState;

	MIX_ADD_REF( pState );
	( *ppMotionState ) = pState;

	return True;
}

void MotionController::SetListener( Mix::Scene::IMotionListener* pListener )
{
	MotionController::MotionMap::iterator it_begin = m_MotionMap.begin();
	MotionController::MotionMap::iterator it_end = m_MotionMap.end();
	MotionController::MotionMap::iterator it;

	MIX_ADD_REF( pListener );

	for( it = it_begin; it != it_end; ++it )
	{
		MotionController::MOTION* pMotion = &( it->second );

		MIX_RELEASE( pMotion->pListener );
		pMotion->pListener = pListener;

		pMotion->pEntryProcesser->SetListener( pListener );
	}
}

Boolean MotionController::SetListener( const Mix::Scene::MOTION_HANDLE& handle, Mix::Scene::IMotionListener* pListener )
{
	MotionController::MotionMap::iterator it = m_MotionMap.find( handle );
	if( it == m_MotionMap.end() )
	{
		return False;
	}

	MotionController::MOTION* pMotion = &( it->second );

	MIX_ADD_REF( pListener );
	MIX_RELEASE( pMotion->pListener );
	pMotion->pListener = pListener;

	pMotion->pEntryProcesser->SetListener( pListener );

	return True;
}

void MotionController::SendCommand( Mix::Scene::MOTION_COMMAND_EXECUTE_TYPE exeType, const Mix::Scene::MOTION_COMMAND& com )
{
	if( exeType != Mix::Scene::MOTION_CE_DEFAULT )
	{
		m_CommandQueue.Drain();
	}

	if( ( m_CommandQueue.GetCount() > 0 ) &&
		( m_CommandQueue.GetTail().handle == com.handle ) )
	{
		//Ōƈꏏꍇ́A㏑
		m_CommandQueue[m_CommandQueue.GetTailIndex()] = com;
	}
	else
	{
		m_CommandQueue.Enqueue( com );
	}

	if( exeType == Mix::Scene::MOTION_CE_FORCE )
	{
		m_Action = MotionController::ACTION_QUEUEING;
	}
}

UInt32 MotionController::GetCommandCount( void ) const
{
	return m_CommandQueue.GetCount();
}

void MotionController::OnRootTransform( Mix::Container<Mix::Scene::Common::MOTION_ROOT_BLEND>& list )
{
	MotionController::MOTION* pCurMotion = m_MotionTable[MotionController::MOTION_CURRENT];
	if( pCurMotion == NULL )
	{
		return;
	}

	MotionController::MOTION* pPreMotion = m_MotionTable[MotionController::MOTION_PREVIOUS];

	if( ( ( pPreMotion != NULL ) && ( pPreMotion->pInterface->GetRootCurvePtr()->IsAvailable() == True ) ) ||
		( pCurMotion->pInterface->GetRootCurvePtr()->IsAvailable() == True ) )
	{
		Motion* pInterface;
		MotionState* pState;

		////////////////////////////////////////////////////////////////////////////////////////////////////
		// Õ[V
		////////////////////////////////////////////////////////////////////////////////////////////////////

		if( pPreMotion != NULL )
		{
			pInterface = pPreMotion->pInterface;
			pState = pPreMotion->pState;

			pInterface->GetRootCurvePtr()->Execute(	pInterface->GetBlendRatio() * pState->GetBlendRatio(),
													pPreMotion->preFrame,
													pState->GetFrame(),
													list );
		}

		////////////////////////////////////////////////////////////////////////////////////////////////////
		// ݂̃[V
		////////////////////////////////////////////////////////////////////////////////////////////////////

		pInterface = pCurMotion->pInterface;
		pState = pCurMotion->pState;

		pInterface->GetRootCurvePtr()->Execute(	pInterface->GetBlendRatio() * pState->GetBlendRatio() * m_TransitionRatio,
												pCurMotion->preFrame,
												pState->GetFrame(),
												list );
	}
}

void MotionController::OnBodyTransform( Mix::Container<Mix::Scene::Common::MOTION_BODY_BLEND>& list )
{
	if( m_MotionTable[MotionController::MOTION_CURRENT] == NULL )
	{
		return;
	}

	MotionController::MOTION* pMotion;
	MotionState* pState;

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Õ[V
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////

	pMotion = m_MotionTable[MotionController::MOTION_PREVIOUS];

	if( pMotion != NULL )
	{
		pState = pMotion->pState;

		pMotion->pDeformer->Execute(	pMotion->pInterface->GetBlendRatio() * pState->GetBlendRatio(),
										pState->GetFrame(),
										list );
	}

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// ݂̃[V
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////

	pMotion = m_MotionTable[MotionController::MOTION_CURRENT];
	pState = pMotion->pState;

	pMotion->pDeformer->Execute(	pMotion->pInterface->GetBlendRatio() * pState->GetBlendRatio() * m_TransitionRatio,
									pState->GetFrame(),
									list );
}

void MotionController::OnUpdate( Float32 dt )
{
	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ANV̏
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_Action == MotionController::ACTION_QUEUEING )
	{
		//R}h̃L[CO
		if( m_CommandQueue.GetCount() > 0 )
		{
			Mix::Scene::MOTION_COMMAND com = m_CommandQueue.Dequeue();

			if( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_ACTIVATE ) == Mix::Scene::MOTION_CF_ACTIVATE )
			{
				//Đ
				MotionController::MOTION* pCurMotion = m_MotionTable[MotionController::MOTION_CURRENT];

				if( ( pCurMotion == NULL ) ||
					( pCurMotion->handle != com.handle ) )
				{
					MotionController::MotionMap::iterator it = m_MotionMap.find( com.handle );
					if( it != m_MotionMap.end() )
					{
						MotionController::MOTION* pPreMotion = m_MotionTable[MotionController::MOTION_PREVIOUS];
						MotionController::MOTION* pNextMotion = &( it->second );

						if( ( pPreMotion != NULL ) &&
							( pPreMotion->handle == pNextMotion->handle ) )
						{
							//PREVIOUS Ɠ[VĐ悤ƂĂꍇ( PREVIOUS ̒~͍sȂ )

							//[Ve[u̐ݒ
							m_MotionTable[MotionController::MOTION_PREVIOUS] = pCurMotion;
							m_MotionTable[MotionController::MOTION_CURRENT] = pNextMotion;

							//[V( CURRENT ) : Xe[g̐ݒ
							if( pNextMotion->state == MotionController::MS_STOP )
							{
								pNextMotion->state = MotionController::MS_PLAY;
							}

							//[V( CURRENT ) : [v̐ݒ
							if( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_LOOP ) == Mix::Scene::MOTION_CF_LOOP )
							{
								if( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_LOOP_INHERIT ) != Mix::Scene::MOTION_CF_LOOP_INHERIT )
								{
									pNextMotion->pState->SetLoopCount( com.loopCount );
								}
							}
							else
							{
								pNextMotion->pState->SetLoopCount( 0 );
							}

							//[V( CURRENT ) : t[̐ݒ
							if( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_FRAME ) == Mix::Scene::MOTION_CF_FRAME )
							{
								if( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_FRAME_INHERIT ) != Mix::Scene::MOTION_CF_FRAME_INHERIT )
								{
									pNextMotion->preFrame = com.frame;
									pNextMotion->pState->SetFrame( com.frame );
								}
							}
							else
							{
								pNextMotion->preFrame = -1.0f;
								pNextMotion->pState->SetFrame( 0.0f );
							}

							//gWV̐ݒ
							if( ( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_TRANSITION ) == Mix::Scene::MOTION_CF_TRANSITION ) &&
								( MIX_FLOAT_IS_ZERO( com.transitionTimeLength ) == False ) )
							{
								Float32 fps = pNextMotion->pInterface->GetFramesPerSec();

								if( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_TRANSITION_INHERIT ) == Mix::Scene::MOTION_CF_TRANSITION_INHERIT )
								{
									//PREVIOUS p
									Float32 ratio = MIX_FLOAT_SATURATE( 1.0f - m_TransitionRatio );
									Float32 elapsedTime = com.transitionTimeLength * ratio;

									m_TransitionFramesInv = MIX_FLOAT_RECIPROCAL( com.transitionTimeLength * fps );
									m_TransitionFrame = elapsedTime * fps;
									m_TransitionRatio = ratio;
								}
								else
								{
									//PREVIOUS pȂ
									m_TransitionFramesInv = MIX_FLOAT_RECIPROCAL( com.transitionTimeLength * fps );
									m_TransitionFrame = 0.0f;
									m_TransitionRatio = 0.0f;
								}
							}
							else
							{
								m_TransitionFramesInv = 1.0f;
								m_TransitionFrame = 1.0f;
								m_TransitionRatio = 1.0f;
							}

							//ANV̐ݒ
							m_Action = MotionController::ACTION_FADEIN;
						}
						else
						{
							//PREVIOUS CURRENT ƂႤV[VĐ悤ƂĂꍇ

							//[V( PREVIOUS ) ̒~
							Stop( pPreMotion );

							//[Ve[u̐ݒ
							m_MotionTable[MotionController::MOTION_PREVIOUS] = pCurMotion;
							m_MotionTable[MotionController::MOTION_CURRENT] = pNextMotion;

							//[V( CURRENT ) : Xe[g̐ݒ
							pNextMotion->state = MotionController::MS_PLAY;

							//[V( CURRENT ) : [v̐ݒ
							if( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_LOOP ) == Mix::Scene::MOTION_CF_LOOP )
							{
								pNextMotion->pState->SetLoopCount( com.loopCount );
							}
							else
							{
								pNextMotion->pState->SetLoopCount( 0 );
							}

							//[V( CURRENT ) : t[̐ݒ
							if( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_FRAME ) == Mix::Scene::MOTION_CF_FRAME )
							{
								pNextMotion->preFrame = com.frame;
								pNextMotion->pState->SetFrame( com.frame );
							}
							else
							{
								pNextMotion->preFrame = -1.0f;
								pNextMotion->pState->SetFrame( 0.0f );
							}

							//gWV̐ݒ
							if( ( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_TRANSITION ) == Mix::Scene::MOTION_CF_TRANSITION ) &&
								( MIX_FLOAT_IS_ZERO( com.transitionTimeLength ) == False ) )
							{
								m_TransitionFramesInv = MIX_FLOAT_RECIPROCAL( com.transitionTimeLength * pNextMotion->pInterface->GetFramesPerSec() );
								m_TransitionFrame = 0.0f;
								m_TransitionRatio = 0.0f;
							}
							else
							{
								m_TransitionFramesInv = 1.0f;
								m_TransitionFrame = 1.0f;
								m_TransitionRatio = 1.0f;
							}

							//ANV̐ݒ
							m_Action = MotionController::ACTION_FADEIN;
						}
					}
				}
				else
				{
					//CURRENT Ɠ[VĐ悤ƂĂꍇ
					//
					//[vł͂ȂAŌ̃t[ɓBĂԂŁu[vvpݒĂꍇ́A
					//xĐԂɂ͂Ȃ邪ɒ~ԂɂȂB

					//[V( CURRENT ) : Xe[g̐ݒ
					if( pCurMotion->state == MotionController::MS_STOP )
					{
						pCurMotion->state = MotionController::MS_PLAY;
					}

					//[V( CURRENT ) : [v̐ݒ
					if( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_LOOP ) == Mix::Scene::MOTION_CF_LOOP )
					{
						if( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_LOOP_INHERIT ) != Mix::Scene::MOTION_CF_LOOP_INHERIT )
						{
							pCurMotion->pState->SetLoopCount( com.loopCount );
						}
					}
					else
					{
						pCurMotion->pState->SetLoopCount( 0 );
					}

					//[V( CURRENT ) : t[̐ݒ
					if( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_FRAME ) == Mix::Scene::MOTION_CF_FRAME )
					{
						if( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_FRAME_INHERIT ) != Mix::Scene::MOTION_CF_FRAME_INHERIT )
						{
							pCurMotion->preFrame = com.frame;
							pCurMotion->pState->SetFrame( com.frame );
						}
					}
					else
					{
						pCurMotion->preFrame = -1.0f;
						pCurMotion->pState->SetFrame( 0.0f );
					}

					//gWV : ݒ
					if( ( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_TRANSITION ) == Mix::Scene::MOTION_CF_TRANSITION ) &&
						( MIX_FLOAT_IS_ZERO( com.transitionTimeLength ) == False ) )
					{
						Float32 fps = pCurMotion->pInterface->GetFramesPerSec();

						if( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_TRANSITION_INHERIT ) == Mix::Scene::MOTION_CF_TRANSITION_INHERIT )
						{
							m_TransitionFramesInv = MIX_FLOAT_RECIPROCAL( com.transitionTimeLength * fps );
							m_TransitionFrame = com.transitionTimeLength * m_TransitionRatio * fps;
						}
						else
						{
							m_TransitionFramesInv = MIX_FLOAT_RECIPROCAL( com.transitionTimeLength * fps );
							m_TransitionFrame = 0.0f;
							m_TransitionRatio = 0.0f;
						}
					}
					else
					{
						m_TransitionFramesInv = 1.0f;
						m_TransitionFrame = 1.0f;
						m_TransitionRatio = 1.0f;
					}

					//ANV̐ݒ
					m_Action = MotionController::ACTION_FADEIN;
				}
			}
			else
			{
				//~
				MotionController::MOTION* pCurMotion = m_MotionTable[MotionController::MOTION_CURRENT];

				if( pCurMotion != NULL )
				{
					MotionController::MOTION* pPreMotion = m_MotionTable[MotionController::MOTION_PREVIOUS];

					//[V( PREVIOUS ) ̒~
					Stop( pPreMotion );

					//[Ve[u̐ݒ
					m_MotionTable[MotionController::MOTION_PREVIOUS] = NULL;

					//[V( CURRENT ) : [v̐ݒ
					if( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_LOOP ) == Mix::Scene::MOTION_CF_LOOP )
					{
						pCurMotion->pState->SetLoopCount( com.loopCount );
					}

					//[V( CURRENT ) : t[̐ݒ
					if( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_FRAME ) == Mix::Scene::MOTION_CF_FRAME )
					{
						pCurMotion->preFrame = com.frame;
						pCurMotion->pState->SetFrame( com.frame );
					}

					//gWV̐ݒ
					if( ( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_TRANSITION ) == Mix::Scene::MOTION_CF_TRANSITION ) &&
						( MIX_FLOAT_IS_ZERO( com.transitionTimeLength ) == False ) )
					{
						Float32 fps = pCurMotion->pInterface->GetFramesPerSec();
						Float32 ratio = MIX_FLOAT_SATURATE( 1.0f - m_TransitionRatio );

						m_TransitionFramesInv = MIX_FLOAT_RECIPROCAL( com.transitionTimeLength * fps );
						m_TransitionFrame = com.transitionTimeLength * ratio * fps;
					}
					else
					{
						m_TransitionFramesInv = 1.0f;
						m_TransitionFrame = 1.0f;
						m_TransitionRatio = 0.0f;
					}

					//ANV̐ݒ
					m_Action = MotionController::ACTION_FADEOUT;
				}
			}
		}
	}

	if( m_Action != MotionController::ACTION_QUEUEING )
	{
		//gWV
		MotionController::MOTION* pCurMotion = m_MotionTable[MotionController::MOTION_CURRENT];
		if( pCurMotion != NULL )
		{
			Motion* pCurInterface = pCurMotion->pInterface;
			Float32 curFrameStep = pCurInterface->GetFramesPerSec() * dt * ( pCurInterface->GetSpeed() * pCurMotion->pState->GetSpeed() );

			if( m_Action == MotionController::ACTION_FADEIN )
			{
				MotionController::MOTION* pPreMotion = m_MotionTable[MotionController::MOTION_PREVIOUS];
				MotionController::MOTION* pCurMotion = m_MotionTable[MotionController::MOTION_CURRENT];

				MIX_ASSERT( pCurMotion != NULL );

				//tF[hC
				m_TransitionFrame += curFrameStep;
				m_TransitionRatio = m_TransitionFrame * m_TransitionFramesInv;

				if( m_TransitionRatio >= 1.0f )
				{
					//[V( PREVIOUS ) ̒~
					Stop( m_MotionTable[MotionController::MOTION_PREVIOUS] );

					//[Ve[u̐ݒ
					m_MotionTable[MotionController::MOTION_PREVIOUS] = NULL;

					m_TransitionRatio = 1.0f;
					m_Action = MotionController::ACTION_QUEUEING;
				}
				else
				{
					if( pPreMotion != NULL )
					{
						pPreMotion->pState->SetTransitionRatio( 1.0f - m_TransitionRatio );
					}
				}

				pCurMotion->pState->SetTransitionRatio( m_TransitionRatio );
			}
			else if( m_Action == MotionController::ACTION_FADEOUT )
			{
				MotionController::MOTION* pCurMotion = m_MotionTable[MotionController::MOTION_CURRENT];

				MIX_ASSERT( pCurMotion != NULL );

				//tF[hAEg
				m_TransitionFrame += curFrameStep;
				m_TransitionRatio = 1.0f - ( m_TransitionFrame * m_TransitionFramesInv );

				if( m_TransitionRatio <= 0.0f )
				{
					//[V( CURRENT ) ̒~
					Stop( pCurMotion );

					//[Ve[u̐ݒ
					m_MotionTable[MotionController::MOTION_CURRENT] = NULL;

					m_TransitionRatio = 0.0f;
					m_Action = MotionController::ACTION_QUEUEING;
				}
				else
				{
					pCurMotion->pState->SetTransitionRatio( m_TransitionRatio );
				}
			}
		}
		else
		{
			m_Action = MotionController::ACTION_QUEUEING;
		}
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// [V̏
	////////////////////////////////////////////////////////////////////////////////////////////////////

	for( UInt32 i = 0; i < MotionController::MOTION_MAX; i++ )
	{
		MotionController::MOTION* pMotion = m_MotionTable[i];
		if( pMotion == NULL )
		{
			continue;
		}

		if( pMotion->state == MotionController::MS_PLAY )
		{
			//Gg[ʒm̃Zbg
			pMotion->pEntryProcesser->ResetNotify();

			//ĐJn̗Uv
			if( pMotion->pListener != NULL )
			{
				pMotion->pListener->OnActivate( pMotion->pInterface, pMotion->handle, True );
			}

			pMotion->state = MotionController::MS_PLAYING;
		}
		else if( pMotion->state == MotionController::MS_PLAYING )
		{
			Motion* pInterface = pMotion->pInterface;
			MotionState* pState = pMotion->pState;
			Float32 frameStep = pInterface->GetFramesPerSec() * dt * ( pInterface->GetSpeed() * pMotion->pState->GetSpeed() );
			Float32 frame = pMotion->pState->GetFrame();
			UInt32 loopCount = pState->GetLoopCount();

			pMotion->preFrame = frame;
			frame += frameStep;

			if( loopCount > 0 )
			{
				Float32 startFrame = pInterface->GetLoopStartFrame();
				Float32 endFrame = pInterface->GetLoopEndFrame();

				if( endFrame < frame )
				{
					Float32 frameLength = endFrame - startFrame;

					//ԏubNĂ܂ۂ̑΍
					if( MIX_FLOAT_IS_ZERO( frameLength ) == True )
					{
						frame = 0.0f;
					}
					else
					{
						do { frame -= frameLength; } while( endFrame < frame );
					}

					//[v
					if( loopCount != Mix::Scene::MOTION_INFINITE_LOOP )
					{
						loopCount--;

						//[vJEg̐ݒ
						pState->SetLoopCount( loopCount );
					}

					//Gg[̒ʒm
					pMotion->pEntryProcesser->Notify( pMotion->preFrame, endFrame );
					pMotion->pEntryProcesser->ResetNotify();
					pMotion->pEntryProcesser->Notify( startFrame, frame );
				}
				else
				{
					//Gg[̒ʒm
					pMotion->pEntryProcesser->Notify( pMotion->preFrame, frame );
				}
			}
			else
			{
				Float32 lastFrame = pInterface->GetLastFrame();

				//t[̐ݒ
				if( lastFrame < frame )
				{
					frame = lastFrame;

					Stop( pMotion );
				}

				//Gg[̒ʒm
				pMotion->pEntryProcesser->Notify( pMotion->preFrame, frame );
			}

			//t[̐ݒ
			pMotion->pState->SetFrame( frame );
		}
	}
}

void MotionController::OnReset( void )
{
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// tF[h̏I
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_Action != MotionController::ACTION_QUEUEING )
	{
		MIX_ASSERT( ( m_Action == MotionController::ACTION_FADEIN ) || ( m_Action == MotionController::ACTION_FADEOUT ) );

		//gWV
		if( m_MotionTable[MotionController::MOTION_CURRENT] != NULL )
		{
			if( m_Action == MotionController::ACTION_FADEIN )
			{
				//tF[hC
				m_TransitionFrame = MIX_FLOAT_RECIPROCAL( m_TransitionFramesInv );
				m_TransitionRatio = 1.0f;

				//Õ[V - ~
				Stop( m_MotionTable[MotionController::MOTION_PREVIOUS] );
				m_MotionTable[MotionController::MOTION_PREVIOUS] = NULL;

				//݂̃[V - Đ
				m_MotionTable[MotionController::MOTION_CURRENT]->pState->SetTransitionRatio( 1.0f );
			}
			else if( m_Action == MotionController::ACTION_FADEOUT )
			{
				//tF[hAEg
				m_TransitionFrame = MIX_FLOAT_RECIPROCAL( m_TransitionFramesInv );
				m_TransitionRatio = 0.0f;

				//݂̃[V - ~
				Stop( m_MotionTable[MotionController::MOTION_CURRENT] );
				m_MotionTable[MotionController::MOTION_CURRENT] = NULL;
			}
		}

		m_Action = MotionController::ACTION_QUEUEING;
	}

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// L[CO
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////

	while( m_CommandQueue.IsEmpty() == False )
	{
		MIX_ASSERT( m_Action == MotionController::ACTION_QUEUEING );

		Mix::Scene::MOTION_COMMAND com = m_CommandQueue.Dequeue();
	
		if( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_ACTIVATE ) == Mix::Scene::MOTION_CF_ACTIVATE )
		{
			//Đ
			MotionController::MotionMap::iterator it = m_MotionMap.find( com.handle );

			if( it != m_MotionMap.end() )
			{
				MotionController::MOTION* pPreMotion = m_MotionTable[MotionController::MOTION_PREVIOUS];
				MotionController::MOTION* pCurMotion = m_MotionTable[MotionController::MOTION_CURRENT];
				MotionController::MOTION* pNextMotion = &( it->second );

				//Õ[V - ~
				Stop( pPreMotion );
				m_MotionTable[MotionController::MOTION_PREVIOUS] = pCurMotion;

				//݂̃[V - Đ
				m_MotionTable[MotionController::MOTION_CURRENT] = pNextMotion;

				//݂̃[V - Xe[g̐ݒ
				pNextMotion->state = MotionController::MS_PLAY;

				//݂̃[V - [v̐ݒ
				if( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_LOOP ) == Mix::Scene::MOTION_CF_LOOP )
				{
					pNextMotion->pState->SetLoopCount( com.loopCount );
				}
				else
				{
					pNextMotion->pState->SetLoopCount( 0 );
				}

				//݂̃[V - t[̐ݒ
				if( MIX_TESTBIT( com.flags, Mix::Scene::MOTION_CF_FRAME ) == Mix::Scene::MOTION_CF_FRAME )
				{
					pNextMotion->preFrame = com.frame;
					pNextMotion->pState->SetFrame( com.frame );
				}
				else
				{
					pNextMotion->preFrame = -1.0f;
					pNextMotion->pState->SetFrame( 0.0f );
				}

				//݂̃[V - gWV̐ݒ
				m_TransitionFramesInv = 1.0f;
				m_TransitionFrame = 1.0f;
				m_TransitionRatio = 1.0f;

				//݂̃[V - Gg[ʒm̃Zbg
				pNextMotion->pEntryProcesser->ResetNotify();

				//݂̃[V - ĐJn̒ʒm
				if( pNextMotion->pListener != NULL )
				{
					pNextMotion->pListener->OnActivate( pNextMotion->pInterface, pNextMotion->handle, True );
				}

				//݂̃[V - Xe[g̐ݒ
				pNextMotion->state = MotionController::MS_PLAYING;
			}
		}
		else
		{
			//~
			MotionController::MOTION* pCurMotion = m_MotionTable[MotionController::MOTION_CURRENT];

			if( pCurMotion != NULL )
			{
				MotionController::MOTION* pPreMotion = m_MotionTable[MotionController::MOTION_PREVIOUS];

				//Õ[V - ~
				Stop( pPreMotion );
				m_MotionTable[MotionController::MOTION_PREVIOUS] = NULL;

				//݂̃[V - ~
				Stop( pCurMotion );
				m_MotionTable[MotionController::MOTION_CURRENT] = NULL;

				//gWV̐ݒ
				m_TransitionFramesInv = 1.0f;
				m_TransitionFrame = 1.0f;
				m_TransitionRatio = 0.0f;
			}
		}
	}
}

void MotionController::Stop( MotionController::MOTION* pMotion )
{
	if( ( pMotion == NULL ) ||
		( pMotion->state == MotionController::MS_STOP ) )
	{
		return;
	}

	//Xe[^X~Ԃ
	pMotion->state = MotionController::MS_STOP;

	//ĐI̒ʒm
	if( pMotion->pListener != NULL )
	{
		pMotion->pListener->OnActivate( pMotion->pInterface, pMotion->handle, False );
	}

	//p[^̃Zbg
	pMotion->pState->SetFrame( 0.0f );
	pMotion->pState->SetLoopCount( 0 );
	pMotion->pState->SetTransitionRatio( 0.0f );
}

Boolean MotionController::AddMotion( Mix::Scene::Common::Motion* pMotion )
{
	MIX_ASSERT( pMotion != NULL );

#ifdef _DEBUG
	MotionController::HandleMap::iterator it_h = m_HandleMap.find( pMotion->GetName() );
	MIX_ASSERT( it_h == m_HandleMap.end() );
#endif //_DEBUG

	return ( ProcAdd( pMotion ) != NULL );
}

Boolean MotionController::CloneMotion( MotionController* pDstController, Boolean bShared, Boolean bDuplicateEntries )
{
	MotionController::MotionMap::iterator it_begin = m_MotionMap.begin();
	MotionController::MotionMap::iterator it_end = m_MotionMap.end();
	MotionController::MotionMap::iterator it;

	if( bShared == True )
	{
		for( it = it_begin; it != it_end; ++it )
		{
			Motion* pMotion = it->second.pInterface;

			MIX_ADD_REF( pMotion );
			pDstController->AddMotion( pMotion );
		}
	}
	else
	{
		for( it = it_begin; it != it_end; ++it )
		{
			Motion* pSrcMotion = it->second.pInterface;
			IMotion* pDstMotion = NULL;

			if( pSrcMotion->Clone( &pDstMotion, bDuplicateEntries ) == False )
			{
				return False;
			}

			if( pDstController->AddMotion( static_cast<Motion*>( pDstMotion ) ) == False )
			{
				MIX_RELEASE( pDstMotion );
				return False;
			}
		}
	}

	return True;
}

Mix::Scene::MOTION_HANDLE MotionController::ProcAdd( Mix::Scene::Common::Motion* pMotion )
{
	Mix::Scene::MOTION_HANDLE handle( this, pMotion );
	Mix::Scene::Common::MotionState* pState = Mix::Scene::Common::MotionState::CreateInstance( this, handle );
	Mix::Scene::Common::MotionEntryProcesser* pEntryProcesser = pMotion->AddEntryProcesser();

	if( ( pState == NULL ) ||
		( pEntryProcesser == NULL ) )
	{
		MIX_RELEASE( pState );
		MIX_RELEASE( pEntryProcesser );

		return NULL;
	}

	pEntryProcesser->SetHandle( handle );

	m_HandleMap.insert( MotionController::HandleMap::value_type( pMotion->GetName(), handle ) );
	m_MotionMap.insert( MotionController::MotionMap::value_type( handle, MotionController::MOTION( handle, pMotion, pState, pEntryProcesser ) ) );

	return handle;
}

UInt32 MotionController::GetCondition( const Mix::Scene::MOTION_HANDLE& handle ) const
{
	const MotionController::MOTION* pPreMotion = m_MotionTable[MotionController::MOTION_PREVIOUS];
	const MotionController::MOTION* pCurMotion = m_MotionTable[MotionController::MOTION_CURRENT];

	UInt32 ret = 0;
/*
	if( m_CommandQueue.IsScheduled( handle ) == True )
	{
		ret |= Mix::Scene::MOTION_COND_WATING;
	}
*/
	if( ( m_CommandQueue.GetCount() > 0 ) &&
		( m_CommandQueue.GetTail().handle == handle ) )
	{
		ret |= Mix::Scene::MOTION_COND_WATING;
	}

	if( ( pPreMotion != NULL ) &&
		( pPreMotion->handle == handle ) &&
		( pPreMotion->state != MotionController::MS_STOP ) )
	{
		ret |= Mix::Scene::MOTION_COND_ACTIVE;
	}
	else if(	( pCurMotion != NULL ) &&
				( pCurMotion->handle == handle ) &&
				( pCurMotion->state != MotionController::MS_STOP ) )
	{
		ret |= Mix::Scene::MOTION_COND_CUR_ACTIVE;
	}

	return ret;
}

Boolean MotionController::Finalize( void )
{
	////////////////////////////////////////////////////////////////////////////////////////////////////
	// [ṼJ[uWJ
	////////////////////////////////////////////////////////////////////////////////////////////////////

	for( MotionController::MotionMap::iterator it = m_MotionMap.begin(); it != m_MotionMap.end(); ++it )
	{
		MotionController::MOTION* pMotion = &( it->second );

		MotionDeformer* pDeformer = MotionDeformer::CreateInstance( m_pModel, pMotion->pInterface );
		if( pDeformer != NULL )
		{
			pMotion->pDeformer = pDeformer;
		}
		else
		{
			return False;
		}
	}

	return True;
}

void MotionController::Dispose( void )
{
	m_pModel = NULL;
	m_pMixer = NULL;
}

}}}
