#include "Mix/Class/Scene/Common/Manager.h"

#include "Mix/File/IManager.h"
#include "Mix/File/IReader.h"
#include "Mix/Graphics/IManager.h"
#include "Mix/Graphics/IDevice.h"
#include "Mix/Scene/IMaterial.h"

#include "Mix/Class/Scene/Common/Renderer.h"
#include "Mix/Class/Scene/Standard/DX9/Factory.h"
#include "Mix/Class/Scene/Standard/DX10/Factory.h"
#include "Mix/Class/Scene/Standard/DX11/Factory.h"

#ifdef _DEBUG
	#include "Mix/Class/Scene/Common/Debug.h"
#endif //_DEBUG

namespace Mix{ namespace Scene{ namespace Common{

Manager* Manager::CreateInstance( void )
{
	return new Manager();
}

Manager::Manager( void ) :
m_pFileManager( NULL ),
m_pGraphicsDevice( NULL )
{
	////////////////////////////////////////////////////////////////////////////////////////////////////

	Mix::File::IManager* pFileMgr = Mix::File::GetManagerPtr();

	MIX_ASSERT( pFileMgr != NULL );

	MIX_ADD_REF( pFileMgr );
	m_pFileManager = pFileMgr;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	Mix::Graphics::IManager* pGraphcisMgr = Mix::Graphics::GetManagerPtr();

	MIX_ASSERT( pGraphcisMgr != NULL );

	pGraphcisMgr->GetDevice( &m_pGraphicsDevice );
	MIX_ASSERT( m_pGraphicsDevice != NULL );
}

Manager::~Manager( void )
{
	MIX_ASSERT( m_RendererList.size() == 0 );
	MIX_ASSERT( m_UpdateMaterialMap.size() == 0 );
	MIX_ASSERT( m_pGraphicsDevice == NULL );
	MIX_ASSERT( m_pFileManager == NULL );
}

void Manager::Update( Float32 dt )
{
	Manager::RendererList::iterator it_rdr_begin = m_RendererList.begin();
	Manager::RendererList::iterator it_rdr_end = m_RendererList.end();
	Manager::RendererList::iterator it_rdr;

	Manager::UpdateMaterialMap::iterator it_um_begin = m_UpdateMaterialMap.begin();
	Manager::UpdateMaterialMap::iterator it_um_end = m_UpdateMaterialMap.end();
	Manager::UpdateMaterialMap::iterator it_um;

	Mix::Scene::Common::Renderer* pRenderer;
	Mix::Scene::IMaterial* pMaterial;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// _[̃tBjbV
	////////////////////////////////////////////////////////////////////////////////////////////////////

	for( it_rdr = it_rdr_begin; it_rdr != it_rdr_end; ++it_rdr )
	{
		pRenderer = ( *it_rdr );
		pRenderer->Finish();
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// }eA(gAj[V)̃Abvf[g
	////////////////////////////////////////////////////////////////////////////////////////////////////

	for( it_um = it_um_begin; it_um != it_um_end; ++it_um )
	{
		pMaterial = it_um->first;
		pMaterial->Update( dt );
	}
}

void Manager::AddUpdateMaterial( Mix::Scene::IMaterial* pMaterial )
{
	MIX_ASSERT( pMaterial != NULL );

	Manager::UpdateMaterialMap::iterator it = m_UpdateMaterialMap.find( pMaterial );

	MIX_ADD_REF( pMaterial );

	if( it == m_UpdateMaterialMap.end() )
	{
		m_UpdateMaterialMap.insert( Manager::UpdateMaterialMap::value_type( pMaterial, Mix::AtomicValue( 1 ) ) );

		//QƂ
		it = m_UpdateMaterialMap.find( pMaterial );
		MIX_ASSERT( it != m_UpdateMaterialMap.end() );
	}
	else
	{
		//QƂCNg
		it->second.Increment();
	}
}

void Manager::RemoveUpdateMaterial( Mix::Scene::IMaterial* pMaterial )
{
	MIX_ASSERT( pMaterial != NULL );

	Manager::UpdateMaterialMap::iterator it = m_UpdateMaterialMap.find( pMaterial );

	if( it != m_UpdateMaterialMap.end() )
	{
		if( it->second.Get() == 1 )
		{
			m_UpdateMaterialMap.erase( it );
		}
		else
		{
			it->second.Decrement();
		}

		MIX_RELEASE( pMaterial );
	}
}

void Manager::AddRenderer( Mix::Scene::Common::Renderer* pRenderer )
{
	MIX_ASSERT( pRenderer != NULL );
	MIX_ASSERT( std::find( m_RendererList.begin(), m_RendererList.end(), pRenderer ) == m_RendererList.end() );

	m_RendererList.push_back( pRenderer );
}

void Manager::RemoveRenderer( Mix::Scene::Common::Renderer* pRenderer )
{
	MIX_ASSERT( pRenderer != NULL );
	MIX_ASSERT( std::find( m_RendererList.begin(), m_RendererList.end(), pRenderer ) != m_RendererList.end() );

	Mix::Vector_FirstErase( m_RendererList, pRenderer );
}

void Manager::Dispose( void )
{
	MIX_RELEASE( m_pGraphicsDevice );
	MIX_RELEASE( m_pFileManager );
}

Boolean Manager::CreateFactory( const wchar_t* pMEPDirPath, const wchar_t* pMEPName, Mix::Scene::IFactory** ppFactory )
{
	if( ( pMEPDirPath == NULL ) ||
		( pMEPName == NULL ) ||
		( wcslen( pMEPName ) == 0 ) ||
		( ppFactory == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pMEPDirPath[%s] pMEPName[%s] ppFactory[%s]",
			Factory::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_STR( pMEPDirPath ),
			MIX_LOG_STR( pMEPName ),
			MIX_LOG_PTR( ppFactory ) );

		return False;
	}

	Mix::Graphics::SHADER_MODEL shaderModel = m_pGraphicsDevice->GetShaderModel();
	Mix::StringW mepFileName;
	Mix::StringW mepFilePath;
	Factory::MEP_FILE_HEADER mepFileHeader;
	Factory::MEP_DATA_HEADER mepDataHeader;
	Mix::File::IReader* pMEPReader = NULL;
	Mix::Scene::Common::Factory* pFactory = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"V[t@Ng[̍쐬" );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// }eAGtFNgpbP[W̃f[^wb_擾
	////////////////////////////////////////////////////////////////////////////////////////////////////

	/*
		t@CpX̍쐬
	*/

	mepFileName.Sprintf( L"%s%s", pMEPName, Factory::MEP_FILEEXT_TABLE[shaderModel] );

	if( Mix::CombinePath( pMEPDirPath, mepFileName.GetConstPtr(), mepFilePath ) == False )
	{
		MIX_LOG_ERROR( L"%s : }eAGtFNgpbP[W̃t@CpX쐬ł܂ł : MEPDirPath[%s] MEPName[%s] MEPFileName[%s]",
			Factory::FAILED_CREATE,
			pMEPDirPath,
			pMEPName,
			mepFileName.GetConstPtr() );

		MIX_LOG_INFO_SECT_END();

		return False;
	}

	/*
		f[^wb_擾
	*/

	if( m_pFileManager->CreateFileReader( mepFilePath.GetConstPtr(), &pMEPReader ) == False )
	{
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	if( Factory::MEP_LoadHeader( pMEPReader, mepFileHeader, mepDataHeader, mepFilePath.GetConstPtr() ) == False )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_RELEASE( pMEPReader );
		return False;
	}

	MIX_LOG_INFO( L"MEP : FilePath[%s]", mepFilePath.GetConstPtr() );
	MIX_LOG_INFO( L"MEP : Version[0x%08x]", mepFileHeader.version );
	MIX_LOG_INFO( L"MEP : RendererType[%s]", Factory::MEP_RENDERER_TYPE_TABLE[mepDataHeader.rendererType] );
	MIX_LOG_INFO( L"MEP : ProcessFlags[0x%08x]", mepDataHeader.processFlags );
	MIX_LOG_INFO( L"MEP : VertexShaderNum[%d]", mepDataHeader.vertexShaderNum );
	MIX_LOG_INFO( L"MEP : PixelShaderNum[%d]", mepDataHeader.pixelShaderNum );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( mepDataHeader.rendererType == Factory::MEP_RT_STANDARD )
	{
		/*
			X^_[h
		*/

		switch( shaderModel )
		{
		case Mix::Graphics::SHADER_MODEL_3:
			pFactory = Mix::Scene::Standard::DX9::Factory::CreateInstance();
			break;
		case Mix::Graphics::SHADER_MODEL_4:
			pFactory = Mix::Scene::Standard::DX10::Factory::CreateInstance();
			break;
		case Mix::Graphics::SHADER_MODEL_5:
			pFactory = Mix::Scene::Standard::DX11::Factory::CreateInstance();
			break;

		default:
			MIX_LOG_ERROR( L"%s : T|[gȂVF[_[fw肳Ă܂", Factory::FAILED_CREATE );
			MIX_LOG_INFO_SECT_END();

			MIX_RELEASE( pMEPReader );

			return False;
		}
	}
	else
	{
		MIX_LOG_ERROR( L"%s : T|[gȂ_[w肳Ă܂ : RendererType[%d]", Factory::FAILED_CREATE, mepDataHeader.rendererType );
		MIX_LOG_INFO_SECT_END();

		MIX_RELEASE( pMEPReader );

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pFactory->Initialize(	this,
								pMEPReader,
								mepDataHeader.processFlags,
								mepDataHeader.vertexShaderNum,
								mepDataHeader.pixelShaderNum,
								mepFilePath.GetConstPtr() ) == False )
	{
		MIX_LOG_INFO_SECT_END();

		MIX_RELEASE( pMEPReader );

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ㏈
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_RELEASE( pMEPReader );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppFactory ) = pFactory;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

const Mix::Vector4& Manager::Debug_GetDrawColor( Mix::Scene::DEBUG_DRAW_COLOR_TYPE type ) const
{
#ifdef _DEBUG
	return Mix::Scene::Common::Debug::GetDrawColor( type );
#else //_DEBUG
	return Mix::Vector4::Zero();
#endif //_DEBUG
}

void Manager::Debug_SetDrawColor( Mix::Scene::DEBUG_DRAW_COLOR_TYPE type, const Mix::Vector4& color )
{
#ifdef _DEBUG
	return Mix::Scene::Common::Debug::SetDrawColor( type, color );
#endif //_DEBUG
}

}}}
