#include "Mix/Class/Scene/Common/Factory.h"

#include "Mix/IEngine.h"
#include "Mix/File/IManager.h"
#include "Mix/File/IReader.h"
#include "Mix/Graphics/IManager.h"
#include "Mix/Graphics/IDevice.h"
#include "Mix/Graphics/IVertexLayout.h"
#include "Mix/Graphics/IVertexShader.h"
#include "Mix/Graphics/IPixelShader.h"
#include "Mix/Dynamics/IManager.h"

#include "Mix/Class/Scene/Common/Renderer.h"
#include "Mix/Class/Scene/Common/Motion.h"
#include "Mix/Class/Scene/Common/DefaultParticleSystem.h"
#include "Mix/Class/Scene/Common/BillboardParticleSystem.h"
#include "Mix/Class/Scene/Common/LeavingParticle.h"
#include "Mix/Class/Scene/Common/ScatterParticle.h"
#include "Mix/Class/Scene/Common/ActorLoader.h"
#include "Mix/Class/Scene/Common/ActorModel.h"
#include "Mix/Class/Scene/Common/TerrainLoader.h"
#include "Mix/Class/Scene/Common/TerrainModel.h"

#include "Mix/Class/Scene/Standard/Common/DefaultMaterial.h"

namespace Mix{ namespace Scene{ namespace Common{

////////////////////////////////////////////////////////////////////////////////////////////////////
// Factory : 萔
////////////////////////////////////////////////////////////////////////////////////////////////////

const wchar_t* Factory::FAILED_CREATE              = L"V[t@Ng̍쐬Ɏs";
const wchar_t* Factory::FAILED_CREATE_MEP_IO_ERROR = L"ǂݍƂt@C̓}eAGtFNgpbP[Wł͂܂";

//}eAGtFNgpbP[W̃t@Č
const wchar_t* Factory::MEP_FILEEXT_TABLE[Mix::Graphics::SHADER_MODEL_MAX] =
{
	L"_3.mep",
	L"_4.mep",
	L"_5.mep",
};

const wchar_t* Factory::MEP_RENDERER_TYPE_TABLE[Factory::MEP_RT_MAX] =
{
	L"Standard",
};

//o[ebNCAEg : TvO
static const Mix::Graphics::VERTEX_ELEMENT VL_SAMPLING[] =
{
	{ Mix::Graphics::VLS_POSITION, 0, Mix::Graphics::VLF_FLOAT32_4,  0, },
	{ Mix::Graphics::VLS_TEXTURE,  0, Mix::Graphics::VLF_FLOAT32_2, 16, },
};
static const UInt32 VL_SAMPLING_COUNT = sizeof( VL_SAMPLING ) / sizeof( Mix::Graphics::VERTEX_ELEMENT );
static const wchar_t* VL_SAMPLING_NAME = L"Sampling";

//o[ebNCAEg : KEVAu[
static const Mix::Graphics::VERTEX_ELEMENT VL_GAUSSIAN_BLUR[] =
{
	{ Mix::Graphics::VLS_POSITION, 0, Mix::Graphics::VLF_FLOAT32_4,  0, },
	{ Mix::Graphics::VLS_TEXTURE,  0, Mix::Graphics::VLF_FLOAT32_2, 16, },
};
static const UInt32 VL_GAUSSIAN_BLUR_COUNT = sizeof( VL_GAUSSIAN_BLUR ) / sizeof( Mix::Graphics::VERTEX_ELEMENT );
static const wchar_t* VL_GAUSSIAN_BLUR_NAME = L"GaussianBlur";

//o[ebNCAEg : XJCh[
static const Mix::Graphics::VERTEX_ELEMENT VL_SKYDOME[] =
{
	{ Mix::Graphics::VLS_POSITION, 0, Mix::Graphics::VLF_FLOAT32_4,  0, },
	{ Mix::Graphics::VLS_TEXTURE,  0, Mix::Graphics::VLF_FLOAT32_2, 16, },
};
static const UInt32 VL_SKYDOME_COUNT = sizeof( VL_SKYDOME ) / sizeof( Mix::Graphics::VERTEX_ELEMENT );
static const wchar_t* VL_SKYDOME_NAME = L"SkyDome";

//o[ebNCAEg : YtA( S[Xg )
static const Mix::Graphics::VERTEX_ELEMENT VL_LF_GHOST[] =
{
	{ Mix::Graphics::VLS_POSITION, 0, Mix::Graphics::VLF_FLOAT32_4,  0, },
	{ Mix::Graphics::VLS_COLOR,    0, Mix::Graphics::VLF_FLOAT32_4, 16, },
	{ Mix::Graphics::VLS_TEXTURE,  0, Mix::Graphics::VLF_FLOAT32_2, 32, },
};
static const UInt32 VL_LF_GHOST_COUNT = sizeof( VL_LF_GHOST ) / sizeof( Mix::Graphics::VERTEX_ELEMENT );
static const wchar_t* VL_LF_GHOST_NAME = L"LensFlare(Ghost)";

////////////////////////////////////////////////////////////////////////////////////////////////////
// Manager::VERTEX_EFFECT
////////////////////////////////////////////////////////////////////////////////////////////////////

Factory::VERTEX_EFFECT::VERTEX_EFFECT( void )
{
	pLayout = NULL;

	for( UInt32 i = 0; i < Factory::VES_MAX; i++ )
	{
		pShader[i] = NULL;
	}
}

void Factory::VERTEX_EFFECT::Dispose( void )
{
	MIX_RELEASE( pLayout );

	for( UInt32 i = 0; i < Factory::VES_MAX; i++ )
	{
		MIX_RELEASE( pShader[i] );
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Manager::PIXEL_EFFECT
////////////////////////////////////////////////////////////////////////////////////////////////////

Factory::PIXEL_EFFECT::PIXEL_EFFECT( void )
{
	for( UInt32 i = 0; i < Factory::PES_MAX; i++ )
	{
		pShader[i] = NULL;
	}
}

void Factory::PIXEL_EFFECT::Dispose( void )
{
	for( UInt32 i = 0; i < Factory::PES_MAX; i++ )
	{
		MIX_RELEASE( pShader[i] );
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Factory
////////////////////////////////////////////////////////////////////////////////////////////////////

Factory::Factory( void ) :
m_pEngine( NULL ),
m_pFileManager( NULL ),
m_pGraphicsDevice( NULL ),
m_pActorLoader( NULL ),
m_pTerrainLoader( NULL ),
m_MEPProcessFlags( 0 )
{
	////////////////////////////////////////////////////////////////////////////////////////////////////

	Mix::IEngine* pEngine = Mix::GetEnginePtr();

	MIX_ASSERT( pEngine != NULL );

	MIX_ADD_REF( pEngine );
	m_pEngine = pEngine;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	Mix::File::IManager* pFileMgr = Mix::File::GetManagerPtr();

	MIX_ASSERT( pFileMgr != NULL );

	MIX_ADD_REF( pFileMgr );
	m_pFileManager = pFileMgr;


	////////////////////////////////////////////////////////////////////////////////////////////////////

	Mix::Graphics::IManager* pGraphicsManager = Mix::Graphics::GetManagerPtr();

	MIX_ASSERT( pGraphicsManager != NULL );

	if( pGraphicsManager->GetDevice( &m_pGraphicsDevice ) == False )
	{
		MIX_ASSERT( m_pGraphicsDevice != NULL );
	}
}

Factory::~Factory( void )
{
	for( Factory::VertexEffectMap::iterator it = m_ResidentVEMap.begin(); it != m_ResidentVEMap.end(); ++it )
	{
		it->second.Dispose();
	}

	for( Factory::PixelEffectMap::iterator it = m_ResidentPEMap.begin(); it != m_ResidentPEMap.end(); ++it )
	{
		it->second.Dispose();
	}

	for( Factory::VertexEffectMap::iterator it = m_MaterialVEMap.begin(); it != m_MaterialVEMap.end(); ++it )
	{
		it->second.Dispose();
	}

	for( Factory::PixelEffectMap::iterator it = m_MaterialPEMap.begin(); it != m_MaterialPEMap.end(); ++it )
	{
		it->second.Dispose();
	}

	MIX_DELETE( m_pTerrainLoader );
	MIX_DELETE( m_pActorLoader );

	MIX_RELEASE( m_pGraphicsDevice );
	MIX_RELEASE( m_pFileManager );

	MIX_RELEASE( m_pEngine );
}

Mix::IEngine* Factory::GetEnginePtr( void ) const
{
	return m_pEngine;
}

Boolean Factory::Initialize( Mix::Scene::Common::Manager* pSceneManager, Mix::File::IReader* pReader, UInt32 mepProcFlags, UInt32 mepVSNum, UInt32 mepPSNum, const wchar_t* pFilePath )
{
	MIX_ASSERT( pSceneManager != NULL );
	MIX_ASSERT( pReader != NULL );
	MIX_ASSERT( m_pActorLoader == NULL );
	MIX_ASSERT( m_pTerrainLoader == NULL );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// AN^[[_[̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	m_pActorLoader = new Mix::Scene::Common::ActorLoader( this );
	if( m_pActorLoader == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : FilePath[%s]", Factory::FAILED_CREATE, Mix::STR_OUTOFMEMORY, pFilePath );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n`[_[̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	m_pTerrainLoader = new Mix::Scene::Common::TerrainLoader( this );
	if( m_pTerrainLoader == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : FilePath[%s]", Factory::FAILED_CREATE, Mix::STR_OUTOFMEMORY, pFilePath );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// }eAGtFNgpbP[W̓ǂݍ
	////////////////////////////////////////////////////////////////////////////////////////////////////

	std::vector<UInt8> buff;
	Mix::StringW tempStr;
	std::vector<Mix::Graphics::VERTEX_ELEMENT> elements;

	/*
		vZXtO
	*/

	m_MEPProcessFlags = mepProcFlags;

	/*
		o[ebNXVF[_[
	*/

	for( UInt32 i = 0; i < mepVSNum; i++ )
	{
		Factory::MEP_VS_HEADER vsHeader;
		Factory::VERTEX_EFFECT vsEffect;

		//wb_
		if( pReader->Read( &vsHeader, sizeof( vsHeader ) ) == sizeof( vsHeader ) )
		{
			MIX_ASSERT( vsHeader.buffSize[Factory::VES_MATERIAL_COLOR] > 0 );
			MIX_ASSERT( vsHeader.elementNum > 0 );
		}
		else
		{
			MIX_LOG_ERROR( L"%s : %s(1) : FilePath[%s]", Factory::FAILED_CREATE, Factory::FAILED_CREATE_MEP_IO_ERROR, pFilePath );
			return False;
		}

		//VF[_[
		for( UInt32 j = 0; j < Factory::VES_MATERIAL_MAX; j++ )
		{
			UInt32 buffSize = vsHeader.buffSize[j];

			if( buffSize == 0 )
			{
				continue;
			}

			buff.clear();
			buff.resize( buffSize );

			if( pReader->Read( &( buff[0] ), buff.size() ) == buffSize )
			{
				tempStr.Sprintf( L"MEP/VS_%08x_%d", vsHeader.id, j );

				if( m_pGraphicsDevice->CreateVertexShaderFromMemory( &( buff[0] ), buff.size(), &( vsEffect.pShader[j] ), tempStr.GetConstPtr() ) == False )
				{
					return False;
				}
			}
			else
			{
				MIX_LOG_ERROR( L"%s : %s(2) : FilePath[%s]", Factory::FAILED_CREATE, Factory::FAILED_CREATE_MEP_IO_ERROR, pFilePath );
				return False;
			}
		}

		//CAEg
		elements.clear();
		elements.resize( vsHeader.elementNum );

		if( pReader->Read( &( elements[0] ), sizeof( Mix::Graphics::VERTEX_ELEMENT ) * elements.size() ) == ( sizeof( Mix::Graphics::VERTEX_ELEMENT ) * vsHeader.elementNum ) )
		{
			tempStr.Sprintf( L"MEP/VL_%08x", vsHeader.id );

			if( m_pGraphicsDevice->CreateVertexLayout( &( elements[0] ), elements.size(), vsEffect.pShader[Factory::VES_DEFAULT], &( vsEffect.pLayout ), tempStr.GetConstPtr() ) == False )
			{
				return False;
			}
		}
		else
		{
			MIX_LOG_ERROR( L"%s : %s(3) : FilePath[%s]", Factory::FAILED_CREATE, Factory::FAILED_CREATE_MEP_IO_ERROR, pFilePath );
			return False;
		}

		m_MaterialVEMap.insert( Factory::VertexEffectMap::value_type( vsHeader.id, vsEffect ) );
	}

	/*
		sNZVF[_[
	*/

	for( UInt32 i = 0; i < mepPSNum; i++ )
	{
		Factory::MEP_PS_HEADER psHeader;
		Factory::PIXEL_EFFECT psEffect;

		//wb_
		if( pReader->Read( &psHeader, sizeof( psHeader ) ) == sizeof( psHeader ) )
		{
			MIX_ASSERT( psHeader.buffSize[Factory::PES_MATERIAL_COLOR] > 0 );
		}
		else
		{
			MIX_LOG_ERROR( L"%s : %s(4) : FilePath[%s]", Factory::FAILED_CREATE, Factory::FAILED_CREATE_MEP_IO_ERROR, pFilePath );
			return False;
		}

		//VF[_[
		for( UInt32 j = 0; j < Factory::PES_MATERIAL_MAX; j++ )
		{
			UInt32 buffSize = psHeader.buffSize[j];
			if( buffSize == 0 )
			{
				continue;
			}

			buff.clear();
			buff.resize( buffSize );

			if( pReader->Read( &( buff[0] ), buff.size() ) == buffSize )
			{
				tempStr.Sprintf( L"MEP/PS_%08x_%d", psHeader.id, j );

				if( m_pGraphicsDevice->CreatePixelShaderFromMemory( &( buff[0] ), buff.size(), &( psEffect.pShader[j] ), tempStr.GetConstPtr() ) == False )
				{
					return False;
				}
			}
			else
			{
				MIX_LOG_ERROR( L"%s : %s(5) : FilePath[%s]", Factory::FAILED_CREATE, Factory::FAILED_CREATE_MEP_IO_ERROR, pFilePath );
				return False;
			}
		}

		m_MaterialPEMap.insert( Factory::PixelEffectMap::value_type( psHeader.id, psEffect ) );
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Factory::MEP_LoadHeader( Mix::File::IReader* pReader, Factory::MEP_FILE_HEADER& fileHeader, Factory::MEP_DATA_HEADER& dataHeader, const wchar_t* pFilePath )
{
	MIX_ASSERT( pReader != NULL );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// t@Cwb_
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pReader->Read( &fileHeader, sizeof( fileHeader ) ) != sizeof( fileHeader ) )
	{
		MIX_LOG_ERROR( L"%s : %s(1) : FilePath[%s]",
			Factory::FAILED_CREATE,
			Factory::FAILED_CREATE_MEP_IO_ERROR,
			pFilePath );

		return False;
	}

	if( fileHeader.magicNumber != Factory::MEP_MAGICNUMBER )
	{
		MIX_LOG_ERROR( L"%s : ǂݍƂt@C̓}eAGtFNgpbP[Wł͂܂ : FilePath[%s]",
			Factory::FAILED_CREATE,
			pFilePath );

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// f[^wb_
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( fileHeader.version == Factory::MEP_VERSION )
	{
		if( pReader->Read( &dataHeader, sizeof( dataHeader ) ) != sizeof( dataHeader ) )
		{
			MIX_LOG_ERROR( L"%s : %s(2) : FilePath[%s]",
				Factory::FAILED_CREATE,
				Factory::FAILED_CREATE_MEP_IO_ERROR,
				pFilePath );

			return False;
		}
	}
	else if( fileHeader.version == Factory::MEP_VERSION_OLD )
	{
		//_[^Cv
		dataHeader.rendererType = Factory::MEP_RT_STANDARD;

		//vZXtO
		if( pReader->Read( &( dataHeader.processFlags ), sizeof( dataHeader.processFlags ) ) != sizeof( dataHeader.processFlags ) )
		{
			MIX_LOG_ERROR( L"%s : %s(3-1) : FilePath[%s]",
				Factory::FAILED_CREATE,
				Factory::FAILED_CREATE_MEP_IO_ERROR,
				pFilePath );

			return False;
		}

		//o[ebNXVF[_[̐
		if( pReader->Read( &( dataHeader.vertexShaderNum ), sizeof( dataHeader.vertexShaderNum ) ) != sizeof( dataHeader.vertexShaderNum ) )
		{
			MIX_LOG_ERROR( L"%s : %s(3-2) : FilePath[%s]",
				Factory::FAILED_CREATE,
				Factory::FAILED_CREATE_MEP_IO_ERROR,
				pFilePath );

			return False;
		}

		//sNZVF[_[̐
		if( pReader->Read( &( dataHeader.pixelShaderNum ), sizeof( dataHeader.pixelShaderNum ) ) != sizeof( dataHeader.pixelShaderNum ) )
		{
			MIX_LOG_ERROR( L"%s : %s(3-3) : FilePath[%s]",
				Factory::FAILED_CREATE,
				Factory::FAILED_CREATE_MEP_IO_ERROR,
				pFilePath );

			return False;
		}
	}
	else
	{
		UInt32 curVersion = Factory::MEP_VERSION;
		UInt32 oldVersion = Factory::MEP_VERSION_OLD;

		MIX_LOG_ERROR( L"%s : }eAGtFNgpbP[W̃o[WÂ܂ : FilePath[%s] FileVersion[0x%08x] SupportVersion[0x%08x, 0x%08x]",
			Factory::FAILED_CREATE,
			pFilePath,
			fileHeader.version,
			curVersion,
			oldVersion );

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

UInt32 Factory::MEP_GetProcessFlags( void ) const
{
	return m_MEPProcessFlags;
}

Factory::VERTEX_EFFECT* Factory::GetVertexEffectPtr( Factory::EFFECT_TYPE type, UInt32 opt )
{
	Factory::VERTEX_EFFECT* pEffect = NULL;

	if( type == Factory::E_RESIDENT )
	{
		Factory::VertexEffectMap::iterator it = m_ResidentVEMap.find( opt );

		if( it != m_ResidentVEMap.end() )
		{
			pEffect = &( it->second );
		}
		else
		{
			Factory::SHADER_BINARY binarys[Factory::VES_MAX] = { NULL };
			UInt32 binaryCount = 0;

			const Mix::Graphics::VERTEX_ELEMENT* elements = NULL;
			UInt32 elementCount = 0;
			const wchar_t* pLayoutName = NULL;

			switch( opt )
			{
			case Factory::ER_SAMPLING:
				GetShaderBinary( Factory::SHADER_SAMPLING_VS, binarys[Factory::VES_DEFAULT] );
				binaryCount = 1;
				elements = &( VL_SAMPLING[0] );
				elementCount = VL_SAMPLING_COUNT;
				pLayoutName = VL_SAMPLING_NAME;
				break;
			case Factory::ER_GAUSSIAN_BLUR:
				GetShaderBinary( Factory::SHADER_GAUSSIAN_BLUR_X_VS, binarys[Factory::VES_GAUSSIAN_BLUR_X] );
				GetShaderBinary( Factory::SHADER_GAUSSIAN_BLUR_Y_VS, binarys[Factory::VES_GAUSSIAN_BLUR_Y] );
				binaryCount = 2;
				elements = &( VL_GAUSSIAN_BLUR[0] );
				elementCount = VL_GAUSSIAN_BLUR_COUNT;
				pLayoutName = VL_GAUSSIAN_BLUR_NAME;
				break;

			case Factory::ER_SKYDOME_DEFAULT:
				GetShaderBinary( Factory::SHADER_SKYDOME_DEFAULT_VS, binarys[Factory::VES_DEFAULT] );
				binaryCount = 1;
				elements = &( VL_SKYDOME[0] );
				elementCount = VL_SKYDOME_COUNT;
				pLayoutName = VL_SKYDOME_NAME;
				break;
			case Factory::ER_SKYDOME_ATMOSPHERE:
				GetShaderBinary( Factory::SHADER_SKYDOME_ATMOSPHERE_VS, binarys[Factory::VES_DEFAULT] );
				binaryCount = 1;
				elements = &( VL_SKYDOME[0] );
				elementCount = VL_SKYDOME_COUNT;
				pLayoutName = VL_SKYDOME_NAME;
				break;

			case Factory::ER_SKYDOME_PANORAMA_D:
				GetShaderBinary( Factory::SHADER_SKYDOME_PANORAMA_VS, binarys[Factory::VES_SKYDOME_PANORAMA] );
				GetShaderBinary( Factory::SHADER_SKYDOME_PANORAMA_VS, binarys[Factory::VES_SKYDOME_PANORAMA_F] );
				binaryCount = 2;
				elements = &( VL_SKYDOME[0] );
				elementCount = VL_SKYDOME_COUNT;
				pLayoutName = VL_SKYDOME_NAME;
				break;
			case Factory::ER_SKYDOME_PANORAMA_A:
				GetShaderBinary( Factory::SHADER_SKYDOME_PANORAMA_VS,    binarys[Factory::VES_SKYDOME_PANORAMA] );
				GetShaderBinary( Factory::SHADER_SKYDOME_PANORAMA_FA_VS, binarys[Factory::VES_SKYDOME_PANORAMA_F] );
				binaryCount = 2;
				elements = &( VL_SKYDOME[0] );
				elementCount = VL_SKYDOME_COUNT;
				pLayoutName = VL_SKYDOME_NAME;
				break;
			}

			MIX_ASSERT( binaryCount <= Factory::VES_MAX );

			if( ( binaryCount > 0 ) &&
				( elementCount > 0 ) )
			{
				Boolean bSuccess = True;
				Factory::VERTEX_EFFECT ve;

				for( UInt32 i = 0; ( i < binaryCount ) && ( bSuccess == True ); i++ )
				{
					const Factory::SHADER_BINARY* psb = &( binarys[i] );

					if( m_pGraphicsDevice->CreateVertexShaderFromMemory( psb->pData, psb->dataSize, &( ve.pShader[i] ), psb->name ) == False )
					{
						bSuccess = False;
					}
				}

				if( bSuccess == True )
				{
					MIX_ASSERT( ve.pShader[Factory::VES_DEFAULT] != NULL );

					if( m_pGraphicsDevice->CreateVertexLayout( elements, elementCount, ve.pShader[Factory::VES_DEFAULT], &( ve.pLayout ), pLayoutName ) == False )
					{
						bSuccess = False;
					}
				}

				if( bSuccess == True )
				{
					m_ResidentVEMap.insert( Factory::VertexEffectMap::value_type( opt, ve ) );

					it = m_ResidentVEMap.find( opt );
					if( it != m_ResidentVEMap.end() )
					{
						pEffect = &( it->second );
					}
					else
					{
						ve.Dispose();
					}
				}
				else
				{
					ve.Dispose();
				}
			}
		}
	}
	else if( type == Factory::E_MATERIAL )
	{
		Factory::VertexEffectMap::iterator it = m_MaterialVEMap.find( opt );
		if( it != m_MaterialVEMap.end() )
		{
			pEffect = &( it->second );
		}
	}

	return pEffect;
}

Factory::PIXEL_EFFECT* Factory::GetPixelEffectPtr( Factory::EFFECT_TYPE type, UInt32 opt )
{
	Factory::PIXEL_EFFECT* pEffect = NULL;

	if( type == Factory::E_RESIDENT )
	{
		Factory::PixelEffectMap::iterator it = m_ResidentPEMap.find( opt );
		if( it != m_ResidentPEMap.end() )
		{
			pEffect = &( it->second );
		}
		else
		{
			Factory::SHADER_BINARY binarys[Factory::PES_MAX] = { NULL };
			UInt32 binaryCount = 0;
			Boolean bSuccess = True;

			Factory::PIXEL_EFFECT pe;

			switch( opt )
			{
			case Factory::ER_SAMPLING:
				GetShaderBinary( Factory::SHADER_SAMPLING_COPY_PS, binarys[Factory::PES_SAMPLING_COPY] );
				GetShaderBinary( Factory::SHADER_SAMPLING_DOWN2X_PS, binarys[Factory::PES_SAMPLING_DOWN2X] );
				GetShaderBinary( Factory::SHADER_SAMPLING_DOWN4X_PS, binarys[Factory::PES_SAMPLING_DOWN4X] );
				binaryCount = 3;
				break;

			case Factory::ER_BLUR3X:
				GetShaderBinary( Factory::SHADER_BLUR3X_X_PS, binarys[Factory::PES_COMMON_BLUR_X] );
				GetShaderBinary( Factory::SHADER_BLUR3X_Y_PS, binarys[Factory::PES_COMMON_BLUR_Y] );
				binaryCount = 2;
				break;
			case Factory::ER_BLUR5X:
				GetShaderBinary( Factory::SHADER_BLUR5X_X_PS, binarys[Factory::PES_COMMON_BLUR_X] );
				GetShaderBinary( Factory::SHADER_BLUR5X_Y_PS, binarys[Factory::PES_COMMON_BLUR_Y] );
				binaryCount = 2;
				break;

			case Factory::ER_GAUSSIAN_BLUR:
				GetShaderBinary( Factory::SHADER_GAUSSIAN_BLUR_X_PS, binarys[Factory::PES_COMMON_BLUR_X] );
				GetShaderBinary( Factory::SHADER_GAUSSIAN_BLUR_Y_PS, binarys[Factory::PES_COMMON_BLUR_Y] );
				binaryCount = 2;
				break;

			case Factory::ER_GAUSSIAN_BLUR_EX:
				GetShaderBinary( Factory::SHADER_GAUSSIAN_BLUR_EX_PS, binarys[Factory::PES_DEFAULT] );
				binaryCount = 1;
				break;

			case Factory::ER_BRIGHTPASS:
				GetShaderBinary( Factory::SHADER_BP____PS, binarys[Factory::PES_BP___] );
				GetShaderBinary( Factory::SHADER_BP_L__PS, binarys[Factory::PES_BP_L_] );
				GetShaderBinary( Factory::SHADER_BP_LA_PS, binarys[Factory::PES_BP_LA] );
				binaryCount = 3;
				break;

			case Factory::ER_SKYDOME_DEFAULT:
				GetShaderBinary( Factory::SHADER_SKYDOME_DEFAULT______PS, binarys[Factory::PES_SD_DEF_____] );
				GetShaderBinary( Factory::SHADER_SKYDOME_DEFAULT__C___PS, binarys[Factory::PES_SD_DEF__C__] );
				GetShaderBinary( Factory::SHADER_SKYDOME_DEFAULT__C2__PS, binarys[Factory::PES_SD_DEF__C2_] );
				GetShaderBinary( Factory::SHADER_SKYDOME_DEFAULT_B____PS, binarys[Factory::PES_SD_DEF_B___] );
				GetShaderBinary( Factory::SHADER_SKYDOME_DEFAULT_BC___PS, binarys[Factory::PES_SD_DEF_BC__] );
				GetShaderBinary( Factory::SHADER_SKYDOME_DEFAULT_BC2__PS, binarys[Factory::PES_SD_DEF_BC2_] );
				binaryCount = 6;
				break;
			case Factory::ER_SKYDOME_ATMOSPHERE:
				GetShaderBinary( Factory::SHADER_SKYDOME_ATMOSPHERE______PS, binarys[Factory::PES_SD_ASS_____] );
				GetShaderBinary( Factory::SHADER_SKYDOME_ATMOSPHERE__C___PS, binarys[Factory::PES_SD_ASS__C__] );
				GetShaderBinary( Factory::SHADER_SKYDOME_ATMOSPHERE__C_L_PS, binarys[Factory::PES_SD_ASS__C_L] );
				GetShaderBinary( Factory::SHADER_SKYDOME_ATMOSPHERE__C2__PS, binarys[Factory::PES_SD_ASS__C2_] );
				GetShaderBinary( Factory::SHADER_SKYDOME_ATMOSPHERE__C2L_PS, binarys[Factory::PES_SD_ASS__C2L] );
				GetShaderBinary( Factory::SHADER_SKYDOME_ATMOSPHERE_B____PS, binarys[Factory::PES_SD_ASS_B___] );
				GetShaderBinary( Factory::SHADER_SKYDOME_ATMOSPHERE_BC___PS, binarys[Factory::PES_SD_ASS_BC__] );
				GetShaderBinary( Factory::SHADER_SKYDOME_ATMOSPHERE_BC_L_PS, binarys[Factory::PES_SD_ASS_BC_L] );
				GetShaderBinary( Factory::SHADER_SKYDOME_ATMOSPHERE_BC2__PS, binarys[Factory::PES_SD_ASS_BC2_] );
				GetShaderBinary( Factory::SHADER_SKYDOME_ATMOSPHERE_BC2L_PS, binarys[Factory::PES_SD_ASS_BC2L] );
				binaryCount = 10;
				break;

			case Factory::ER_SKYDOME_PANORAMA_D:
				GetShaderBinary( Factory::SHADER_SKYDOME_PANORAMA_PS, binarys[Factory::PES_SD_PANORAMA] );
				GetShaderBinary( Factory::SHADER_SKYDOME_PANORAMA_FD_PS, binarys[Factory::PES_SD_PANORAMA_F] );
				binaryCount = 2;
				break;
			case Factory::ER_SKYDOME_PANORAMA_A:
				GetShaderBinary( Factory::SHADER_SKYDOME_PANORAMA_PS, binarys[Factory::PES_SD_PANORAMA] );
				GetShaderBinary( Factory::SHADER_SKYDOME_PANORAMA_FA_PS, binarys[Factory::PES_SD_PANORAMA_F] );
				binaryCount = 2;
				break;

			case Factory::ER_SSAO:
				GetShaderBinary( Factory::SHADER_SSAO_PS, binarys[Factory::PES_SSAO] );
				GetShaderBinary( Factory::SHADER_SSAO_TEX_PS, binarys[Factory::PES_SSAO_TEX] );
				binaryCount = 2;
				break;

			case Factory::ER_SM:
				GetShaderBinary( Factory::SHADER_SM_DOWN2X_PS, binarys[Factory::PES_SM_DOWN2X] );
				GetShaderBinary( Factory::SHADER_SM_DOWN4X_PS, binarys[Factory::PES_SM_DOWN4X] );
				GetShaderBinary( Factory::SHADER_SM_FINISH_PS, binarys[Factory::PES_SM_FINISH] );
				binaryCount = 3;
				break;

			case Factory::ER_UNDER_WATER:
				GetShaderBinary( Factory::SHADER_UNDER_WATER_PS, binarys[Factory::PES_DEFAULT] );
				binaryCount = 1;
				break;

			case Factory::ER_LUMINANCE:
				GetShaderBinary( Factory::SHADER_LUM_TRANSFORM_INITAL_PS,  binarys[Factory::PES_LUM_TR_INITAL] );
				GetShaderBinary( Factory::SHADER_LUM_TRANSFORM_ITERATE_PS, binarys[Factory::PES_LUM_TR_ITERATE] );
				GetShaderBinary( Factory::SHADER_LUM_TRANSFORM_FINAL_PS,   binarys[Factory::PES_LUM_TR_FINAL] );
				GetShaderBinary( Factory::SHADER_LUM_ADAPTED_PS,           binarys[Factory::PES_LUM_ADAPTED] );
				GetShaderBinary( Factory::SHADER_LUM_OPERATOR___B_PS,      binarys[Factory::PES_LUM_OP___B] );
				GetShaderBinary( Factory::SHADER_LUM_OPERATOR_T___PS,      binarys[Factory::PES_LUM_OP_T__] );
				GetShaderBinary( Factory::SHADER_LUM_OPERATOR_T_B_PS,      binarys[Factory::PES_LUM_OP_T_B] );
				GetShaderBinary( Factory::SHADER_LUM_OPERATOR_TA__PS,      binarys[Factory::PES_LUM_OP_TA_] );
				GetShaderBinary( Factory::SHADER_LUM_OPERATOR_TAB_PS,      binarys[Factory::PES_LUM_OP_TAB] );
				binaryCount = 9;
				break;

			case Factory::ER_LIGHT_SHAFTS:
				GetShaderBinary( Factory::SHADER_LS_INIT___PS, binarys[Factory::PES_LS_INIT__] );
				GetShaderBinary( Factory::SHADER_LS_INIT_A_PS, binarys[Factory::PES_LS_INIT_A] );
				GetShaderBinary( Factory::SHADER_LS_BLUR_L_PS, binarys[Factory::PES_LS_BLUR_L] );
				GetShaderBinary( Factory::SHADER_LS_BLUR_M_PS, binarys[Factory::PES_LS_BLUR_M] );
				GetShaderBinary( Factory::SHADER_LS_BLUR_H_PS, binarys[Factory::PES_LS_BLUR_H] );
				binaryCount = 5;
				break;

			case Factory::ER_LENS_FLARE:
				GetShaderBinary( Factory::SHADER_LF_IMAGE____PS, binarys[Factory::PES_LF_IMAGE___] );
				GetShaderBinary( Factory::SHADER_LF_IMAGE__H_PS, binarys[Factory::PES_LF_IMAGE__H] );
				GetShaderBinary( Factory::SHADER_LF_IMAGE_G__PS, binarys[Factory::PES_LF_IMAGE_G_] );
				GetShaderBinary( Factory::SHADER_LF_IMAGE_GH_PS, binarys[Factory::PES_LF_IMAGE_GH] );
				GetShaderBinary( Factory::SHADER_LF_MODIFY____PS, binarys[Factory::PES_LF_MODIFY___] );
				GetShaderBinary( Factory::SHADER_LF_MODIFY_D__PS, binarys[Factory::PES_LF_MODIFY_D_] );
				GetShaderBinary( Factory::SHADER_LF_MODIFY__B_PS, binarys[Factory::PES_LF_MODIFY__B] );
				GetShaderBinary( Factory::SHADER_LF_MODIFY_DB_PS, binarys[Factory::PES_LF_MODIFY_DB] );
				binaryCount = 8;
				break;

			case Factory::ER_FILMIC:
				GetShaderBinary( Factory::SHADER_FI_HEJL_PS, binarys[Factory::PES_FI_HEJL] );
				GetShaderBinary( Factory::SHADER_FI_HABLE_PS, binarys[Factory::PES_FI_HABLE] );
				binaryCount = 2;
				break;

			case Factory::ER_GAMMA:
				GetShaderBinary( Factory::SHADER_GAMMA_PS, binarys[Factory::PES_DEFAULT] );
				binaryCount = 1;
				break;

			case Factory::ER_CHROMATISM:
				GetShaderBinary( Factory::SHADER_CA_IMAGE___PS, binarys[Factory::PES_CA_IMAGE__] );
				GetShaderBinary( Factory::SHADER_CA_IMAGE_K_PS, binarys[Factory::PES_CA_IMAGE_K] );
				GetShaderBinary( Factory::SHADER_CA_FINISH___PS, binarys[Factory::PES_CA_FINISH__] );
				GetShaderBinary( Factory::SHADER_CA_FINISH_D_PS, binarys[Factory::PES_CA_FINISH_D] );
				binaryCount = 4;
				break;

			case Factory::ER_DOF:
				GetShaderBinary( Factory::SHADER_DOF_PS, binarys[Factory::PES_DEFAULT] );
				binaryCount = 1;
				break;

			case Factory::ER_LENS_DIST:
				GetShaderBinary( Factory::SHADER_LENS_DIST___PS, binarys[Factory::PES_LD__] );
				GetShaderBinary( Factory::SHADER_LENS_DIST_K_PS, binarys[Factory::PES_LD_K] );
				binaryCount = 2;
				break;

			case Factory::ER_VIGNETTE:
				GetShaderBinary( Factory::SHADER_VIGNETTE_PS, binarys[Factory::PES_DEFAULT] );
				binaryCount = 1;
				break;

			case Factory::ER_FXAA:
				GetShaderBinary( Factory::SHADER_AA_FXAA_PS, binarys[Factory::PES_DEFAULT] );
				binaryCount = 1;
				break;
			}

			MIX_ASSERT( binaryCount <= Factory::PES_MAX );

			for( UInt32 i = 0; ( i < binaryCount ) && ( bSuccess == True ); i++ )
			{
				MIX_ASSERT( binarys[i].pData != NULL );
				MIX_ASSERT( binarys[i].dataSize > 0 );

				const Factory::SHADER_BINARY* psb = &( binarys[i] );

				if( m_pGraphicsDevice->CreatePixelShaderFromMemory( psb->pData, psb->dataSize, &( pe.pShader[i] ), psb->name ) == False )
				{
					bSuccess = False;
				}
			}

			if( bSuccess == True )
			{
				m_ResidentPEMap.insert( Factory::PixelEffectMap::value_type( opt, pe ) );

				it = m_ResidentPEMap.find( opt );
				if( it != m_ResidentPEMap.end() )
				{
					pEffect = &( it->second );
				}
				else
				{
					pe.Dispose();
				}
			}
			else
			{
				pe.Dispose();
			}
		}
	}
	else if( type == Factory::E_MATERIAL )
	{
		Factory::PixelEffectMap::iterator it = m_MaterialPEMap.find( opt );
		if( it != m_MaterialPEMap.end() )
		{
			pEffect = &( it->second );
		}
	}

	return pEffect;
}

UInt32 Factory::ModifyRendererCaps( UInt32 caps ) const
{
	UInt32 mepProcFlags = MEP_GetProcessFlags();
	UInt32 ret = 0;

	if( ( Mix::Dynamics::GetManagerPtr() != NULL ) &&
		( MIX_TESTBIT( caps, Mix::Scene::RCAP_DYNAMICS ) == Mix::Scene::RCAP_DYNAMICS ) )
	{
		ret |= Mix::Scene::RCAP_DYNAMICS;
	}

	if( ( MIX_TESTBIT( caps, Mix::Scene::RCAP_WIDGET ) == Mix::Scene::RCAP_WIDGET ) )
	{
		ret |= Mix::Scene::RCAP_WIDGET;
	}

	if( ( MIX_TESTBIT( caps, Mix::Scene::RCAP_ATMOSPHERE ) == Mix::Scene::RCAP_ATMOSPHERE ) &&
		( MIX_TESTBIT( mepProcFlags, Mix::Scene::Common::Factory::EP_ATMOSPHERE ) == Mix::Scene::Common::Factory::EP_ATMOSPHERE ) )
	{
		ret |= Mix::Scene::RCAP_ATMOSPHERE;
	}

	if( ( MIX_TESTBIT( caps, Mix::Scene::RCAP_AMBIENT_OCCLUSION ) == Mix::Scene::RCAP_AMBIENT_OCCLUSION ) &&
		( MIX_TESTBIT( mepProcFlags, Mix::Scene::Common::Factory::EP_SSAO ) == Mix::Scene::Common::Factory::EP_SSAO ) )
	{
		ret |= Mix::Scene::RCAP_AMBIENT_OCCLUSION;
	}

	if( ( MIX_TESTBIT( caps, Mix::Scene::RCAP_SHADOW_MAPPING ) == Mix::Scene::RCAP_SHADOW_MAPPING ) &&
		( MIX_TESTBIT( mepProcFlags, Mix::Scene::Common::Factory::EP_SHADOW ) == Mix::Scene::Common::Factory::EP_SHADOW ) )
	{
		ret |= Mix::Scene::RCAP_SHADOW_MAPPING;
	}

	if( MIX_TESTBIT( caps, Mix::Scene::RCAP_LUMINOSITY ) == Mix::Scene::RCAP_LUMINOSITY )
	{
		ret |= Mix::Scene::RCAP_LUMINOSITY;
	}

	if( MIX_TESTBIT( caps, Mix::Scene::RCAP_BLOOM ) == Mix::Scene::RCAP_BLOOM )
	{
		ret |= Mix::Scene::RCAP_BLOOM;
	}

	if( ( MIX_TESTBIT( caps, Mix::Scene::RCAP_LIGHT_SHAFTS ) == Mix::Scene::RCAP_LIGHT_SHAFTS ) )
	{
		ret |= Mix::Scene::RCAP_LIGHT_SHAFTS;
	}

	if( ( MIX_TESTBIT( caps, Mix::Scene::RCAP_LENS_FLARE ) == Mix::Scene::RCAP_LENS_FLARE ) )
	{
		ret |= Mix::Scene::RCAP_LENS_FLARE;
	}

	if( ( MIX_TESTBIT( caps, Mix::Scene::RCAP_CHROMATISM ) == Mix::Scene::RCAP_CHROMATISM ) )
	{
		ret |= Mix::Scene::RCAP_CHROMATISM;
	}

	if( MIX_TESTBIT( caps, Mix::Scene::RCAP_DEPTH_OF_FIELD ) == Mix::Scene::RCAP_DEPTH_OF_FIELD )
	{
		ret |= Mix::Scene::RCAP_DEPTH_OF_FIELD;
	}

	return ret;
}

Boolean Factory::CreateRenderer( const Mix::Scene::RENDERER_CONFIG& config, Mix::Scene::IRenderer** ppRenderer, const wchar_t* pName )
{
	if( ppRenderer == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s] : pRenderer[%s]",
			Mix::Scene::Common::Renderer::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pName ),
			MIX_LOG_PTR( ppRenderer ) );

		return False;
	}

	Mix::Scene::IRenderer* pRenderer;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"V[_[̍쐬 : Name[%s] Type[Standard]", MIX_SAFE_NAME( pName ) );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pRenderer = CreateRenderer( config, MIX_SAFE_NAME( pName ) );
	if( pRenderer == NULL )
	{
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppRenderer ) = pRenderer;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Factory::CreateMaterialFromFile( const wchar_t* pFilePath, Mix::Scene::IMaterial** ppMaterial )
{
	if( ( MIX_STR_LENGTH( pFilePath ) == 0 ) ||
		( ppMaterial == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pFilePath[%s] ppMaterial[%s]", Mix::Scene::Standard::Common::DefaultMaterial::FAILED_CREATE, Mix::STR_ILLEGALARG,
			MIX_SAFE_STR( pFilePath ),
			MIX_LOG_PTR( ppMaterial ) );

		return False;
	}

	Mix::File::IReader* pReader = NULL;
	Mix::Scene::IMaterial* pMaterial = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"}eA̍쐬 : FilePath[%s] Type[Standard( Default )]", pFilePath );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// t@C[_[쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_pFileManager->CreateFileReader( pFilePath, &pReader ) == False )
	{
		MIX_LOG_ERROR( L"%s : %s : FilePath[%s]", Mix::Scene::Standard::Common::DefaultMaterial::FAILED_CREATE, Mix::STR_FILENOTFOUND, pFilePath );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pMaterial = CreateDefaultMaterial( L"FilePath", pFilePath, True, pReader, this, m_pGraphicsDevice );
	if( pMaterial == NULL )
	{
		MIX_RELEASE( pReader );
		return False;
	}

	MIX_RELEASE( pReader );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppMaterial ) = pMaterial;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Factory::CreateMaterialFromMemory( const wchar_t* pName, Mix::Memory::IBuffer* pBuffer, Mix::Scene::IMaterial** ppMaterial )
{
	if(	( MIX_STR_LENGTH( pName ) == 0 ) ||
		( pBuffer == NULL ) ||
		( ppMaterial == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pName[%s] : pBuffer[%s] ppMaterial[%s]", Mix::Scene::Standard::Common::DefaultMaterial::FAILED_CREATE, Mix::STR_ILLEGALARG,
			MIX_SAFE_STR( pName ),
			MIX_LOG_PTR( pBuffer ),
			MIX_LOG_PTR( ppMaterial ) );

		return False;
	}

	Mix::File::IReader* pReader = NULL;
	Mix::Scene::IMaterial* pMaterial = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"}eA̍쐬 : Name[%s] Type[Standard( Default )]", MIX_SAFE_NAME( pName ) );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// obt@h[_[쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_pFileManager->CreateBufferedReader( pName, pBuffer, &pReader ) == False )
	{
		MIX_LOG_ERROR( L"%s : Mix::File::IManager::CreateBufferedReader %s : Name[%s]", Mix::Scene::Standard::Common::DefaultMaterial::FAILED_CREATE, Mix::STR_RETERROR, pName );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pMaterial = CreateDefaultMaterial( L"Name", pName, True, pReader, this, m_pGraphicsDevice );
	if( pMaterial == NULL )
	{
		MIX_RELEASE( pReader );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ㏈
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_RELEASE( pReader );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppMaterial ) = pMaterial;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Mix::Scene::IMaterial* Factory::CreateMaterialWithoutMagicNumber( const wchar_t* pName, Mix::File::IReader* pReader )
{
	Mix::Scene::IMaterial* pMaterial;

	MIX_LOG_INFO_SECT_START( L"}eA̍쐬 : Name[%s] Type[Standard( Default )]", MIX_SAFE_NAME( pName ) );

	pMaterial = CreateDefaultMaterial( L"Name", pName, False, pReader, this, m_pGraphicsDevice );

	MIX_LOG_INFO_SECT_END();

	return pMaterial;
}

Boolean Factory::CreateMotionFromFile( const wchar_t* pFilePath, Mix::Scene::IMotion** ppMotion )
{
	if( ( pFilePath == NULL ) ||
		( ::wcslen( pFilePath ) == 0 ) ||
		( ppMotion == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pFilePath[%s], ppMotion[%s]",
			Mix::Scene::Common::Motion::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_STR( pFilePath ),
			MIX_LOG_PTR( ppMotion ) );

		return False;
	}

	Mix::File::IReader* pReader = NULL;
	Mix::Scene::Common::Motion* pMotion = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// [_[̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_pFileManager->CreateFileReader( pFilePath, &pReader ) == False )
	{
		MIX_LOG_ERROR( L"%s : %s : FilePath[%s]", Mix::Scene::Common::Motion::FAILED_CREATE, Mix::STR_FILENOTFOUND, pFilePath );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pMotion = Mix::Scene::Common::Motion::CreateInstance( pReader, True, L"FilePath", pFilePath );
	if( pMotion == NULL )
	{
		MIX_RELEASE( pReader );
		return False;
	}

	MIX_RELEASE( pReader );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppMotion ) = pMotion;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return False;
}

Boolean Factory::CreateMotionFromMemory( const wchar_t* pName, Mix::Memory::IBuffer* pBuffer, Mix::Scene::IMotion** ppMotion )
{
	if( ( pName == NULL ) ||
		( ::wcslen( pName ) == 0 ) ||
		( ppMotion == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pName[%s], ppMotion[%s]",
			Mix::Scene::Common::Motion::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_STR( pName ),
			MIX_LOG_PTR( ppMotion ) );

		return False;
	}

	Mix::File::IReader* pReader = NULL;
	Mix::Scene::Common::Motion* pMotion = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// [_[̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_pFileManager->CreateBufferedReader( pName, pBuffer, &pReader ) == False )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s]", Mix::Scene::Common::Motion::FAILED_CREATE, Mix::STR_FILENOTFOUND, pName );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pMotion = Mix::Scene::Common::Motion::CreateInstance( pReader, True, L"Name", pName );
	if( pMotion == NULL )
	{
		MIX_RELEASE( pReader );
		return False;
	}

	MIX_RELEASE( pReader );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppMotion ) = pMotion;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return False;
}

Boolean Factory::CreateDefaultParticleSystem( Boolean bSimWorldSpace, Mix::Scene::IDefaultParticleSystem::PLANE_TYPE planeType, Mix::Scene::IDefaultParticleSystem** ppSystem, const wchar_t* pDebugName )
{
	if( ppSystem == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppProducer[%s]",
			MIX_SAFE_NAME( pDebugName ),
			DefaultParticleSystem::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_PTR( ppSystem ) );

		return False;
	}

	Mix::Scene::Common::DefaultParticleSystem* pSystem = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pSystem = Mix::Scene::Common::DefaultParticleSystem::CreateInstance( bSimWorldSpace, MIX_SAFE_NAME( pDebugName ) );
	if( pSystem == NULL )
	{
		MIX_LOG_ERROR( L"%s : DebugName[%s]", Mix::STR_OUTOFMEMORY, MIX_SAFE_NAME( pDebugName ) );

		return False;
	}

	if( pSystem->Initialize( planeType ) == False )
	{
		//G[bZ[W Initialize \bhŏoĂ܂
		MIX_RELEASE( pSystem );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppSystem ) = pSystem;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Factory::CreateBillboardParticleSystem( Boolean bSimWorldSpace, Mix::Scene::IBillboardParticleSystem** ppSystem, const wchar_t* pDebugName )
{
	if( ppSystem == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppProducer[%s]",
			MIX_SAFE_NAME( pDebugName ),
			BillboardParticleSystem::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_PTR( ppSystem ) );

		return False;
	}

	Mix::Scene::Common::BillboardParticleSystem* pSystem = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pSystem = Mix::Scene::Common::BillboardParticleSystem::CreateInstance( bSimWorldSpace, MIX_SAFE_NAME( pDebugName ) );
	if( pSystem == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]",
			BillboardParticleSystem::FAILED_CREATE,
			Mix::STR_OUTOFMEMORY,
			MIX_SAFE_NAME( pDebugName ) );

		return False;
	}

	if( pSystem->Initialize() == False )
	{
		//G[bZ[W Initialize \bhŏoĂ܂
		MIX_RELEASE( pSystem );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppSystem ) = pSystem;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Factory::CreateLeavingParticle( Mix::Scene::IMaterial* pMaterial, Mix::Scene::IParticleSystem* pSystem, Mix::Scene::ILeavingParticle** ppParticle, const wchar_t* pDebugName )
{
	if( ( pMaterial == NULL ) ||
		( pSystem == NULL ) ||
		( ppParticle == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s] : pMaterial[%s] pSystem[%s] ppParticle[%s]",
			Mix::Scene::Common::LeavingParticle::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( pMaterial ),
			MIX_LOG_PTR( pSystem ),
			MIX_LOG_PTR( ppParticle ) );

		return False;
	}

	Mix::Scene::Common::LeavingParticle* pParticle = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"[rOp[eBN̍쐬 : DebugName[%s]", MIX_SAFE_NAME( pDebugName ) );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// }eÃ`FbN
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( InternalParticle::CheckMaterial( pMaterial, NULL, MIX_SAFE_NAME( pDebugName ) ) == False )
	{
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pParticle = Mix::Scene::Common::LeavingParticle::CreateInstance( pMaterial, pSystem, MIX_SAFE_NAME( pDebugName ) );
	if( pParticle == NULL )
	{
		MIX_LOG_ERROR( L"%s", Mix::STR_OUTOFMEMORY );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pParticle->Initialize() == False )
	{
		MIX_RELEASE( pParticle );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppParticle ) = pParticle;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Factory::CreateScatterParticle( Mix::Scene::IMaterial* pMaterial, Mix::Scene::IParticleSystem* pSystem, Mix::Scene::IScatterParticle** ppParticle, const wchar_t* pDebugName )
{
	if( ( pMaterial == NULL ) ||
		( pSystem == NULL ) ||
		( ppParticle == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s] : pMaterial[%s] pSystem[%s] ppParticle[%s]",
			Mix::Scene::Common::ScatterParticle::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_SAFE_NAME( pDebugName ),
			MIX_LOG_PTR( pMaterial ),
			MIX_LOG_PTR( pSystem ),
			MIX_LOG_PTR( ppParticle ) );

		return False;
	}

	Mix::Scene::Common::ScatterParticle* pParticle = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"XLb^[p[eBN̍쐬 : DebugName[%s]", MIX_SAFE_NAME( pDebugName ) );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// }eÃ`FbN
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( InternalParticle::CheckMaterial( pMaterial, NULL, MIX_SAFE_NAME( pDebugName ) ) == False )
	{
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pParticle = Mix::Scene::Common::ScatterParticle::CreateInstance( pMaterial, pSystem, 32, 256, 16, MIX_SAFE_NAME( pDebugName ) );
	if( pParticle == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s", Mix::Scene::Common::ScatterParticle::FAILED_CREATE, Mix::STR_OUTOFMEMORY );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppParticle ) = pParticle;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Factory::CreateActorModelFromFile( const wchar_t* pFilePath, Mix::Scene::IActorModel** ppModel )
{
	if( ( pFilePath == NULL ) ||
		( ::wcslen( pFilePath ) == 0 ) ||
		( ppModel == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pFilePath[%s] ppModel[%s]", Mix::Scene::Common::ActorLoader::FAILED_CREATE, Mix::STR_ILLEGALARG,
			MIX_LOG_STR( pFilePath ),
			MIX_LOG_PTR( ppModel ) );

		return False;
	}

	Mix::File::IReader* pReader = NULL;
	Mix::Scene::IActorModel* pModel = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"AN^[f̍쐬 : FilePath[%s]", pFilePath );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// t@C[_[쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_pFileManager->CreateFileReader( pFilePath, &pReader ) == False )
	{
		MIX_LOG_INFO_SECT_END();

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pModel = m_pActorLoader->Create( pReader, L"File", pFilePath );
	if( pModel == NULL )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_RELEASE( pReader );

		return False;
	}

	MIX_RELEASE( pReader );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppModel ) = pModel;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Factory::CreateActorModelFromMemory( const wchar_t* pName, Mix::Memory::IBuffer* pBuffer, Mix::Scene::IActorModel** ppModel )
{
	Mix::StringW safeName = MIX_SAFE_NAME( pName );

	if( ( pBuffer == NULL ) ||
		( ppModel == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pName[%s] pBuffer[%s] ppModel[%s]", Mix::Scene::Common::ActorLoader::FAILED_CREATE, Mix::STR_ILLEGALARG,
			safeName.GetConstPtr(),
			MIX_LOG_PTR( pBuffer ),
			MIX_LOG_PTR( ppModel ) );

		return False;
	}

	Mix::File::IReader* pReader = NULL;
	Mix::Scene::IActorModel* pModel = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( Jn )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"AN^[f̍쐬 : Name[%s]", pName );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// obt@h[_[쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_pFileManager->CreateBufferedReader( pName, pBuffer, &pReader ) == False )
	{
		MIX_LOG_INFO_SECT_END();

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pModel = m_pActorLoader->Create( pReader, L"Name", pName );
	if( pModel == NULL )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_RELEASE( pReader );

		return False;
	}

	MIX_RELEASE( pReader );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( I )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppModel ) = pModel;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Factory::CreateTerrainModelFromFile( const wchar_t* pFilePath, Mix::Scene::ITerrainModel** ppModel )
{
	if( ( pFilePath == NULL ) ||
		( ::wcslen( pFilePath ) == 0 ) ||
		( ppModel == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pFilePath[%s] ppModel[%s]", Mix::Scene::Common::TerrainLoader::FAILED_CREATE, Mix::STR_ILLEGALARG,
			MIX_LOG_STR( pFilePath ),
			MIX_LOG_PTR( ppModel ) );

		return False;
	}

	Mix::File::IReader* pReader = NULL;
	Mix::Scene::ITerrainModel* pModel = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( Jn )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"n`f̍쐬 : FilePath[%s]", pFilePath );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// t@C[_[쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_pFileManager->CreateFileReader( pFilePath, &pReader ) == False )
	{
		MIX_LOG_INFO_SECT_END();

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pModel = m_pTerrainLoader->Create( pReader, pFilePath );
	if( pModel == NULL )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_RELEASE( pReader );

		return False;
	}

	MIX_RELEASE( pReader );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( I )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppModel ) = pModel;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Factory::CreateTerrainModelFromMemory( const wchar_t* pName, Mix::Memory::IBuffer* pBuffer, Mix::Scene::ITerrainModel** ppModel )
{
	Mix::StringW safeName = MIX_SAFE_NAME( pName );

	if( ( pBuffer == NULL ) ||
		( ppModel == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pName[%s] ppModel[%s]", Mix::Scene::Common::TerrainLoader::FAILED_CREATE, Mix::STR_ILLEGALARG,
			safeName.GetConstPtr(),
			MIX_LOG_PTR( ppModel ) );

		return False;
	}

	Mix::File::IReader* pReader = NULL;
	Mix::Scene::ITerrainModel* pModel = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( Jn )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"n`f̍쐬 : Name[%s]", pName );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// obt@h[_[쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_pFileManager->CreateBufferedReader( pName, pBuffer, &pReader ) == False )
	{
		MIX_LOG_INFO_SECT_END();

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pModel = m_pTerrainLoader->Create( pReader, pName );
	if( pModel == NULL )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_RELEASE( pReader );

		return False;
	}

	MIX_RELEASE( pReader );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O( I )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppModel ) = pModel;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

}}}
