#include "Mix/Class/Scene/Common/ActorDynamicsCluster.h"

#include "Mix/Dynamics/IWorld.h"
#include "Mix/Dynamics/IRigidBody.h"
#include "Mix/Dynamics/ISensor.h"

#include "Mix/Class/Scene/Common/ActorRevision.h"
#include "Mix/Class/Scene/Common/ActorCollider.h"
#include "Mix/Class/Scene/Common/Sensor.h"

namespace Mix{ namespace Scene{ namespace Common{

ActorDynamicsCluster::ActorDynamicsCluster( void ) :
m_pParentWorldMat( NULL ),
m_pWorldMat( NULL ),
m_pRevision( NULL ),
m_pWorld( NULL ),
m_ColliderMode( Mix::Scene::DC_DEFAULT ),
m_pCollider( NULL ),
m_bSensorEnabled( False )
{
}

ActorDynamicsCluster::~ActorDynamicsCluster( void )
{
#ifdef _DEBUG

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// fobO : j̊mF
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_ASSERT( m_pWorld == NULL );

	/*
		RC_[
	*/

	if( m_pCollider != NULL )
	{
		MIX_ASSERT( m_pCollider->GetOwnerPtr() == NULL );
	}

	/*
		ZT[
	*/

	if( m_SensorList.size() > 0 )
	{
		ActorDynamicsCluster::SensorList::iterator it_s_begin = m_SensorList.begin();
		ActorDynamicsCluster::SensorList::iterator it_s_end = m_SensorList.end();
		ActorDynamicsCluster::SensorList::iterator it_s;

		for( it_s = it_s_begin; it_s != it_s_end; ++it_s )
		{
			MIX_ASSERT( ( *it_s )->GetOwnerPtr() == NULL );
		}
	}

#endif //_DEBUG

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	for( ActorDynamicsCluster::SensorListMap::iterator it_sl = m_SensorListMap.begin(); it_sl != m_SensorListMap.end(); ++it_sl )
	{
		for( ActorDynamicsCluster::SensorList::iterator it_s = it_sl->second.begin(); it_s != it_sl->second.end(); ++it_s )
		{
			MIX_RELEASE( ( *it_s ) );
		}
	}

	MIX_RELEASE( m_pCollider );
}

Boolean ActorDynamicsCluster::Clone( ActorDynamicsCluster* pCloneCluster )
{
	MIX_ASSERT( pCloneCluster != NULL );
	MIX_ASSERT( pCloneCluster->m_pCollider == NULL );
	MIX_ASSERT( pCloneCluster->m_SensorListMap.size() == 0 );
	MIX_ASSERT( pCloneCluster->m_SensorList.size() == 0 );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// RC_[̕
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_pCollider != NULL )
	{
		Mix::Scene::Common::ActorCollider* pCloneCollider = m_pCollider->Clone();

		if( pCloneCollider != NULL )
		{
			pCloneCluster->m_pCollider = pCloneCollider;
		}
		else
		{
			MIX_RELEASE( pCloneCollider );
			return False;
		}
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ZT[̕
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_SensorList.size() > 0 )
	{
		ActorDynamicsCluster::SensorList::iterator it_s_begin = m_SensorList.begin();
		ActorDynamicsCluster::SensorList::iterator it_s_end = m_SensorList.end();
		ActorDynamicsCluster::SensorList::iterator it_s;

		for( it_s = it_s_begin; it_s != it_s_end; ++it_s )
		{
			Mix::Scene::Common::Sensor* pCloneSensor = ( *it_s )->Clone();

			if( pCloneSensor != NULL )
			{
				pCloneCluster->AddSensor( pCloneSensor );
				MIX_RELEASE( pCloneSensor );
			}
			else
			{
				MIX_RELEASE( pCloneSensor );
				return False;
			}
		}

		pCloneCluster->FinishSensors( m_SensorList.size() );
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

void ActorDynamicsCluster::SetCollider( Mix::Scene::Common::ActorCollider* pCollider )
{
	MIX_ASSERT( pCollider != NULL );
	MIX_ASSERT( m_pCollider == NULL );

	MIX_ADD_REF( pCollider );
	m_pCollider = pCollider;
}

Mix::Scene::Common::ActorCollider* ActorDynamicsCluster::GetColliderPtr( void ) const
{
	return m_pCollider;
}

void ActorDynamicsCluster::AddSensor( Mix::Scene::Common::Sensor* pSensor )
{
	MIX_ASSERT( pSensor != NULL );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// fobO : ̂ǉ悤ƂĂȂ`FbN
	////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG
	for( ActorDynamicsCluster::SensorListMap::iterator it_sl = m_SensorListMap.begin(); it_sl != m_SensorListMap.end(); ++it_sl )
	{
		for( ActorDynamicsCluster::SensorList::iterator it_s = it_sl->second.begin(); it_s != it_sl->second.end(); ++it_s )
		{
			MIX_ASSERT( ( *it_s ) != pSensor );
		}
	}
#endif //_DEBUG

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ǉ
	////////////////////////////////////////////////////////////////////////////////////////////////////

	ActorDynamicsCluster::SensorListMap::iterator it_sl = m_SensorListMap.find( pSensor->GetName() );

	if( it_sl == m_SensorListMap.end() )
	{
		m_SensorListMap.insert( ActorDynamicsCluster::SensorListMap::value_type( pSensor->GetName(), ActorDynamicsCluster::SensorList() ) );
		it_sl = m_SensorListMap.find( pSensor->GetName() );
		MIX_ASSERT( it_sl != m_SensorListMap.end() );
	}

	MIX_ADD_REF( pSensor );
	it_sl->second.push_back( pSensor );
}

void ActorDynamicsCluster::FinishSensors( UInt32 num )
{
	MIX_ASSERT( m_SensorList.size() == 0 );

	m_SensorList.reserve( num );

	for( ActorDynamicsCluster::SensorListMap::iterator it_sl = m_SensorListMap.begin(); it_sl != m_SensorListMap.end(); ++it_sl )
	{
		for( ActorDynamicsCluster::SensorList::iterator it_s = it_sl->second.begin(); it_s != it_sl->second.end(); ++it_s )
		{
			m_SensorList.push_back( ( *it_s ) );
		}
	}

	m_bSensorEnabled = True;
}

UInt32 ActorDynamicsCluster::GetSensorCount( void ) const
{
	return m_SensorList.size();
}

Mix::Scene::Common::Sensor* ActorDynamicsCluster::GetSensorPtr( UInt32 index ) const
{
	if( m_SensorList.size() <= index )
	{
		return NULL;
	}

	return m_SensorList[index];
}

void ActorDynamicsCluster::SetLink( Mix::Scene::IRendererObject* pOwner, Mix::Matrix4x4* pParentWorldMat, Mix::Matrix4x4* pWorldMat )
{
	MIX_ASSERT( pOwner != NULL );
	MIX_ASSERT( pWorldMat != NULL );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// [hs̃|C^
	////////////////////////////////////////////////////////////////////////////////////////////////////

	m_pParentWorldMat = pParentWorldMat;
	m_pWorldMat = pWorldMat;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// I[i[̐ݒ
	////////////////////////////////////////////////////////////////////////////////////////////////////

	/*
		RC_[
	*/

	if( m_pCollider != NULL )
	{
		MIX_ASSERT( m_pCollider->GetOwnerPtr() == NULL );
		m_pCollider->SetOwner( pOwner );
	}

	/*
		ZT[
	*/

	if( m_SensorList.size() > 0 )
	{
		ActorDynamicsCluster::SensorList::iterator it_s_begin = m_SensorList.begin();
		ActorDynamicsCluster::SensorList::iterator it_s_end = m_SensorList.end();
		ActorDynamicsCluster::SensorList::iterator it_s;

		for( it_s = it_s_begin; it_s != it_s_end; ++it_s )
		{
			MIX_ASSERT( ( *it_s )->GetOwnerPtr() == NULL );
			( *it_s )->SetOwner( pOwner );
		}
	}
}

Boolean ActorDynamicsCluster::IsColliderEnabled( void ) const
{
	return ( m_pCollider != NULL )? m_pCollider->IsEnabled() : False;
}

void ActorDynamicsCluster::SetColliderEnabled( Boolean state )
{
	if( ( m_pCollider != NULL ) &&
		( m_pCollider->IsEnabled() != state ) )
	{
		m_pCollider->SetEnabled( state );

		OnColliderStateChanged( state );
	}
}

Mix::Scene::DYNAMICS_COLLIDER_MODE ActorDynamicsCluster::GetColliderMode( void ) const
{
	return m_ColliderMode;
}

void ActorDynamicsCluster::SetColliderMode( Mix::Scene::DYNAMICS_COLLIDER_MODE mode )
{
	if( m_ColliderMode != mode )
	{
		if( m_pCollider != NULL )
		{
			m_pCollider->SetMode( mode );
		}

		m_ColliderMode = mode;

		OnColliderModeChanged( mode );
	}
}

Mix::Matrix4x4 ActorDynamicsCluster::GetColliderMatrix( void ) const
{
	MIX_ASSERT( m_pCollider != NULL );

	return m_pCollider->GetWorldMatrix();
}

Boolean ActorDynamicsCluster::IsSensorEnabled( void ) const
{
	return m_bSensorEnabled;
}

void ActorDynamicsCluster::SetSensorEnabled( Boolean state )
{
	if( ( m_SensorList.size() > 0 ) &&
		( m_bSensorEnabled != state ) )
	{
		ActorDynamicsCluster::SensorList::iterator it_s_begin = m_SensorList.begin();
		ActorDynamicsCluster::SensorList::iterator it_s_end = m_SensorList.end();
		ActorDynamicsCluster::SensorList::iterator it_s;

		for( it_s = it_s_begin; it_s != it_s_end; ++it_s )
		{
			( *it_s )->SetEnabled( state );
		}

		m_bSensorEnabled = state;
	
		OnSensorStateChanged( state );
	}
}

Boolean ActorDynamicsCluster::IsCollide( void ) const
{
	return ( m_pCollider != NULL );
}

Boolean ActorDynamicsCluster::IsControl( void ) const
{
	if( ( m_pCollider == NULL ) ||
		( m_pCollider->CanRefresh( m_ColliderMode ) == False ) )
	{
		return False;
	}

	return True;
}

Boolean ActorDynamicsCluster::IsActive( void ) const
{
	if( ( m_pCollider == NULL ) ||
		( m_pCollider->CanRefresh( m_ColliderMode ) == False ) ||
		( m_pCollider->IsActive() == False ) )
	{
		return False;
	}

	return True;
}

void ActorDynamicsCluster::AttachToWorld( Mix::Dynamics::IWorld* pWorld, Mix::Dynamics::IObjectListener* pObjectListener )
{
	MIX_ASSERT( pWorld != NULL );
	MIX_ASSERT( pObjectListener != NULL );
	MIX_ASSERT( m_pWorld == NULL );

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// [h
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	m_pWorld = pWorld;

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// RC_[
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_pCollider != NULL )
	{
		m_pCollider->Attach( pWorld, pObjectListener );
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// ZT[
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_SensorList.size() > 0 )
	{
		ActorDynamicsCluster::SensorList::iterator it_s_begin = m_SensorList.begin();
		ActorDynamicsCluster::SensorList::iterator it_s_end = m_SensorList.end();
		ActorDynamicsCluster::SensorList::iterator it_s;

		for( it_s = it_s_begin; it_s != it_s_end; ++it_s )
		{
			( *it_s )->Attach( pWorld, pObjectListener );
		}
	}
}

void ActorDynamicsCluster::DetachFromWorld( Mix::Dynamics::IWorld* pWorld )
{
	MIX_ASSERT( pWorld != NULL );
	MIX_ASSERT( m_pWorld != NULL );
	MIX_ASSERT( m_pWorld == pWorld );

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// ZT[
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_SensorList.size() > 0 )
	{
		ActorDynamicsCluster::SensorList::iterator it_s_begin = m_SensorList.begin();
		ActorDynamicsCluster::SensorList::iterator it_s_end = m_SensorList.end();
		ActorDynamicsCluster::SensorList::iterator it_s;

		for( it_s = it_s_begin; it_s != it_s_end; ++it_s )
		{
			( *it_s )->Detach( pWorld );
		}
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// RC_[
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_pCollider != NULL )
	{
		m_pCollider->Detach( pWorld );
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// [h
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	m_pWorld = NULL;
}

void ActorDynamicsCluster::Reset( void )
{
	MIX_ASSERT( m_pWorldMat != NULL );

	Mix::Matrix4x4 baseMat = m_pWorldMat->Remake( False, True, True );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// RC_[
	//   ZT[̃gXtH[̃x[XƂȂs擾
	//   Wbh{fB̃Xe[^XftHgłȂꍇ́AOݒ肳Ă郏[hgXtH[ݒ
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_pCollider != NULL )
	{
		m_pCollider->Reset( baseMat );
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// q̃Zbg
	////////////////////////////////////////////////////////////////////////////////////////////////////

	ResetChild( baseMat );
}

void ActorDynamicsCluster::ResetChild( const Mix::Matrix4x4& baseMat )
{
	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ZT[
	//   ȍ~ɍs郏[hV~[V̂߂ɁAgXtH[ݒ
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_SensorList.size() > 0 )
	{
		ActorDynamicsCluster::SensorList::iterator it_s_begin = m_SensorList.begin();
		ActorDynamicsCluster::SensorList::iterator it_s_end = m_SensorList.end();
		ActorDynamicsCluster::SensorList::iterator it_s;

		for( it_s = it_s_begin; it_s != it_s_end; ++it_s )
		{
			( *it_s )->Reset( baseMat );
		}
	}
}

void ActorDynamicsCluster::Update( void )
{
	MIX_ASSERT( m_pRevision != NULL );

	Boolean bUpdateChild = NeedsUpdateChild();
	Mix::Matrix4x4 baseMat;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// RC_[
	//   Wbh{fB̃Xe[^XftHgłȂꍇ́AOݒ肳Ă郏[hgXtH[ݒ
	//   ȍ~̃gXtH[̃x[XƂȂs擾
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_pCollider != NULL )
	{
		if( m_pCollider->CanUpdateDefault( m_ColliderMode ) == True )
		{
			if( bUpdateChild == True )
			{
				baseMat = m_pCollider->UpdateDefault();
			}
		}
		else
		{
			MIX_ASSERT( m_pWorldMat != NULL );

			baseMat = m_pWorldMat->Remake( False, True, True );

			m_pCollider->UpdateKinematicOrStatic( baseMat );
		}
	}
	else
	{
		if( bUpdateChild == True )
		{
			baseMat = m_pWorldMat->Remake( False, True, True );
		}
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// q̍XV
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( bUpdateChild == True )
	{
		UpdateChild( baseMat );
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// rW
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( IsControl() == True )
	{
		m_pRevision->SetControlWorldTransform();
	}
}

Boolean ActorDynamicsCluster::NeedsUpdateChild( void ) const
{
	return ( m_SensorList.size() > 0 );
}

void ActorDynamicsCluster::UpdateChild( const Mix::Matrix4x4& baseMat )
{
	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ZT[
	//   ȍ~ɍs郏[hV~[V̂߂ɁAgXtH[ݒ
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_SensorList.size() > 0 )
	{
		ActorDynamicsCluster::SensorList::iterator it_s_begin = m_SensorList.begin();
		ActorDynamicsCluster::SensorList::iterator it_s_end = m_SensorList.end();
		ActorDynamicsCluster::SensorList::iterator it_s;

		for( it_s = it_s_begin; it_s != it_s_end; ++it_s )
		{
			( *it_s )->Reset( baseMat );
		}
	}
}

Boolean ActorDynamicsCluster::Refresh( void )
{
	MIX_ASSERT( m_pWorldMat != NULL );
	MIX_ASSERT( m_pRevision != NULL );

	Boolean result = False;

	if( m_pRevision->IsControlWorldTransform() == True )
	{
		if( ActorDynamicsCluster::IsActive() == True )
		{
			*m_pWorldMat = m_pCollider->Refresh( m_pWorldMat->GetScaling() );
			m_pRevision->SetWorldTransform();
			result = True;
		}
	}

	return result;
}

void ActorDynamicsCluster::Dispose( void )
{
	////////////////////////////////////////////////////////////////////////////////////////////////////
	// j
	////////////////////////////////////////////////////////////////////////////////////////////////////

	/*
		RC_[
	*/

	if( m_pCollider != NULL )
	{
		m_pCollider->Dispose();
	}

	/*
		ZT[
	*/

	if( m_SensorList.size() > 0 )
	{
		ActorDynamicsCluster::SensorList::iterator it_s_begin = m_SensorList.begin();
		ActorDynamicsCluster::SensorList::iterator it_s_end = m_SensorList.end();
		ActorDynamicsCluster::SensorList::iterator it_s;

		for( it_s = it_s_begin; it_s != it_s_end; ++it_s )
		{
			( *it_s )->Dispose();
		}
	}

	/*
		rW
	*/

	m_pRevision = NULL;
}

#ifdef _DEBUG

void ActorDynamicsCluster::Debug_Draw(	UInt32 flags,
										Mix::Graphics::Utility::ILineArt* pLineArt,
										Float32 axisScaling,
										Float32 jointFrameMinSize,
										Float32 jointLimitScaling )
{
	MIX_ASSERT( pLineArt != NULL );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// RC_[
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( ( m_pCollider != NULL ) &&
		( MIX_TESTBIT( flags, Mix::Scene::DDF_ACTORMODEL_COLLIDER ) == Mix::Scene::DDF_ACTORMODEL_COLLIDER ) )
	{
		Mix::Dynamics::IRigidBody* pInternalRigidBody = m_pCollider->GetInternalRigidBodyPtr();
		MIX_ASSERT( pInternalRigidBody != NULL );

		Float32 opacity = ( pInternalRigidBody->IsInWorld() == True )? 1.0f : 0.25f;

		pInternalRigidBody->Debug_SetDrawAxisScaling( axisScaling );
		pInternalRigidBody->Debug_Draw( pLineArt, opacity );
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ZT[
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( ( m_SensorList.size() > 0 ) &&
		( MIX_TESTBIT( flags, Mix::Scene::DDF_ACTORMODEL_SENSOR ) == Mix::Scene::DDF_ACTORMODEL_SENSOR ) )
	{
		ActorDynamicsCluster::SensorList::iterator it_s_begin = m_SensorList.begin();
		ActorDynamicsCluster::SensorList::iterator it_s_end = m_SensorList.end();
		ActorDynamicsCluster::SensorList::iterator it_s;

		for( it_s = it_s_begin; it_s != it_s_end; ++it_s )
		{
			Mix::Dynamics::ISensor* pInternalSensor = ( *it_s )->GetInternalSensorPtr();
			MIX_ASSERT( pInternalSensor != NULL );

			Float32 opacity = ( pInternalSensor->IsInWorld() == True )? 1.0f : 0.25f;

			pInternalSensor->Debug_SetDrawAxisScaling( axisScaling );
			pInternalSensor->Debug_Draw( pLineArt, opacity );
		}
	}
}

#endif //_DEBUG

}}}
